#ifndef __EXPR_H__
#define __EXPR_H__

#include "Common.h"
#include "Symbol.h"
#include "Signal.h"
#include "Environment.h"

namespace AScript {

class Expr;
class Expr_Field;
class ICallable;

//-----------------------------------------------------------------------------
// ExprType
//-----------------------------------------------------------------------------
enum ExprType {
	EXPRTYPE_UnaryOp,
	EXPRTYPE_Quote,
	EXPRTYPE_Force,
	EXPRTYPE_Prefix,
	EXPRTYPE_Suffix,
	EXPRTYPE_BinaryOp,
	EXPRTYPE_Assign,
	EXPRTYPE_DictAssign,
	EXPRTYPE_Field,
	EXPRTYPE_BlockParam,
	EXPRTYPE_Block,
	EXPRTYPE_Lister,
	EXPRTYPE_Value,
	EXPRTYPE_Symbol,
	EXPRTYPE_Indexer,
	EXPRTYPE_Function,
};

const char *GetExprTypeName(ExprType exprType);

//-----------------------------------------------------------------------------
// ExprVisitor
//-----------------------------------------------------------------------------
class ExprVisitor {
public:
	virtual bool Visit(const Expr *pExpr) = 0;
};

//-----------------------------------------------------------------------------
// Expr
// [class hierarchy under Expr]
// Expr <-+- Expr_Unary <-----+- Expr_UnaryOp
//        |                   +- Expr_Quote
//        |                   +- Expr_Force
//        |                   +- Expr_Prefix
//        |                   +- Expr_Suffix
//        +- Expr_Binary <----+- Expr_BinaryOp
//        |                   +- Expr_Assign
//        |                   +- Expr_DictAssign
//        |                   +- Expr_Field
//        +- Expr_Container <-+- Expr_BlockParam
//        |                   +- Expr_Block
//        |                   +- Expr_Lister
//        +- Expr_Complex <---+- Expr_Indexer
//        |                   +- Expr_Caller
//        +- Expr_Value
//        +- Expr_Symbol
//-----------------------------------------------------------------------------
class Expr {
public:
	class ExprVisitor_FindSymbol : public ExprVisitor {
	private:
		const Symbol *_pSymbol;
		bool _flag;
	public:
		inline ExprVisitor_FindSymbol(const Symbol *pSymbol) :
									_pSymbol(pSymbol), _flag(false) {}
		virtual bool Visit(const Expr *pExpr);
		inline bool GetFlag() const { return _flag; }
	};
	class ExprVisitor_GatherSymbol : public ExprVisitor {
	private:
		SymbolSet &_symbolSet;
	public:
		inline ExprVisitor_GatherSymbol(SymbolSet &symbolSet) :
												_symbolSet(symbolSet) {}
		virtual bool Visit(const Expr *pExpr);
	};
private:
	ExprType _exprType;
	int _cntRef;	// const_cast is used to update this value
	int _lineNo;
public:
	inline Expr(ExprType exprType) : _exprType(exprType), _cntRef(1), _lineNo(0) {}
	inline Expr(const Expr &expr) :
				_exprType(expr._exprType), _cntRef(1), _lineNo(expr._lineNo) {}
	virtual ~Expr();
	virtual Expr *IncRef() const;
	inline int DecRef() { _cntRef--; return _cntRef; }
	inline int GetRefCnt() const { return _cntRef; }
	inline static void Delete(Expr *pExpr) {
		if (pExpr != NULL && pExpr->DecRef() <= 0) delete pExpr;
	}
	inline void SetLineNo(int lineNo) { _lineNo = lineNo; }
	inline int GetLineNo() const { return _lineNo; }
	bool FindSymbol(const Symbol *pSymbol) const;
	void GatherSymbol(SymbolSet &symbolSet) const;
	bool GetChainedSymbolList(SymbolList &symbolList) const;
	void SetError(Signal sig, ErrorType errType, const char *format, ...) const;
	void SetError_NotAssignableSymbol(Signal sig, const Symbol *pSymbol) const;
	inline const char *GetTypeName() const { return GetExprTypeName(_exprType); }
	inline ExprType GetType() const { return _exprType; }
	virtual Expr *Clone() const = 0;
	virtual Expr *CloneSubstituted(Environment &env, Signal sig) const;
	virtual Value Exec(Environment &env, Signal sig) const = 0;
	virtual ValueType GetStaticValueType() const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual void Accept(ExprVisitor &visitor) const = 0;
	virtual const Expr *Unquote() const;
	virtual bool IsValue() const;
	virtual bool IsSymbol() const;
	virtual bool IsConstructor() const;
	virtual bool IsComplex() const;
	virtual bool IsCaller() const;
	virtual bool IsIndexer() const;
	virtual bool IsContainer() const;
	virtual bool IsBlock() const;
	virtual bool IsBlockParam() const;
	virtual bool IsLister() const;
	virtual bool IsUnary() const;
	virtual bool IsBinary() const;
	virtual bool IsField() const;
	virtual bool IsUnaryOp() const;
	virtual bool IsBinaryOp() const;
	virtual bool IsNeg() const;
	virtual bool IsPlus() const;
	virtual bool IsMinus() const;
	virtual bool IsMultiply() const;
	virtual bool IsDivide() const;
	virtual bool IsPower() const;
	virtual bool IsAssign() const;
	virtual bool IsDictAssign() const;
	virtual bool IsContainCheck() const;
	virtual bool IsQuote() const;
	virtual bool IsForce() const;
	virtual bool IsPrefix() const;
	virtual bool IsSuffix() const;
	virtual bool IsConstZero() const;
	virtual bool IsConstOne() const;
	virtual bool IsConstMinusOne() const;
	virtual bool IsConstMinus() const;
	virtual String ToString(Signal sig) const = 0;
};

//-----------------------------------------------------------------------------
// ExprList
//-----------------------------------------------------------------------------
class ExprList : public std::vector<Expr *> {
private:
	class ValueVisitorEx : public ValueVisitor {
	private:
		ValueList &_valList;
	public:
		inline ValueVisitorEx(ValueList &valList) : _valList(valList) {}
		virtual void Visit(Signal sig, const Value &value);
	};
public:
	static const ExprList Null;
public:
	inline ExprList() {};
	inline ExprList(Expr *pExpr) : std::vector<Expr *>(1) {
		(*this)[0] = pExpr;
	}
	inline ExprList(Expr *pExpr1, Expr *pExpr2) : std::vector<Expr *>(2) {
		(*this)[0] = pExpr1, (*this)[1] = pExpr2;
	}
	Value Exec(Environment &env, Signal sig, bool evalSymFuncFlag) const;
	Value ExecForList(Environment &env, Signal sig,
							bool flattenFlag, bool evalSymFuncFlag) const;
	void IncRef() const;
	String ToString(Signal sig, const char *sep = ", ") const;
	void Accept(ExprVisitor &visitor) const;
private:
	inline ExprList(const ExprList &exprList) {}; // not supported
};

class ExprOwner : public ExprList {
public:
	inline ExprOwner() {}
	ExprOwner(const ExprList &exprList);
	ExprOwner(const ExprOwner &exprOwner);
	~ExprOwner();
};

//-----------------------------------------------------------------------------
// Expr_Unary
//-----------------------------------------------------------------------------
class Expr_Unary : public Expr {
private:
	ExprOwner _exprOwner;
public:
	inline Expr_Unary(ExprType exprType, Expr *pExprChild) : Expr(exprType) {
		_exprOwner.push_back(pExprChild);
	}
	inline Expr_Unary(const Expr_Unary &expr) : Expr(expr), _exprOwner(expr._exprOwner) {}
	virtual ~Expr_Unary();
	virtual bool IsUnary() const;
	virtual Expr *IncRef() const;
	virtual void Accept(ExprVisitor &visitor) const;
	inline Expr *GetChild() { return const_cast<Expr *>(_exprOwner[0]); }
	inline const Expr *GetChild() const { return _exprOwner[0]; }
	inline const ExprList &GetExprList() const { return _exprOwner; }
};

//-----------------------------------------------------------------------------
// Expr_Binary
//-----------------------------------------------------------------------------
class Expr_Binary : public Expr {
private:
	ExprOwner _exprOwner;
public:
	inline Expr_Binary(ExprType exprType, Expr *pExprLeft, Expr *pExprRight) : Expr(exprType) {
		_exprOwner.push_back(pExprLeft); _exprOwner.push_back(pExprRight);
	}
	inline Expr_Binary(const Expr_Binary &expr) : Expr(expr), _exprOwner(expr._exprOwner) {}
	virtual ~Expr_Binary();
	virtual bool IsBinary() const;
	virtual Expr *IncRef() const;
	virtual void Accept(ExprVisitor &visitor) const;
	inline Expr *GetLeft() { return const_cast<Expr *>(_exprOwner[0]); }
	inline Expr *GetRight() { return const_cast<Expr *>(_exprOwner[1]); }
	inline const Expr *GetLeft() const { return _exprOwner[0]; }
	inline const Expr *GetRight() const { return _exprOwner[1]; }
	inline const ExprList &GetExprList() const { return _exprOwner; }
};

//-----------------------------------------------------------------------------
// Expr_Container
//-----------------------------------------------------------------------------
class Expr_Container : public Expr {
protected:
	ExprOwner _exprOwner;
public:
	inline Expr_Container(ExprType exprType) : Expr(exprType) {}
	inline Expr_Container(const Expr_Container &expr) : Expr(expr), _exprOwner(expr._exprOwner) {}
	virtual bool IsContainer() const;
	virtual ~Expr_Container();
	virtual Expr *IncRef() const;
	virtual void Accept(ExprVisitor &visitor) const;
	inline void AddExpr(Expr *pExpr) { _exprOwner.push_back(pExpr); }
	inline ExprList &GetExprList() { return _exprOwner; }
	inline const ExprList &GetExprList() const { return _exprOwner; }
};

//-----------------------------------------------------------------------------
// Expr_Value
//-----------------------------------------------------------------------------
class Expr_Value : public Expr {
protected:
	const Value _value;
public:
	inline Expr_Value(Number num) : Expr(EXPRTYPE_Value), _value(num) {}
	inline Expr_Value(const Complex &comp) : Expr(EXPRTYPE_Value), _value(comp) {}
	inline Expr_Value(const Value &value) : Expr(EXPRTYPE_Value), _value(value) {}
	inline Expr_Value(const Expr_Value &expr) : Expr(expr), _value(expr._value) {}
	virtual ~Expr_Value();
	virtual Expr *IncRef() const;
	inline const Value &GetValue() const { return _value; }
	virtual bool IsValue() const;
	virtual bool IsConstZero() const;
	virtual bool IsConstOne() const;
	virtual bool IsConstMinusOne() const;
	virtual bool IsConstMinus() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual ValueType GetStaticValueType() const;
	virtual void Accept(ExprVisitor &visitor) const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Symbol
//-----------------------------------------------------------------------------
class Expr_Symbol : public Expr {
protected:
	const Symbol *_pSymbol;
	SymbolSet _attrs;
	SymbolList _attrFront;
public:
	inline Expr_Symbol(const Symbol *pSymbol) : Expr(EXPRTYPE_Symbol), _pSymbol(pSymbol) {}
	inline Expr_Symbol(const Expr_Symbol &expr) : Expr(expr),
							_pSymbol(expr._pSymbol), _attrs(expr._attrs) {}
	virtual ~Expr_Symbol();
	virtual Expr *IncRef() const;
	virtual bool IsSymbol() const;
	virtual Expr *Clone() const;
	virtual Expr *CloneSubstituted(Environment &env, Signal sig) const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual void Accept(ExprVisitor &visitor) const;
	inline void AddAttr(const Symbol *pSymbol) { _attrs.Insert(pSymbol); }
	inline const SymbolSet &GetAttrs() const { return _attrs; }
	inline const Symbol *GetSymbol() const { return _pSymbol; }
	inline SymbolList &GetAttrFront() { return _attrFront; }
	inline const SymbolList &GetAttrFront() const { return _attrFront; }
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_BlockParam
//-----------------------------------------------------------------------------
class Expr_BlockParam : public Expr_Container {
public:
	inline Expr_BlockParam() : Expr_Container(EXPRTYPE_BlockParam) {}
	inline Expr_BlockParam(const Expr_BlockParam &expr) : Expr_Container(expr) {}
	virtual ~Expr_BlockParam();
	virtual bool IsBlockParam() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Block
//-----------------------------------------------------------------------------
class Expr_Block : public Expr_Container {
protected:
	Expr_BlockParam *_pExprBlockParam;	// this may be NULL
public:
	inline Expr_Block() : Expr_Container(EXPRTYPE_Block), _pExprBlockParam(NULL) {}
	inline Expr_Block(const Expr_Block &expr) : Expr_Container(expr),
		_pExprBlockParam((expr._pExprBlockParam == NULL)? NULL :
				dynamic_cast<Expr_BlockParam *>(expr._pExprBlockParam->Clone())) {}
	virtual ~Expr_Block();
	virtual bool IsBlock() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual void Accept(ExprVisitor &visitor) const;
	virtual String ToString(Signal sig) const;
	inline void SetParam(Expr_BlockParam *pExprBlockParam) {
		_pExprBlockParam = pExprBlockParam;
	}
	inline const Expr_BlockParam *GetParam() const { return _pExprBlockParam; }
};

//-----------------------------------------------------------------------------
// Expr_Lister
//-----------------------------------------------------------------------------
class Expr_Lister : public Expr_Container {
public:
	inline Expr_Lister() : Expr_Container(EXPRTYPE_Lister) {}
	inline Expr_Lister(const Expr_Lister &expr) : Expr_Container(expr) {}
	virtual ~Expr_Lister();
	virtual bool IsLister() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Complex
//-----------------------------------------------------------------------------
class Expr_Complex : public Expr {
protected:
	Expr *_pExprCar;
	Expr_Lister *_pExprLister;
public:
	inline Expr_Complex(ExprType exprType, Expr *pExprCar, Expr_Lister *pExprLister) :
			Expr(exprType), _pExprCar(pExprCar), _pExprLister(pExprLister) {}
	inline Expr_Complex(const Expr_Complex &expr) : Expr(expr),
			_pExprCar(expr._pExprCar->Clone()),
			_pExprLister(dynamic_cast<Expr_Lister *>(expr._pExprLister->Clone())) {}
	virtual ~Expr_Complex();
	virtual bool IsComplex() const;
	virtual Expr *IncRef() const;
	inline Expr *GetCar() { return _pExprCar; }
	inline const Expr *GetCar() const { return _pExprCar; }
	inline Expr_Lister *GetLister() { return _pExprLister; }
	inline const Expr_Lister *GetLister() const { return _pExprLister; }
	inline ExprList &GetExprList() { return _pExprLister->GetExprList(); }
	inline const ExprList &GetExprList() const { return _pExprLister->GetExprList(); }
};

//-----------------------------------------------------------------------------
// Expr_Indexer
//-----------------------------------------------------------------------------
class Expr_Indexer : public Expr_Complex {
private:
	class ValueVisitor_AssignList : public ValueVisitor {
	private:
		Object &_objDst;
		const ValueList &_valListSrc;
		ValueList::const_iterator _pValueSrc;
	public:
		inline ValueVisitor_AssignList(Object &objDst, const ValueList &valListSrc) :
								_objDst(objDst), _valListSrc(valListSrc),
								_pValueSrc(valListSrc.begin()) {}
		virtual void Visit(Signal sig, const Value &valueIdx);
	};
	class ValueVisitor_AssignValue : public ValueVisitor {
	private:
		Object &_objDst;
		const Value &_value;
	public:
		inline ValueVisitor_AssignValue(Object &objDst, const Value &value) :
								_objDst(objDst), _value(value) {}
		virtual void Visit(Signal sig, const Value &valueIdx);
	};
public:
	inline Expr_Indexer(Expr *pExprCar, Expr_Lister *pExprLister) :
			Expr_Complex(EXPRTYPE_Indexer, pExprCar, pExprLister) {}
	inline Expr_Indexer(const Expr_Indexer &expr) : Expr_Complex(expr) {}
	virtual ~Expr_Indexer();
	virtual Expr *IncRef() const;
	virtual bool IsIndexer() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual void Accept(ExprVisitor &visitor) const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Caller
//-----------------------------------------------------------------------------
class Expr_Caller : public Expr_Complex {
protected:
	Expr_Block *_pExprBlock;		// this may be NULL
	Expr_Caller *_pExprCallerSucc;	// this may be NULL
	SymbolSet _attrs;
	SymbolSet _attrsOpt;
	SymbolList _attrFront;
public:
	Expr_Caller(Expr *pExprCar, Expr_Lister *pExprLister, Expr_Block *pExprBlock);
	Expr_Caller(const Function *pFunc, Expr *pExprArg);
	Expr_Caller(const Function *pFunc, Expr *pExprArg1, Expr *pExprArg2);
	Expr_Caller(const Expr_Caller &expr);
	virtual ~Expr_Caller();
	virtual Expr *IncRef() const;
	virtual bool IsCaller() const;
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual void Accept(ExprVisitor &visitor) const;
	virtual String ToString(Signal sig) const;
	Value EvalEach(Environment &env, Signal sig,
				Value &valueSelf, const Function **ppFuncSuccRequester) const;
	inline void AddAttr(const Symbol *pSymbol) { _attrs.Insert(pSymbol); }
	inline void AddAttrOpt(const Symbol *pSymbol) { _attrsOpt.Insert(pSymbol); }
	inline const SymbolSet &GetAttrs() const { return _attrs; }
	inline const SymbolSet &GetAttrsOpt() const { return _attrsOpt; }
	inline SymbolList &GetAttrFront() { return _attrFront; }
	inline const SymbolList &GetAttrFront() const { return _attrFront; }
	inline void SetBlock(Expr_Block *pExprBlock) {
		Expr::Delete(_pExprBlock);
		_pExprBlock = pExprBlock;
	}
	inline void SetSucceeding(Expr_Caller *pExprCaller) {
		// _pExprCallerSucc must be NULL at this time
		_pExprCallerSucc = pExprCaller;
	}
	inline const Expr_Block *GetBlock() const { return _pExprBlock; }
	inline const Expr_Caller *GetSucceeding() const { return _pExprCallerSucc; }
	inline Expr_Caller *GetLastSucceeding() {
		return (_pExprCallerSucc == NULL)? this : _pExprCallerSucc->GetLastSucceeding();
	}
private:
	Value DoExec(Environment &env, Signal sig, const Function **ppFuncSuccRequester) const;
};

//-----------------------------------------------------------------------------
// Expr_UnaryOp
//-----------------------------------------------------------------------------
class Expr_UnaryOp : public Expr_Unary {
protected:
	const Function &_func;
	bool _suffixSymbolFlag;
public:
	inline Expr_UnaryOp(const Function &func, Expr *pExprChild, bool suffixSymbolFlag) :
					Expr_Unary(EXPRTYPE_UnaryOp, pExprChild),
					_func(func), _suffixSymbolFlag(suffixSymbolFlag) {}
	inline Expr_UnaryOp(const Expr_UnaryOp &expr) :
					Expr_Unary(expr),
					_func(expr._func), _suffixSymbolFlag(expr._suffixSymbolFlag) {}
	virtual ~Expr_UnaryOp();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual bool IsUnaryOp() const;
	virtual bool IsNeg() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_BinaryOp
//-----------------------------------------------------------------------------
class Expr_BinaryOp : public Expr_Binary {
protected:
	const Function &_func;
public:
	inline Expr_BinaryOp(const Function &func, Expr *pExprLeft, Expr *pExprRight) :
					Expr_Binary(EXPRTYPE_BinaryOp, pExprLeft, pExprRight), _func(func) {}
	inline Expr_BinaryOp(const Expr_BinaryOp &expr) :
					Expr_Binary(expr), _func(expr._func) {}
	virtual ~Expr_BinaryOp();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual bool IsBinaryOp() const;
	virtual bool IsPlus() const;
	virtual bool IsMinus() const;
	virtual bool IsMultiply() const;
	virtual bool IsDivide() const;
	virtual bool IsPower() const;
	virtual bool IsContainCheck() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Quote
//-----------------------------------------------------------------------------
class Expr_Quote : public Expr_Unary {
public:
	inline Expr_Quote(Expr *pExprChild) : Expr_Unary(EXPRTYPE_Quote, pExprChild) {}
	inline Expr_Quote(const Expr_Quote &expr) : Expr_Unary(expr) {}
	virtual ~Expr_Quote();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual ValueType GetStaticValueType() const;
	virtual const Expr *Unquote() const;
	virtual bool IsQuote() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Force
//-----------------------------------------------------------------------------
class Expr_Force : public Expr_Unary {
public:
	inline Expr_Force(Expr *pExprChild) : Expr_Unary(EXPRTYPE_Force, pExprChild) {}
	inline Expr_Force(const Expr_Force &expr) : Expr_Unary(expr) {}
	virtual ~Expr_Force();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual bool IsForce() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Prefix
//-----------------------------------------------------------------------------
class Expr_Prefix : public Expr_Unary {
private:
	const Symbol *_pSymbol;
public:
	inline Expr_Prefix(Expr *pExprChild, const Symbol *pSymbol) :
					Expr_Unary(EXPRTYPE_Prefix, pExprChild), _pSymbol(pSymbol) {}
	inline Expr_Prefix(const Expr_Prefix &expr) :
					Expr_Unary(expr), _pSymbol(expr._pSymbol) {}
	virtual ~Expr_Prefix();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual bool IsPrefix() const;
	inline const Symbol *GetSymbol() const { return _pSymbol; }
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Suffix
//-----------------------------------------------------------------------------
class Expr_Suffix : public Expr_Unary {
private:
	const Symbol *_pSymbol;
public:
	inline Expr_Suffix(Expr *pExprChild, const Symbol *pSymbol) :
					Expr_Unary(EXPRTYPE_Suffix, pExprChild), _pSymbol(pSymbol) {}
	inline Expr_Suffix(const Expr_Suffix &expr) :
					Expr_Unary(expr), _pSymbol(expr._pSymbol) {}
	virtual ~Expr_Suffix();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual bool IsSuffix() const;
	inline const Symbol *GetSymbol() const { return _pSymbol; }
	OccurPattern GetOccurPattern() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Assign
//-----------------------------------------------------------------------------
class Expr_Assign : public Expr_Binary {
private:
	const Function *_pFuncToApply;
public:
	inline Expr_Assign(Expr *pExprLeft, Expr *pExprRight, const Function *pFuncToApply) :
				Expr_Binary(EXPRTYPE_Assign, pExprLeft, pExprRight), _pFuncToApply(pFuncToApply) {}
	inline Expr_Assign(const Expr_Assign &expr) :
				Expr_Binary(expr), _pFuncToApply(expr._pFuncToApply) {}
	virtual ~Expr_Assign();
	virtual Value Exec(Environment &env, Signal sig) const;
	Value Exec(Environment &env, Signal sig,
				Environment &envDst, const SymbolSet *pSymbolsAssignable) const;
	virtual Expr *Clone() const;
	virtual bool IsAssign() const;
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_DictAssign
//-----------------------------------------------------------------------------
class Expr_DictAssign : public Expr_Binary {
public:
	inline Expr_DictAssign(Expr *pExprLeft, Expr *pExprRight) :
				Expr_Binary(EXPRTYPE_DictAssign, pExprLeft, pExprRight) {}
	inline Expr_DictAssign(const Expr_DictAssign &expr) : Expr_Binary(expr) {}
	virtual ~Expr_DictAssign();
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Expr *Clone() const;
	virtual bool IsDictAssign() const;
	virtual String ToString(Signal sig) const;
	Value GetKey(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Expr_Field
//-----------------------------------------------------------------------------
class Expr_Field : public Expr_Binary {
public:
	enum Mode {
		MODE_Normal, MODE_Mapping, MODE_MapToIter,
	};
private:
	Mode _mode;
public:
	inline Expr_Field(Expr *pExprLeft, Expr *pExprRight, Mode mode) :
				Expr_Binary(EXPRTYPE_Field, pExprLeft, pExprRight), _mode(mode) {}
	inline Expr_Field(const Expr_Field &expr) : Expr_Binary(expr), _mode(expr._mode) {}
	inline Mode GetMode() const { return _mode; }
	virtual ~Expr_Field();
	virtual Expr *Clone() const;
	virtual Value Exec(Environment &env, Signal sig) const;
	virtual Value DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const;
	virtual bool IsField() const;
	virtual String ToString(Signal sig) const;
};

}

#endif
