//-----------------------------------------------------------------------------
// AScript re module
//-----------------------------------------------------------------------------
#include "Module_csv.h"
#include "Object_File.h"

AScript_BeginModule(csv)

static bool ReadLine(Environment &env, Signal sig, File &file, ValueList &valList);
static void PutValue(Environment &env, Signal sig, File &file, const Value &value);

//-----------------------------------------------------------------------------
// AScript module functions: csv
//-----------------------------------------------------------------------------
// result = csv#read(file:File)
AScript_DeclareFunction(read)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
}

AScript_ImplementFunction(read)
{
	const Value valueEmpty(env, "");
	File &file = context.GetFile(0);
	Value result;
	ValueList &valListCol = result.InitAsList(env);
	ValueList valListFields;
	size_t nLine = 0;
	while (ReadLine(env, sig, file, valListFields)) {
		ValueList::iterator pValueCol = valListCol.begin();
		ValueList::iterator pValueColEnd = valListCol.end();
		foreach_const (ValueList, pValueField, valListFields) {
			if (pValueCol == pValueColEnd) {
				Value valueCol;
				ValueList &valList = valueCol.InitAsList(env, nLine, valueEmpty);
				valListCol.push_back(valueCol);
				valList.push_back(*pValueField);
			} else {
				pValueCol->GetList().push_back(*pValueField);
				pValueCol++;
			}
		}
		for ( ; pValueCol != pValueColEnd; pValueCol++) {
			pValueCol->GetList().push_back(valueEmpty);
		}
		nLine++;
		valListFields.clear();
	}
	return result;
}

// result = csv#readline(file:File)
AScript_DeclareFunction(readline)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
}

AScript_ImplementFunction(readline)
{
	File &file = context.GetFile(0);
	Value result;
	ValueList &valList = result.InitAsList(env);
	if (!ReadLine(env, sig, file, valList) || sig.IsSignalled()) return Value::Null;
	return result;
}

// result = csv#readlines(file:File, nlines?:number) {block?}
class Iterator_readlines : public Iterator {
private:
	Object_File *_pObj;
	int _nLinesMax;
	int _nLines;
public:
	inline Iterator_readlines(Object_File *pObj, int nLinesMax) :
			Iterator(false), _pObj(pObj), _nLinesMax(nLinesMax), _nLines(0) {}
	virtual ~Iterator_readlines();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

Iterator_readlines::~Iterator_readlines()
{
	Object::Delete(_pObj);
}

bool Iterator_readlines::DoNext(Signal sig, Value &value)
{
	if (_nLines == _nLinesMax) return false;
	_nLines++;
	File &file = _pObj->GetFile();
	ValueList &valList = value.InitAsList(*_pObj);
	return ReadLine(*_pObj, sig, file, valList);
}

String Iterator_readlines::ToString(Signal sig) const
{
	return String("<iterator:csv#readlines>");
}

AScript_DeclareFunction(readlines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "nlines", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(readlines)
{
	Object_File *pObjFile = context.GetFileObj(0);
	int nLinesMax = context.IsNumber(1)? context.GetNumber(1) : -1;
	return ReturnIterator(env, sig, context, new Iterator_readlines(
			dynamic_cast<Object_File *>(pObjFile->IncRef()), nLinesMax));
}

// csv#write(file:File, elems[]+)
AScript_DeclareFunction(write)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "elems", VTYPE_AnyType, true, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(write)
{
	File &file = context.GetFile(0);
	const ValueList &valList = context.GetList(1);
	if (valList.empty()) return Value::Null;
	IteratorOwner iteratorOwner;
	foreach_const (ValueList, pValue, valList) {
		Iterator *pIterator = pValue->CreateIterator(sig);
		if (pIterator == NULL) return Value::Null;
		iteratorOwner.push_back(pIterator);
	}
	for (;;) {
		bool doneFlag = true;
		ValueList valList;
		foreach_const (IteratorOwner, ppIterator, iteratorOwner) {
			Iterator *pIterator = *ppIterator;
			Value value;
			if (pIterator->Next(sig, value)) {
				if (sig.IsSignalled()) return Value::Null;
				doneFlag = false;
				valList.push_back(value);
			} else {
				valList.push_back(Value::Null);
			}
		}
		if (doneFlag) break;
		foreach (ValueList, pValue, valList) {
			if (pValue != valList.begin()) file.PutChar(',');
			if (pValue->IsValid()) {
				PutValue(env, sig, file, *pValue);
				if (sig.IsSignalled()) break;
			}
		}
		file.PutChar('\n');
	}
	return Value::Null;
}

// csv#writeline(file:File, elem[])
AScript_DeclareFunction(writeline)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
}

AScript_ImplementFunction(writeline)
{
	File &file = context.GetFile(0);
	const ValueList &valList = context.GetList(1);
	foreach_const (ValueList, pValue, valList) {
		if (pValue != valList.begin()) file.PutChar(',');
		PutValue(env, sig, file, *pValue);
	}
	file.PutChar('\n');
	return Value::Null;
}

// csv#writelines(file:File, iter:iterator)
AScript_DeclareFunction(writelines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(writelines)
{
	File &file = context.GetFile(0);
	Iterator *pIterator = context.GetIterator(1);
	Value value;
	while (pIterator->Next(sig, value)) {
		if (!value.IsList()) {
			sig.SetError(ERR_ValueError, "each element must be a list");
			return Value::Null;
		}
		const ValueList &valList = value.GetList();
		foreach_const (ValueList, pValue, valList) {
			if (pValue != valList.begin()) file.PutChar(',');
			if (pValue->IsList()) {
				const ValueList &valListSub = pValue->GetList();
				foreach_const (ValueList, pValueSub, valListSub) {
					if (pValueSub != valListSub.begin()) file.PutChar(',');
					PutValue(env, sig, file, *pValueSub);
					if (sig.IsSignalled()) return Value::Null;
				}
			} else {
				PutValue(env, sig, file, *pValue);
				if (sig.IsSignalled()) return Value::Null;
			}
		}
		file.PutChar('\n');
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value::Null;
}

// Module entry
AScript_ModuleEntry()
{
	AScript_AssignFunction(read);
	AScript_AssignFunction(readline);
	AScript_AssignFunction(readlines);
	AScript_AssignFunction(write);
	AScript_AssignFunction(writeline);
	AScript_AssignFunction(writelines);
	AScript_AssignValue(Symbol::Add("format"), Value(env, "%.6f"));
}

AScript_ModuleTerminate()
{
}

//-----------------------------------------------------------------------------
// Utilities
//-----------------------------------------------------------------------------
static bool ReadLine(Environment &env, Signal sig, File &file, ValueList &valList)
{
	enum {
		STAT_LineTop, STAT_FieldTop, STAT_Field, STAT_Quoted, STAT_QuotedEnd,
	} stat = STAT_LineTop;
	String field;
	int ch;
	bool rtn = true;
	bool eatNextChar = true;
	for (;;) {
		if (eatNextChar) {
			while ((ch = file.GetChar()) == '\r') ;
		}
		eatNextChar = true;
		if (stat == STAT_LineTop) {
			if (ch < 0) {
				rtn = false;
				break;
			}
			eatNextChar = false;
			stat = STAT_FieldTop;
		} else if (stat == STAT_FieldTop) {
			field.clear();
			if (ch == '"') {
				stat = STAT_Quoted;
			} else if (ch == '\n' || ch < 0) {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				eatNextChar = false;
				stat = STAT_Field;
			}
		} else if (stat == STAT_Field) {
			if (ch == ',') {
				valList.push_back(Value(env, field.c_str()));
				stat = STAT_FieldTop;
			} else if (ch == '\n' || ch < 0) {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				field.push_back(ch);
			}
		} else if (stat == STAT_Quoted) {
			if (ch == '"') {
				stat = STAT_QuotedEnd;
			} else if (ch < 0) {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				field.push_back(ch);
			}
		} else if (stat == STAT_QuotedEnd) {
			if (ch == '"') {
				field.push_back(ch);
				stat = STAT_Quoted;
			} else if (ch < 0) {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				eatNextChar = false;
				stat = STAT_Field;
			}
		}
	}
	return rtn;
}

static void PutValue(Environment &env, Signal sig, File &file, const Value &value)
{
	if (value.IsInvalid()) {
		// nothing to do
	} else if (value.IsNumberOrComplex()) {
		Value result(env, Formatter::Format(sig, "%g", ValueList(value)).c_str());
		if (sig.IsSignalled()) return;
		file.PutString(result.ToString(sig, false).c_str());
	} else if (value.IsString()) {
		file.PutChar('"');
		for (const char *p = value.GetString(); *p != '\0'; p++) {
			char ch = *p;
			file.PutChar(ch);
			if (ch == '"') file.PutChar(ch);
		}
		file.PutChar('"');
	} else {
		sig.SetError(ERR_TypeError, "can't output in CSV format");
	}
}

AScript_EndModule(csv)

AScript_DLLModuleEntry(csv)
