#include "Object.h"
#include "Object_Custom.h"
#include "Expr.h"
#include "Module.h"

namespace AScript {

//-----------------------------------------------------------------------------
// ObjectBase
//-----------------------------------------------------------------------------
ObjectBase::ObjectBase(Environment *pEnvOuter, EnvType envType) :
							Environment(pEnvOuter, envType), _cntRef(1)
{
}

ObjectBase::~ObjectBase()
{
}

bool ObjectBase::IsList() const { return false; }
bool ObjectBase::IsDict() const { return false; }
bool ObjectBase::IsFile() const { return false; }
bool ObjectBase::IsExpr() const { return false; }
bool ObjectBase::IsFunction() const { return false; }
bool ObjectBase::IsWrappedMethod() const { return false; }

bool ObjectBase::BuildContent(Environment &env, Signal sig, const Expr_Block *pExprBlock,
											const SymbolSet *pSymbolsAssignable)
{
	Environment envLocal(&env, ENVTYPE_Local);
	foreach_const (ExprList, ppExpr, pExprBlock->GetExprList()) {
		if ((*ppExpr)->IsAssign()) {
			const Expr_Assign *pExprAssign =
								dynamic_cast<const Expr_Assign *>(*ppExpr);
			pExprAssign->Exec(envLocal, sig, *this, pSymbolsAssignable);
		} else {
			(*ppExpr)->Exec(envLocal, sig);
		}
		if (sig.IsSignalled()) return false;
	}
	return true;
}

Iterator *ObjectBase::CreateIterator(Signal sig)
{
	sig.SetError(ERR_ValueError, "object cannot generate iterator");
	return NULL;
}

Value ObjectBase::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	sig.SetError(ERR_TypeError, "object is not callable");
	return Value::Null;
}

//-----------------------------------------------------------------------------
// Object
//-----------------------------------------------------------------------------
bool Object::IsObject() const { return true; }

Object::Object(Class *pClass) :
			ObjectBase(pClass, ENVTYPE_Instance), _pClass(pClass->IncRef())
{
}

Object::~Object()
{
	if (_pClass->IsCustom()) {
		Class_Custom *pClassCustom = dynamic_cast<Class_Custom *>(_pClass);
		const Function *pFunc =
				pClassCustom->LookupFunction(AScript_Symbol(__del__), false);
		if (pFunc != NULL) {
			Signal sig;
			Value valueSelf(this, VTYPE_Object, false); // reference to self
			Context context(SymbolSet::Null, SymbolSet::Null,
									NULL, NULL, valueSelf, ValueList::Null);
			pFunc->Eval(*this, sig, context);
			//::printf("%s\n", sig.GetErrString().c_str());
		}
	}
	Object::Delete(_pClass);
}

Object *Object::Clone() const
{
	return new Object(*this);
}

const char *Object::GetClassName() const
{
	return (_pClass == NULL)? "class" : _pClass->GetName();
}

Value Object::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (!valueIdx.IsSymbol()) {
		sig.SetError(ERR_IndexError, "index must be a symbol");
		return Value::Null;
	}
	const Symbol *pSymbol = valueIdx.GetSymbol();
	const Value *pValue = LookupValue(pSymbol, false);
	if (pValue == NULL) {
		sig.SetError(ERR_ValueError, "object doesn't have a property named '%s'", pSymbol->GetName());
		return Value::Null;
	}
	return *pValue;
}

void Object::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (!valueIdx.IsSymbol()) {
		sig.SetError(ERR_IndexError, "index must be a symbol");
		return;
	}
	const Symbol *pSymbol = valueIdx.GetSymbol();
	AssignValue(pSymbol, value, false);
}

Value Object::EvalMethod(Signal sig, const Symbol *pSymbol,
							const ValueList &valListArg, bool &evaluatedFlag)
{
	const Function *pFuncSuccRequester = NULL;
	evaluatedFlag = false;
	const Function *pFunc = LookupFunction(pSymbol, true);
	if (pFunc == NULL) return Value::Null;
	Value valueSelf(this, VTYPE_Object, false); // reference to self
	evaluatedFlag = true;
	Context context(SymbolSet::Null, SymbolSet::Null,
							NULL, &pFuncSuccRequester, valueSelf, valListArg);
	return pFunc->Eval(*this, sig, context);
}

Value Object::EvalGetter(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag)
{
	const Function *pFunc = LookupFunction(AScript_Symbol(__getprop__), true);
	if (pFunc == NULL) return Value::Null;
	Value valueSelf(this, VTYPE_Object, false); // reference to self
	evaluatedFlag = true;
	Context context(SymbolSet::Null, SymbolSet::Null,
					NULL, NULL, valueSelf, ValueList(Value(pSymbol)));
	return pFunc->Eval(*this, sig, context);
}

Value Object::EvalSetter(Signal sig, const Symbol *pSymbol,
										const Value &value, bool &evaluatedFlag)
{
	const Function *pFunc = LookupFunction(AScript_Symbol(__setprop__), true);
	if (pFunc == NULL) return Value::Null;
	Value valueSelf(this, VTYPE_Object, false); // reference to self
	evaluatedFlag = true;
	Context context(SymbolSet::Null, SymbolSet::Null,
					NULL, NULL, valueSelf, ValueList(Value(pSymbol), value));
	return pFunc->Eval(*this, sig, context);
}

String Object::ToString(Signal sig, bool exprFlag)
{
	bool evaluatedFlag = false;
	Value value = EvalMethod(sig, AScript_Symbol(__str__),
											ValueList::Null, evaluatedFlag);
	if (evaluatedFlag) return value.ToString(sig, false);
	String str;
	str += "<object:";
	str += _pClass->GetName();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object
//-----------------------------------------------------------------------------
// Object#clone()
AScript_DeclareMethod(Object, clone)
{
}

AScript_ImplementMethod(Object, clone)
{
	return Value(context.GetSelfObj()->Clone(), context.GetSelf().GetType());
}

// num = Object#tonumber():map:[strict,raise,zero,nil]
AScript_DeclareMethod(Object, tonumber)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(strict));
	DeclareAttr(AScript_Symbol(raise));
	DeclareAttr(AScript_Symbol(zero));
	DeclareAttr(AScript_Symbol(nil));
}

AScript_ImplementMethod(Object, tonumber)
{
	bool allowPartFlag = !context.IsSet(AScript_Symbol(strict));
	bool successFlag;
	Number num = context.GetSelf().ToNumber(allowPartFlag, successFlag);
	if (successFlag) {
		return Value(num);
	} else if (context.IsSet(AScript_Symbol(raise))) {
		sig.SetError(ERR_ValueError, "failed to convert to number");
		return Value::Null;
	} else if (context.IsSet(AScript_Symbol(zero))) {
		return Value(0.);
	} else { // context.IsSet(AScript_PrivSymbol(nil)
		return Value::Null;
	}
}

// num = Object#tostring()
AScript_DeclareMethod(Object, tostring)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Object, tostring)
{
	// bug bug bug bug bug
	String str = context.GetSelf().ToString(sig, false);
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, str.c_str());
}

// Object#tap() {block}
AScript_DeclareMethod(Object, tap)
{
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementMethod(Object, tap)
{
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	ValueList valListArg(context.GetSelf());
	Context contextSub(valListArg);
	pFuncBlock->Eval(env, sig, contextSub);
	if (sig.IsSignalled()) return Value::Null;
	return context.GetSelf();
}

// num = Object#__iter__()
AScript_DeclareMethod(Object, __iter__)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Object, __iter__)
{
	Iterator *pIterator = context.GetSelf().CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, pIterator);
}

// assignment
void Class::Prepare()
{
	AScript_AssignMethod(Object, clone);
	AScript_AssignMethod(Object, tonumber);
	//AScript_AssignMethod(Object, tostring);
	AScript_AssignMethod(Object, tap);
	AScript_AssignMethod(Object, __iter__);
}

//-----------------------------------------------------------------------------
// Class
//-----------------------------------------------------------------------------
bool Class::IsClass() const { return true; }
bool Class::IsCustom() const { return false; }

Class::Class(Environment *pEnvOuter, const Symbol *pSymbol) :
	ObjectBase(pEnvOuter, ENVTYPE_Class), _pSymbol(pSymbol),
	_pClassSuper(pEnvOuter->IsClass()? dynamic_cast<Class *>(pEnvOuter) : NULL),
	_pConstructor(NULL), _valType(VTYPE_Object)
{
	// don't assign functions here, because the function objects themselves
	// shall be constructed in Class_Function that is to be created after
	// this class!!!!!! instead, they must be assigned in Prepare() funtion below.
}

Class::~Class()
{
	Function::Delete(_pConstructor);
}

Object *Class::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object((pClass == NULL)? this : pClass);
}

String Class::ToString(Signal sig, bool exprFlag)
{
	String str;
	str += "<class:";
	str += GetName();
	str += ">";
	return str;
}

}
