//
// Object_List
//

#include "Object_List.h"
#include "Object_Function.h"
#include "Object_Bytes.h"
#include "Expr.h"
#include "combination.hpp"

//-----------------------------------------------------------------------------
// extra
//-----------------------------------------------------------------------------
namespace std {

// extracted from combination.hpp which is supposed to be used with boost
template <class BidirectionalIterator>
  bool next_partial_permutation(BidirectionalIterator first,
                                BidirectionalIterator middle,
                                BidirectionalIterator last)
{
  reverse (middle, last);
  return next_permutation(first, last);
}

}

namespace AScript {

//-----------------------------------------------------------------------------
// Object_List
//-----------------------------------------------------------------------------
bool Object_List::IsList() const { return true; }

Object *Object_List::Clone() const
{
	return new Object_List(*this);
}

Value Object_List::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (valueIdx.IsNumber()) {
		size_t idx = valueIdx.GetSizeT();
		if (idx < GetList().size()) {
			return GetList()[idx];
		} else {
			sig.SetError(ERR_IndexError, "index is out of range");
		}
	} else {
		sig.SetError(ERR_IndexError, "index must be a number for List");
	}
	return Value::Null;
}

void Object_List::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (valueIdx.IsNumber()) {
		size_t idx = valueIdx.GetSizeT();
		if (idx < GetList().size()) {
			GetList()[idx] = value;
		} else {
			sig.SetError(ERR_IndexError, "index is out of range");
		}
	} else {
		sig.SetError(ERR_IndexError, "index must be a number for List");
	}
}

Iterator *Object_List::CreateIterator(Signal sig)
{
	return new IteratorNormal(dynamic_cast<Object_List *>(IncRef()));
}

String Object_List::ToString(Signal sig, bool exprFlag)
{
	String str;
	const ValueList &valList = GetList();
	str += "[";
	foreach_const (ValueList, pValue, valList) {
		if (pValue != valList.begin()) str += ", ";
		str += pValue->ToString(sig);
	}
	str += "]";
	return str;
}

void Object_List::DoPadding(Signal sig, ValueList &valList, const Value &value)
{
	if (sig.IsSignalled()) return;	// detect stack overflow
	int sizeMax = -1;
	foreach (ValueList, pValue, valList) {
		if (pValue->IsList()) {
			int size = static_cast<int>(pValue->GetList().size());
			if (sizeMax < size) sizeMax = size;
		}
	}
	if (sizeMax < 0) return;
	foreach (ValueList, pValue, valList) {
		if (pValue->IsList()) {
			ValueList &valListElem = pValue->GetList();
			int sizeToPadding = sizeMax - static_cast<int>(valListElem.size());
			while (sizeToPadding-- > 0) {
				valListElem.push_back(value);
			}
			DoPadding(sig, valListElem, value);
		}
	}
}

Value Object_List::SortRank(Signal sig,
						const Value &valDirective, bool rankFlag, bool stableFlag)
{
	Environment &env = *this;
	enum { MODE_Ascend, MODE_Descend, MODE_Custom } mode = MODE_Ascend;
	const Function *pFunc = NULL;
	const ValueList &valList = GetList();
	if (valDirective.IsInvalid()) {
		// nothing to do
	} else if (valDirective.IsSymbol()) {
		const Symbol *pSymbol = valDirective.GetSymbol();
		if (pSymbol->IsIdentical(AScript_Symbol(ascend))) {
			mode = MODE_Ascend;
		} else if (pSymbol->IsIdentical(AScript_Symbol(descend))) {
			mode = MODE_Descend;
		} else {
			sig.SetError(ERR_ValueError,
				"invalid symbol '%s'", pSymbol->GetName());
			return Value::Null;
		}
	} else if (valDirective.IsFunction()) {
		mode = MODE_Custom;
		pFunc = valDirective.GetFunction();
		if (pFunc->GetDeclList().size() != 2) {
			sig.SetError(ERR_TypeError,
				"only a binary function can be specified");
			return Value::Null;
		}
	} else {
		sig.SetError(ERR_TypeError, "invalid argument");
		return Value::Null;
	}
#if 0
	const Value **valPtrList = new const Value *[valList.size()];
	do {
		const Value **ppValue = valPtrList;
		foreach_const (ValueList, pValue, valList) {
			*ppValue = &*pValue;
			ppValue++;
		}
	} while (0);
	::qsort(valPtrList, valList.size(), sizeof(Value *), Comparator);
	Value result;
	ValueList &valListResult = result.InitAsList(env);
	if (rankFlag) {
	} else {
		const Value **ppValue = valPtrList;
		for (size_t cnt = valList.size(); cnt > 0; cnt--, ppValue++) {
			valListResult.push_back(**ppValue);
		}
	}
	delete[] ppValPtrList;
#else
	ValuePtrList valPtrList;
	foreach_const (ValueList, pValue, valList) {
		valPtrList.push_back(&(*pValue));
	}
	if (stableFlag) {
		if (mode == MODE_Ascend) {
			std::stable_sort(valPtrList.begin(), valPtrList.end(), Comparator_Ascend());
		} else if (mode == MODE_Descend) {
			std::stable_sort(valPtrList.begin(), valPtrList.end(), Comparator_Descend());
		} else { // mode == MODE_Custom
			std::stable_sort(valPtrList.begin(), valPtrList.end(),
											Comparator_Custom(env, sig, pFunc));
		}
	} else {
		if (mode == MODE_Ascend) {
			std::sort(valPtrList.begin(), valPtrList.end(), Comparator_Ascend());
		} else if (mode == MODE_Descend) {
			std::sort(valPtrList.begin(), valPtrList.end(), Comparator_Descend());
		} else { // mode == MODE_Custom
			std::sort(valPtrList.begin(), valPtrList.end(),
											Comparator_Custom(env, sig, pFunc));
		}
	}
	Value result;
	ValueList &valListResult = result.InitAsList(env);
	if (rankFlag) {
		foreach_const (ValueList, pValue, valList) {
			ValuePtrList::iterator ppValue = valPtrList.begin();
			for ( ; ppValue != valPtrList.end(); ppValue++) {
				if (Value::Compare(*pValue, **ppValue) == 0) break;
			}
			if (ppValue == valPtrList.end()) {
				sig.SetError(ERR_SystemError, "fatal error in rank() operation");
				return Value::Null;
			} else {
				size_t idx = ppValue - valPtrList.begin();
				valListResult.push_back(Value(idx));
			}
		}
	} else {
		foreach_const (ValuePtrList, ppValue, valPtrList) {
			valListResult.push_back(**ppValue);
		}
	}
#endif
	return result;
}

void Object_List::ValueVisitorEx::Visit(Signal sig, const Value &value)
{
	ASSUME(_env, value.IsNumber());
	size_t idx = value.GetSizeT();
	if (std::find(_indexList.begin(), _indexList.end(), idx) != _indexList.end()) {
		// nothing to do
	} else if (idx < _valList.size()) {
		_indexList.push_back(idx);
	} else {
		sig.SetError(ERR_IndexError, "index is out of range");
	}
}

//-----------------------------------------------------------------------------
// Object_List::IteratorNormal
//-----------------------------------------------------------------------------
Object_List::IteratorNormal::~IteratorNormal()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorNormal::DoNext(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == _pValueEnd) return false;
	value = *_pValue;
	_pValue++;
	return true;
}

String Object_List::IteratorNormal::ToString(Signal sig) const
{
	return String("<iterator:list#normal>");
}

//-----------------------------------------------------------------------------
// Object_List::IteratorReverse
//-----------------------------------------------------------------------------
Object_List::IteratorReverse::~IteratorReverse()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorReverse::DoNext(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == valList.rend()) return false;
	value = *_pValue;
	_pValue++;
	return true;
}

String Object_List::IteratorReverse::ToString(Signal sig) const
{
	return String("<iterator:list#reverse>");
}

//-----------------------------------------------------------------------------
// Object_List::IteratorRound
//-----------------------------------------------------------------------------
Object_List::IteratorRound::~IteratorRound()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorRound::DoNext(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == valList.end() || _cnt == 0) return false;
	value = *_pValue;
	_pValue++;
	if (_cnt > 0) _cnt--;
	if (_pValue == valList.end()) _pValue = valList.begin();
	return true;
}

String Object_List::IteratorRound::ToString(Signal sig) const
{
	return String("<iterator:list#round>");
}

//-----------------------------------------------------------------------------
// Object_List::IteratorFold
//-----------------------------------------------------------------------------
Object_List::IteratorFold::~IteratorFold()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorFold::DoNext(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_offset >= valList.size()) return false;
	Environment &env = *_pObj;
	value.InitAsIterator(env, new IteratorNormal(
			dynamic_cast<Object_List *>(_pObj->IncRef()), _offset, _cntPerFold));
	_offset += _cntStep;
	return true;
}

String Object_List::IteratorFold::ToString(Signal sig) const
{
	return String("<iterator:list#fold>");
}

//-----------------------------------------------------------------------------
// Object_List::IteratorPermutation
//-----------------------------------------------------------------------------
Object_List::IteratorPermutation::IteratorPermutation(Object_List *pObj, int cnt) :
							Iterator(false), _pObj(pObj), _cnt(cnt), _validFlag(true)
{
	ValueList &valList = _pObj->GetList();
	_indexList.reserve(valList.size());
	for (size_t index = 0; index < valList.size(); index++) {
		_indexList.push_back(index);
	}
}

Object_List::IteratorPermutation::~IteratorPermutation()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorPermutation::DoNext(Signal sig, Value &value)
{
	if (!_validFlag) return false;
	ValueList &valList = value.InitAsList(*_pObj);
	ValueList &valListSrc = _pObj->GetList();
	if (_cnt < 0) {
		foreach (IndexList, pIndex, _indexList) {
			valList.push_back(valListSrc[*pIndex]);
		}
		_validFlag = std::next_permutation(_indexList.begin(), _indexList.end());
	} else {
		IndexList::iterator pIndex = _indexList.begin();
		for (int i = 0; i < _cnt; pIndex++, i++) {
			valList.push_back(valListSrc[*pIndex]);
		}
		_validFlag = std::next_partial_permutation(
				_indexList.begin(), _indexList.begin() + _cnt, _indexList.end());
	}
	return true;
}

String Object_List::IteratorPermutation::ToString(Signal sig) const
{
	return String("<iterator:list#permutation>");
}

//-----------------------------------------------------------------------------
// Object_List::IteratorCombination
//-----------------------------------------------------------------------------
Object_List::IteratorCombination::IteratorCombination(Object_List *pObj, int cnt) :
							Iterator(false), _pObj(pObj), _cnt(cnt), _validFlag(true)
{
	ValueList &valList = _pObj->GetList();
	_indexList.reserve(valList.size());
	for (size_t index = 0; index < valList.size(); index++) {
		_indexList.push_back(index);
	}
}

Object_List::IteratorCombination::~IteratorCombination()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorCombination::DoNext(Signal sig, Value &value)
{
	if (!_validFlag) return false;
	ValueList &valList = value.InitAsList(*_pObj);
	ValueList &valListSrc = _pObj->GetList();
	IndexList::iterator pIndex = _indexList.begin();
	for (int i = 0; i < _cnt; pIndex++, i++) {
		valList.push_back(valListSrc[*pIndex]);
	}
	// the following function is implemented in combination.hpp
	_validFlag = boost::next_combination(
			_indexList.begin(), _indexList.begin() + _cnt, _indexList.end());
	return true;
}

String Object_List::IteratorCombination::ToString(Signal sig) const
{
	return String("<iterator:list#combination>");
}

//-----------------------------------------------------------------------------
// Object_List::Comparator_Custom
//-----------------------------------------------------------------------------
bool Object_List::Comparator_Custom::
				operator()(const Value *pValue1, const Value *pValue2) const
{
	if (_sig.IsSignalled()) return false;
	ValueList valListArg(*pValue1, *pValue2);
	Context contextSub(valListArg);
	Value value = _pFunc->Eval(_env, _sig, contextSub);
	return value.GetNumber() < 0;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_List
//-----------------------------------------------------------------------------
// List#len()
AScript_DeclareMethod(List, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, len)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t cnt = pSelf->GetList().size();
	return Value(cnt);
}

// List#min():[index,last_index,indices]
AScript_DeclareMethod(List, min)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(List, min)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Iterator *pIterator = pSelf->CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	Value result = pIterator->MinMax(env, sig, false, context.GetAttrs());
	Iterator::Delete(pIterator);
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// List#max():[index,last_index,indices]
AScript_DeclareMethod(List, max)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(List, max)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Iterator *pIterator = pSelf->CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	Value result = pIterator->MinMax(env, sig, true, context.GetAttrs());
	Iterator::Delete(pIterator);
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// List#filter(criteria) {block?}
AScript_DeclareMethod(List, filter)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "criteria", VTYPE_AnyType);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, filter)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Iterator *pIteratorSrc = pSelf->CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	Iterator *pIterator = NULL;
	if (context.IsFunction(0)) {
		Object_Function *pFuncObjCriteria = 
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef());
		pIterator = new Iterator_Filter(env, pIteratorSrc, pFuncObjCriteria);
	} else if (context.IsList(0) || context.IsIterator(0)) {
		Iterator *pIteratorCriteria = context.GetValue(0).CreateIterator(sig);
		pIterator = new Iterator_FilterEach(pIteratorSrc, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for filter");
		return Value::Null;
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// List#map(func:Function) {block?}
AScript_DeclareMethod(List, map)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "func", VTYPE_Function);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, map)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Iterator *pIteratorSrc = pSelf->CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	Iterator *pIterator = new Iterator_Map(env, pIteratorSrc,
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// List#reduce(accum) {block}
AScript_DeclareMethod(List, reduce)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "accum", VTYPE_AnyType);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementMethod(List, reduce)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Iterator *pIterator = pSelf->CreateIterator(sig);
	if (sig.IsSignalled()) return Value::Null;
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	return pIterator->Reduce(env, sig, context.GetValue(0), pFuncBlock);
}

// List#count(value)
AScript_DeclareMethod(List, count)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementMethod(List, count)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t cnt = 0;
	const Value &value = context.GetValue(0);
	if (value.IsFunction()) {
		const Function *pFunc = value.GetFunction();
		foreach_const (ValueList, pValue, pSelf->GetList()) {
			ValueList valListArg(*pValue);
			Context contextSub(context, valListArg, Value::Null);
			Value valueFlag = pFunc->Eval(env, sig, contextSub);
			if (sig.IsSignalled()) return Value::Null;
			if (valueFlag.GetBoolean()) cnt++;
		}
	} else {
		foreach_const (ValueList, pValue, pSelf->GetList()) {
			if (Value::Compare(*pValue, value) == 0) cnt++;
		}
	}
	return Value(cnt);
}

// List#get(index):map
AScript_DeclareMethod(List, get)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "index", VTYPE_AnyType);
}

AScript_ImplementMethod(List, get)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return pSelf->GetByIndex(sig, context.GetValue(0));
}

// List#first()
AScript_DeclareMethod(List, first)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, first)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	if (valList.empty()) {
		sig.SetError(ERR_ValueError, "list is empty");
		return Value::Null;
	}
	return valList.front();
}

// List#last()
AScript_DeclareMethod(List, last)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, last)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	if (valList.empty()) {
		sig.SetError(ERR_ValueError, "list is empty");
		return Value::Null;
	}
	return valList.back();
}

// List#sort(directive?):[stable]
AScript_DeclareMethod(List, sort)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "directive", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(stable));
}

AScript_ImplementMethod(List, sort)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return pSelf->SortRank(sig, context.GetValue(0),
							false, context.IsSet(AScript_Symbol(stable)));
}

// List#rank(directive?):[stable]
AScript_DeclareMethod(List, rank)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "directive", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(stable));
}

AScript_ImplementMethod(List, rank)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return pSelf->SortRank(sig, context.GetValue(0),
							true, context.IsSet(AScript_Symbol(stable)));
}

// List#join(sep?:string)
AScript_DeclareMethod(List, join)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "sep", VTYPE_String, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, join)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	return Value(env, Join(valList,
			context.IsString(0)? context.GetString(0) : "").c_str());
}

// List#format(format:string)
AScript_DeclareMethod(List, format)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
}

AScript_ImplementMethod(List, format)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return Value(env, Formatter::Format(sig,
						context.GetString(0), pSelf->GetList()).c_str());
}

// List#pack(format:string)
AScript_DeclareMethod(List, pack)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
}

AScript_ImplementMethod(List, pack)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Value result;
	Object_Bytes *pObjBytes = result.InitAsBytes(env);
	size_t offset = 0;
	pObjBytes->Pack(sig, offset, context.GetString(0), pSelf->GetList());
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// iter = List#each() {block?}
AScript_DeclareMethod(List, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, each)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorNormal(pObj));
}

// iter = List#offset(n:number):map {block?}
AScript_DeclareMethod(List, offset)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareBlock(OCCUR_Once);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, offset)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t offset = context.GetSizeT(0);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorNormal(pObj, offset));
}

// iter = List#skip(n:number):map {block?}
AScript_DeclareMethod(List, skip)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, skip)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	Iterator *pIterator = new Object_List::IteratorNormal(pObj);
	pIterator = new Iterator_Skip(pIterator, context.GetSizeT(0));
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#align(n:number, value?):map {block?}
AScript_DeclareMethod(List, align)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, align)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	Iterator *pIterator = new Object_List::IteratorNormal(pObj);
	pIterator = new Iterator_Align(pIterator, context.GetInt(0), context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#skipnil() {block?}
AScript_DeclareMethod(List, skipnil)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, skipnil)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	Iterator *pIterator = new Object_List::IteratorNormal(pObj);
	pIterator = new Iterator_SkipInvalid(pIterator);
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#head(n:number):map {block?}
AScript_DeclareMethod(List, head)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, head)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	int cnt = context.GetInt(0);
	Iterator *pIterator = new Object_List::IteratorNormal(pObj, 0, cnt);
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#tail(n:number):map {block?}
AScript_DeclareMethod(List, tail)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, tail)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	int cnt = context.GetInt(0);
	int cntMax = static_cast<int>(pObj->GetList().size());
	size_t offset = (cntMax > cnt)? cntMax - cnt : cntMax;
	Iterator *pIterator = new Object_List::IteratorNormal(pObj, offset);
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#reverse() {block?}
AScript_DeclareMethod(List, reverse)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, reverse)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorReverse(pObj));
}

// iter = List#round(n:number) {block?}
AScript_DeclareMethod(List, round)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, round)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	int cnt = context.GetInt(0);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorRound(pObj, cnt));
}

// iter = List#fold(n:number, nstep?:number) {block?}
AScript_DeclareMethod(List, fold)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "nstep", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, fold)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t cnt = context.GetSizeT(0);
	size_t cntStep = context.IsNumber(1)? context.GetSizeT(1) : cnt;
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
						new Object_List::IteratorFold(pObj, cnt, cntStep));
}

// iter = List#permutation(n?:number) {block?}
AScript_DeclareMethod(List, permutation)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, permutation)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	int cnt = context.IsNumber(0)? context.GetInt(0) : -1;
	if (cnt > 0 && pSelf->GetList().size() < static_cast<size_t>(cnt)) {
		sig.SetError(ERR_ValueError, "specified size is out of range");
		return Value::Null;
	}
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorPermutation(pObj, cnt));
}

// iter = List#combination(n:number) {block?}
AScript_DeclareMethod(List, combination)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, combination)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	int cnt = context.GetInt(0);
	if (pSelf->GetList().size() < static_cast<size_t>(cnt)) {
		sig.SetError(ERR_ValueError, "specified size is out of range");
		return Value::Null;
	}
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorCombination(pObj, cnt));
}

// List#clear!()
AScript_DeclareMethod(List, clear_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, clear_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	pSelf->GetList().clear();
	return context.GetSelf();
}

// List#shuffle!()
AScript_DeclareMethod(List, shuffle_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, shuffle_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	RandomNumberGenerator randomNumberGenerator;
	std::random_shuffle(valList.begin(), valList.end(), randomNumberGenerator);
	return context.GetSelf();
}

// List#add!(elem+)
AScript_DeclareMethod(List, add_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, false, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(List, add_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		valList.push_back(*pValue);
	}
	return context.GetSelf();
}

// List#append!(elem*)
AScript_DeclareMethod(List, append_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(List, append_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsList()) {
			foreach_const (ValueList, pValueSub, pValue->GetList()) {
				valList.push_back(*pValueSub);
			}
		} else {
			valList.push_back(*pValue);
		}
	}
	return context.GetSelf();
}

// List#erase!(idx+:number)
AScript_DeclareMethod(List, erase_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "idx", VTYPE_Number, false, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(List, erase_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	Object_List::ValueVisitorEx visitor(env, valList);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pValue->Accept(sig, visitor);
	}
	Object_List::IndexList &indexList = visitor.GetIndexList();
	if (!indexList.empty()) {
		std::sort(indexList.begin(), indexList.end());
		size_t offset = 0;
		foreach_const (Object_List::IndexList, pIdx, indexList) {
			size_t idx = *pIdx;
			valList.erase(valList.begin() + idx - offset);
			offset++;
		}
	}
	return context.GetSelf();
}

// List#padding!(value)
AScript_DeclareMethod(List, padding_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementMethod(List, padding_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List::DoPadding(sig, pSelf->GetList(), context.GetValue(0));
	if (sig.IsSignalled()) return Value::Null;
	return context.GetSelf();
}

// assignment
Class_List::Class_List(Class *pClassSuper) : Class(pClassSuper)
{
	AScript_AssignMethod(List, len);
	AScript_AssignMethod(List, min);
	AScript_AssignMethod(List, max);
	AScript_AssignMethod(List, filter);
	AScript_AssignMethod(List, map);
	AScript_AssignMethod(List, reduce);
	AScript_AssignMethod(List, count);
	AScript_AssignMethod(List, get);
	AScript_AssignMethod(List, first);
	AScript_AssignMethod(List, last);
	AScript_AssignMethod(List, sort);
	AScript_AssignMethod(List, rank);
	AScript_AssignMethod(List, join);
	AScript_AssignMethod(List, format);
	AScript_AssignMethod(List, pack);
	AScript_AssignMethod(List, each);
	AScript_AssignMethod(List, offset);
	AScript_AssignMethod(List, skip);
	AScript_AssignMethod(List, align);
	AScript_AssignMethod(List, skipnil);
	AScript_AssignMethod(List, head);
	AScript_AssignMethod(List, tail);
	AScript_AssignMethod(List, reverse);
	AScript_AssignMethod(List, round);
	AScript_AssignMethod(List, fold);
	AScript_AssignMethod(List, permutation);
	AScript_AssignMethod(List, combination);
	AScript_AssignMethodEx(List, clear_X, "clear!");
	AScript_AssignMethodEx(List, shuffle_X, "shuffle!");
	AScript_AssignMethodEx(List, add_X, "add!");
	AScript_AssignMethodEx(List, append_X, "append!");
	AScript_AssignMethodEx(List, erase_X, "erase!");
	AScript_AssignMethodEx(List, padding_X, "padding!");
}

Object *Class_List::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_List((pClass == NULL)? this : pClass);
}

}
