#include "Object_Bytes.h"
#include "Codec.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Bytes
//-----------------------------------------------------------------------------
Object_Bytes::Object_Bytes(const Object_Bytes &obj) : Object(obj), _bytes(obj._bytes)
{
}

Object_Bytes::~Object_Bytes()
{
}

Object *Object_Bytes::Clone() const
{
	return new Object_Bytes(*this);
}

Value Object_Bytes::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (!valueIdx.IsNumber()) {
		sig.SetError(ERR_IndexError, "index must be a number for bytes");
		return Value::Null;
	}
	int idx = valueIdx.GetInt();
	int len = _bytes.size();
	if (idx >= 0) {
		if (idx >= len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		return Value(static_cast<unsigned char>(_bytes[idx]));
	} else {
		if (-idx > len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		return Value(static_cast<unsigned char>(_bytes[len + idx]));
	}
}

void Object_Bytes::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (!valueIdx.IsNumber()) {
		sig.SetError(ERR_IndexError, "index must be a number for bytes");
		return;
	}
	if (!value.IsNumber()) {
		sig.SetError(ERR_IndexError, "value must be a number for bytes");
		return;
	}
	int idx = valueIdx.GetInt();
	long data = value.GetLong();
	if (data < 0 || data > 255) {
		sig.SetError(ERR_IndexError, "value must be between 0 and 255");
		return;
	}
	int len = _bytes.size();
	if (idx >= 0) {
		if (idx >= len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return;
		}
		_bytes[idx] = static_cast<unsigned char>(data);
	} else {
		if (-idx > len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return;
		}
		_bytes[len + idx] = static_cast<unsigned char>(data);
	}
}

Iterator *Object_Bytes::CreateIterator(Signal sig)
{
	return new IteratorByte(dynamic_cast<Object_Bytes *>(IncRef()), -1);
}

String Object_Bytes::ToString(Signal sig, bool exprFlag)
{
	char buff[64];
	::sprintf(buff, "<bytes:%dbyte>", _bytes.size());
	return String(buff);
}

bool Object_Bytes::Pack(Signal sig, size_t &offset,
									const char *format, const ValueList &valList)
{
	enum {
		STAT_Format,
		STAT_Repeat,
		STAT_Encoding,
	} stat = STAT_Format;
	Environment &env = *this;
	ValueList::const_iterator pValue = valList.begin();
	bool bigEndianFlag = IsBigEndian();
	int nRepeat = 1;
	String encoding;
	std::auto_ptr<Codec> pCodec(new Codec_Encoder_USASCII(false));
	for (const char *p = format; *p != '\0'; ) {
		char ch = *p;
		bool eatNextFlag = true;
		if (stat == STAT_Repeat) {
			if (IsDigit(ch)) {
				nRepeat = nRepeat * 10 + (ch - '0');
			} else {
				eatNextFlag = false;
				stat = STAT_Format;
			}
		} else if (stat == STAT_Encoding) {
			if (ch == '}') {
				if (encoding.empty()) {
					pCodec.reset(new Codec_None());
				} else {
					CodecFactory *pCodecFactory =
									CodecFactory::Lookup(encoding.c_str());
					if (pCodecFactory == NULL) {
						sig.SetError(ERR_CodecError,
								"unsupported encoding '%s'", encoding.c_str());
						return false;
					}
					pCodec.reset(pCodecFactory->CreateEncoder(false));
				}
				stat = STAT_Format;
			} else {
				encoding.push_back(ch);
			}
		} else if (IsDigit(ch)) {
			nRepeat = 0;
			eatNextFlag = false;
			stat = STAT_Repeat;
		} else if (ch == '{') {
			encoding.clear();
			stat = STAT_Encoding;
		} else if (ch == '@') {
			bigEndianFlag = IsBigEndian();
		} else if (ch == '=') {
			bigEndianFlag = IsBigEndian();
		} else if (ch == '<') {
			bigEndianFlag = false;
		} else if (ch == '>') {
			bigEndianFlag = true;
		} else if (ch == '!') {
			bigEndianFlag = true;
		} else if (ch == 'x') {
			if (!PackForward(sig, offset, nRepeat)) return false;
			offset += nRepeat;
			nRepeat = 1;
		} else if (ch == 'c') {
			if (!PackForward(sig, offset, nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte++, pValue++) {
				if (!CheckString(sig, valList, pValue)) return false;
				*pByte = pValue->GetString()[0];
			}
			nRepeat = 1;
		} else if (ch == 'b') {
			if (!PackForward(sig, offset, nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte++, pValue++) {
				if (!CheckNumber(sig, valList, pValue, -128, 127)) return false;
				*pByte = pValue->GetChar();
			}
			nRepeat = 1;
		} else if (ch == 'B') {
			if (!PackForward(sig, offset, nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte++, pValue++) {
				if (!CheckNumber(sig, valList, pValue, 0, 255)) return false;
				*pByte = pValue->GetUChar();
			}
			nRepeat = 1;
		} else if (ch == 'h') {
			if (!PackForward(sig, offset, 2 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 2 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 2, pValue++) {
				if (!CheckNumber(sig, valList, pValue, -32768, 32767)) return false;
				unsigned short num = pValue->GetShort();
				PackUShort(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'H') {
			if (!PackForward(sig, offset, 2 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 2 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 2, pValue++) {
				if (!CheckNumber(sig, valList, pValue, 0, 65535)) return false;
				unsigned short num = pValue->GetUShort();
				PackUShort(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'i') {
			if (!PackForward(sig, offset, 4 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 4 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 4, pValue++) {
				if (!CheckNumber(sig, valList, pValue, -2147483648., 2147483647.)) return false;
				unsigned long num = pValue->GetInt();
				PackULong(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'I') {
			if (!PackForward(sig, offset, 4 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 4 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 4, pValue++) {
				if (!CheckNumber(sig, valList, pValue, 0, 4294967295.)) return false;
				unsigned long num = pValue->GetUInt();
				PackULong(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'l') {
			if (!PackForward(sig, offset, 4 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 4 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 4, pValue++) {
				if (!CheckNumber(sig, valList, pValue, -2147483648., 2147483647.)) return false;
				unsigned long num = pValue->GetLong();
				PackULong(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'L') {
			if (!PackForward(sig, offset, 4 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 4 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pByte += 4, pValue++) {
				if (!CheckNumber(sig, valList, pValue, 0, 4294967295.)) return false;
				unsigned long num = pValue->GetULong();
				PackULong(pByte, bigEndianFlag, num);
			}
			nRepeat = 1;
		} else if (ch == 'q') {
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return false;
		} else if (ch == 'Q') {
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return false;
		} else if (ch == 'f') {
			if (!PackForward(sig, offset, 4 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 4 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pValue++) {
				if (!CheckNumber(sig, valList, pValue)) return false;
				float num = static_cast<float>(pValue->GetNumber());
				unsigned char *buff = reinterpret_cast<unsigned char *>(&num);
				for (int j = 0; j < 4; j++, pByte++) *pByte = buff[j];
			}
			nRepeat = 1;
		} else if (ch == 'd') {
			if (!PackForward(sig, offset, 8 * nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += 8 * nRepeat;
			for (int i = 0; i < nRepeat; i++, pValue++) {
				if (!CheckNumber(sig, valList, pValue)) return false;
				double num = static_cast<double>(pValue->GetNumber());
				unsigned char *buff = reinterpret_cast<unsigned char *>(&num);
				for (int j = 0; j < 8; j++, pByte++) *pByte = buff[j];
			}
			nRepeat = 1;
		} else if (ch == 's') {
			if (!PackForward(sig, offset, nRepeat)) return false;
			Bytes::iterator pByte = _bytes.begin() + offset;
			offset += nRepeat;
			if (!CheckString(sig, valList, pValue)) return false;
			const char *p = pValue->GetString();
			int nPacked = 0;
			char chConv;
			for ( ; nPacked < nRepeat && *p != '\0'; p++) {
				Codec::Result result = pCodec->FeedChar(*p, chConv);
				if (result == Codec::RESULT_Error) {
					sig.SetError(ERR_CodecError,
						"encoding error. specify a proper coding name by {coding}");
					return false;
				} else if (result == Codec::RESULT_Complete) {
					*pByte++ = chConv, nPacked++;
					while (pCodec->FollowChar(chConv) && nPacked < nRepeat) {
						*pByte++ = chConv, nPacked++;
					}
				}
			}
			for ( ; nPacked < nRepeat; nPacked++, pByte++) {
				*pByte = '\0';
			}
			pValue++;
			nRepeat = 1;
		} else if (ch == 'p') {
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return false;
		} else if (ch == 'P') {
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return false;
		} else if (IsWhite(ch)) {
			// just ignore white characters
		} else {
			sig.SetError(ERR_ValueError, "invalid character in format");
			return false;
		}
		if (eatNextFlag) p++;
	}
	return true;
}

Value Object_Bytes::Unpack(Signal sig,
					size_t &offset, const char *format, bool exceedErrorFlag)
{
	enum {
		STAT_Format,
		STAT_Repeat,
		STAT_Encoding,
	} stat = STAT_Format;
	Environment &env = *this;
	Value result;
	ValueList &valList = result.InitAsList(env);
	bool bigEndianFlag = IsBigEndian();
	int nRepeat = 1;
	String encoding;
	std::auto_ptr<Codec> pCodec(new Codec_Decoder_USASCII(false));
	for (const char *p = format; *p != '\0'; ) {
		char ch = *p;
		bool eatNextFlag = true;
		if (stat == STAT_Repeat) {
			if (IsDigit(ch)) {
				nRepeat = nRepeat * 10 + (ch - '0');
			} else {
				eatNextFlag = false;
				stat = STAT_Format;
			}
		} else if (stat == STAT_Encoding) {
			if (ch == '}') {
				if (encoding.empty()) {
					pCodec.reset(new Codec_None());
				} else {
					CodecFactory *pCodecFactory =
									CodecFactory::Lookup(encoding.c_str());
					if (pCodecFactory == NULL) {
						sig.SetError(ERR_CodecError,
								"unsupported encoding '%s'", encoding.c_str());
						return Value::Null;
					}
					pCodec.reset(pCodecFactory->CreateDecoder(false));
				}
				stat = STAT_Format;
			} else {
				encoding.push_back(ch);
			}
		} else if (IsDigit(ch)) {
			nRepeat = 0;
			eatNextFlag = false;
			stat = STAT_Repeat;
		} else if (ch == '{') {
			encoding.clear();
			stat = STAT_Encoding;
		} else if (ch == '@') {
			bigEndianFlag = IsBigEndian();
		} else if (ch == '=') {
			bigEndianFlag = IsBigEndian();
		} else if (ch == '<') {
			bigEndianFlag = false;
		} else if (ch == '>') {
			bigEndianFlag = true;
		} else if (ch == '!') {
			bigEndianFlag = true;
		} else if (ch == 'x') {
			if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			nRepeat = 1;
		} else if (ch == 'c') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			char str[2];
			str[1] = '\0';
			for (int i = 0; i < nRepeat; i++, pByte++) {
				str[0] = *pByte;
				valList.push_back(Value(env, str));
			}
			nRepeat = 1;
		} else if (ch == 'b') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte++) {
				char num = *pByte;
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'B') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte++) {
				unsigned char num = *pByte;
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'h') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 2 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 2) {
				short num = static_cast<short>(UnpackUShort(pByte, bigEndianFlag));
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'H') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 2 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 2) {
				unsigned short num = UnpackUShort(pByte, bigEndianFlag);
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'i') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 4) {
				int num = static_cast<int>(UnpackULong(pByte, bigEndianFlag));
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'I') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 4) {
				unsigned int num = static_cast<unsigned int>(UnpackULong(pByte, bigEndianFlag));
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'l') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 4) {
				long num = static_cast<long>(UnpackULong(pByte, bigEndianFlag));
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'L') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			for (int i = 0; i < nRepeat; i++, pByte += 4) {
				unsigned long num = static_cast<unsigned long>(UnpackULong(pByte, bigEndianFlag));
				valList.push_back(Value(num));
			}
			nRepeat = 1;
		} else if (ch == 'q') {
			//Bytes::iterator pByte = _bytes.begin() + offset;
			//if (!UnpackForward(sig, offset, 8 * nRepeat, exceedErrorFlag)) return Value::Null;
			//for (int i = 0; i < nRepeat; i++, pByte += 8) {
			//}
			//nRepeat = 1;
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return Value::Null;
		} else if (ch == 'Q') {
			//Bytes::iterator pByte = _bytes.begin() + offset;
			//if (!UnpackForward(sig, offset, 8 * nRepeat, exceedErrorFlag)) return Value::Null;
			//for (int i = 0; i < nRepeat; i++, pByte += 8) {
			//}
			//nRepeat = 1;
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return Value::Null;
		} else if (ch == 'f') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			char buff[4];
			for (int i = 0; i < nRepeat; i++) {
				for (int j = 0; j < 4; j++, pByte++) buff[j] = *pByte;
				float num = *(reinterpret_cast<float *>(buff));
				valList.push_back(Value(static_cast<Number>(num)));
			}
			nRepeat = 1;
		} else if (ch == 'd') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, 8 * nRepeat, exceedErrorFlag)) return Value::Null;
			char buff[8];
			for (int i = 0; i < nRepeat; i++) {
				for (int j = 0; j < 8; j++, pByte++) buff[j] = *pByte;
				double num = *(reinterpret_cast<double *>(buff));
				valList.push_back(Value(static_cast<Number>(num)));
			}
			nRepeat = 1;
		} else if (ch == 's') {
			Bytes::iterator pByte = _bytes.begin() + offset;
			if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			String str;
			//str.reserve(nRepeat);
			char chConv;
			for (int nUnpacked = 0; nUnpacked < nRepeat; nUnpacked++, pByte++) {
				Codec::Result result = pCodec->FeedChar(*pByte, chConv);
				if (result == Codec::RESULT_Error) {
					sig.SetError(ERR_CodecError,
						"decoding error. specify a proper coding name by {coding}");
					return false;
				} else if (result == Codec::RESULT_Complete) {
					str.push_back(chConv);
					while (pCodec->FollowChar(chConv)) str.push_back(chConv);
				}
			}
			// flush unprocessed characters
			if (pCodec->Flush(chConv)) while (pCodec->FollowChar(chConv)) ;
			valList.push_back(Value(env, str.c_str()));
			nRepeat = 1;
		} else if (ch == 'p') {
			//Bytes::iterator pByte = _bytes.begin() + offset;
			//if (!UnpackForward(sig, offset, nRepeat, exceedErrorFlag)) return Value::Null;
			//for (int i = 0; i < nRepeat; i++, pByte++) {
			//}
			//nRepeat = 1;
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return Value::Null;
		} else if (ch == 'P') {
			//Bytes::iterator pByte = _bytes.begin() + offset;
			//if (!UnpackForward(sig, offset, 4 * nRepeat, exceedErrorFlag)) return Value::Null;
			//for (int i = 0; i < nRepeat; i++, pByte += 4) {
			//}
			//nRepeat = 1;
			sig.SetError(ERR_ValueError, "sorry, not implemented yet");
			return Value::Null;
		} else if (IsWhite(ch)) {
			// just ignore white characters
		} else {
			sig.SetError(ERR_ValueError, "invalid character in format");
			return Value::Null;
		}
		if (eatNextFlag) p++;
	}
	return result;
}

bool Object_Bytes::PackForward(Signal sig, size_t offset, size_t size)
{
	if (offset + size > _bytes.size()) {
		_bytes.append(offset + size - _bytes.size(), '\0');
	}
	return true;
}

void Object_Bytes::PackUShort(Bytes::iterator pByte,
									bool bigEndianFlag, unsigned short num)
{
	unsigned char byte0 = static_cast<unsigned char>(num >> 8);
	unsigned char byte1 = static_cast<unsigned char>(num >> 0);
	if (bigEndianFlag) {
		*(pByte + 0) = byte0;
		*(pByte + 1) = byte1;
	} else {
		*(pByte + 0) = byte1;
		*(pByte + 1) = byte0;
	}
}

void Object_Bytes::PackULong(Bytes::iterator pByte,
									bool bigEndianFlag, unsigned long num)
{
	unsigned char byte0 = static_cast<unsigned char>(num >> 24);
	unsigned char byte1 = static_cast<unsigned char>(num >> 16);
	unsigned char byte2 = static_cast<unsigned char>(num >> 8);
	unsigned char byte3 = static_cast<unsigned char>(num >> 0);
	if (bigEndianFlag) {
		*(pByte + 0) = byte0;
		*(pByte + 1) = byte1;
		*(pByte + 2) = byte2;
		*(pByte + 3) = byte3;
	} else {
		*(pByte + 0) = byte3;
		*(pByte + 1) = byte2;
		*(pByte + 2) = byte1;
		*(pByte + 3) = byte0;
	}
}

bool Object_Bytes::UnpackForward(Signal sig,
							size_t &offset, int distance, bool exceedErrorFlag)
{
	if ((distance < 0 && offset < static_cast<size_t>(-distance)) ||
					(distance > 0 && offset + distance > _bytes.size())) {
		if (exceedErrorFlag) {
			sig.SetError(ERR_IndexError, "pointer exceeds the range of binary");
		}
		return false;
	}
	offset += distance;
	return true;
}

unsigned short Object_Bytes::UnpackUShort(Bytes::iterator pByte, bool bigEndianFlag)
{
	unsigned char byte0 = static_cast<unsigned char>(*(pByte + 0));
	unsigned char byte1 = static_cast<unsigned char>(*(pByte + 1));
	if (bigEndianFlag) {
		return (static_cast<unsigned short>(byte0) << 8) + byte1;
	} else {
		return (static_cast<unsigned short>(byte1) << 8) + byte0;
	}
}

unsigned long Object_Bytes::UnpackULong(Bytes::iterator pByte, bool bigEndianFlag)
{
	unsigned char byte0 = static_cast<unsigned char>(*(pByte + 0));
	unsigned char byte1 = static_cast<unsigned char>(*(pByte + 1));
	unsigned char byte2 = static_cast<unsigned char>(*(pByte + 2));
	unsigned char byte3 = static_cast<unsigned char>(*(pByte + 3));
	if (bigEndianFlag) {
		return (static_cast<unsigned long>(byte0) << 24) +
				(static_cast<unsigned long>(byte1) << 16) +
				(static_cast<unsigned long>(byte2) << 8) +
				byte3;
	} else {
		return (static_cast<unsigned long>(byte3) << 24) +
				(static_cast<unsigned long>(byte2) << 16) +
				(static_cast<unsigned long>(byte1) << 8) +
				byte0;
	}
}

bool Object_Bytes::CheckString(Signal sig,
					const ValueList &valList, ValueList::const_iterator pValue)
{
	if (pValue == valList.end()) {
		sig.SetError(ERR_ValueError, "not enough arguments");
		return false;
	} else if (!pValue->IsString()) {
		sig.SetError(ERR_ValueError, "string value is expected");
		return false;
	}
	return true;
}

bool Object_Bytes::CheckNumber(Signal sig,
					const ValueList &valList, ValueList::const_iterator pValue)
{
	if (pValue == valList.end()) {
		sig.SetError(ERR_ValueError, "not enough arguments");
		return false;
	} else if (!pValue->IsNumber()) {
		sig.SetError(ERR_ValueError, "number value is expected");
		return false;
	}
	return true;
}

bool Object_Bytes::CheckNumber(Signal sig, const ValueList &valList,
					ValueList::const_iterator pValue, Number numMin, Number numMax)
{
	if (!CheckNumber(sig, valList, pValue)) return false;
	Number num = pValue->GetNumber();
	if (num < numMin || num > numMax) {
		sig.SetError(ERR_ValueError, "number is out of range");
		return false;
	}
	return true;
}

//-----------------------------------------------------------------------------
// Object_Bytes::IteratorByte
//-----------------------------------------------------------------------------
Object_Bytes::IteratorByte::IteratorByte(Object_Bytes *pObj, int cntMax) :
			Iterator(false), _pObj(pObj), _cnt(cntMax), _cntMax(cntMax),
			_pCur(pObj->GetBytes().begin())
{
}

Object_Bytes::IteratorByte::~IteratorByte()
{
	Object::Delete(_pObj);
}

bool Object_Bytes::IteratorByte::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObj;
	const Bytes &bytes = _pObj->GetBytes();
	if (_pCur == bytes.end() || _cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	value = Value(static_cast<unsigned char>(*_pCur));
	_pCur++;
	return true;
}

String Object_Bytes::IteratorByte::ToString(Signal sig) const
{
	return String("<iterator:bytes>");
}

//-----------------------------------------------------------------------------
// Object_Bytes::IteratorUnpack
//-----------------------------------------------------------------------------
Object_Bytes::IteratorUnpack::IteratorUnpack(Object_Bytes *pObj,
						const char *format, size_t offset, int cntMax) :
		Iterator(false), _pObj(pObj), _format(format),
		_offset(offset), _offsetInit(offset),
		_cntMax(cntMax), _cnt(cntMax)
{
}

Object_Bytes::IteratorUnpack::~IteratorUnpack()
{
	Object::Delete(_pObj);
}

bool Object_Bytes::IteratorUnpack::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObj;
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	value = _pObj->Unpack(sig, _offset, _format.c_str(), false);
	return value.IsValid();
}

String Object_Bytes::IteratorUnpack::ToString(Signal sig) const
{
	return String("<iterator:unpacks>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Bytes
//-----------------------------------------------------------------------------
// bytes#len()
AScript_DeclareMethod(Bytes, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Bytes, len)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	return Value(pSelf->GetBytes().size());
}

// bytes#each() {block?}
AScript_DeclareMethod(Bytes, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Bytes, each)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	Object_Bytes *pObj = dynamic_cast<Object_Bytes *>(pSelf->IncRef());
	Iterator *pIterator = new Object_Bytes::IteratorByte(pObj, -1);
	return ReturnIterator(env, sig, context, pIterator);
}

// bytes#pointer(offset?:number)
AScript_DeclareMethod(Bytes, pointer)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "offset", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Bytes, pointer)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	Object_Bytes *pObj = dynamic_cast<Object_Bytes *>(pSelf->IncRef());
	size_t offset = context.IsNumber(0)? context.GetSizeT(0) : 0;
	Value result;
	result.InitAsBytesPtr(env, pObj, offset);
	return result;
}

// bytes#unpack(format:string, offset?:number)
AScript_DeclareMethod(Bytes, unpack)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "offset", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Bytes, unpack)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	size_t offset = context.IsNumber(1)? context.GetSizeT(1) : 0;
	return pSelf->Unpack(sig, offset, context.GetString(0), true);
}

// bytes#unpacks(format:string, offset?:number, cnt?:number)
AScript_DeclareMethod(Bytes, unpacks)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "offset", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareArg(env, "cnt", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Bytes, unpacks)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	Object_Bytes *pObj = dynamic_cast<Object_Bytes *>(pSelf->IncRef());
	const char *format = context.GetString(0);
	size_t offset = context.IsNumber(1)? context.GetSizeT(1) : 0;
	int cntMax = context.IsNumber(2)? context.GetInt(2) : -1;
	Iterator *pIterator =
			new Object_Bytes::IteratorUnpack(pObj, format, offset, cntMax);
	return ReturnIterator(env, sig, context, pIterator);
}

// bytes#dump():void
AScript_DeclareMethod(Bytes, dump)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Bytes, dump)
{
	File *pFile = env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL || !pFile->IsWritable()) return Value::Null;
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	int iCol = 0;
	String buffLine;
	foreach_const (Bytes, p, pSelf->GetBytes()) {
		char buff[8];
		::sprintf(buff, "%02x", static_cast<unsigned char>(*p));
		if (iCol > 0) buffLine += ' ';
		buffLine += buff;
		iCol++;
		if (iCol == 16) {
			pFile->PutString(buffLine.c_str());
			pFile->PutString("\n");
			buffLine.clear();
			iCol = 0;
		}
	}
	if (iCol > 0) {
		pFile->PutString(buffLine.c_str());
		pFile->PutString("\n");
	}
	pFile->Flush();
	return Value::Null;
}
// bytes#add!(buff+:bytes)
AScript_DeclareMethod(Bytes, add_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "buff", VTYPE_Bytes, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(Bytes, add_X)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pSelf->GetBytes() += pValue->GetBytes();
	}
	return Value::Null;
}

// bytes#store!(offset:number, buff+:bytes)
AScript_DeclareMethod(Bytes, store_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "buff", VTYPE_Bytes, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(Bytes, store_X)
{
	Object_Bytes *pSelf = Object_Bytes::GetSelfObj(context);
	size_t offset = context.GetSizeT(0);
	Bytes &bytes = pSelf->GetBytes();
	if (offset > bytes.size()) {
		bytes.append(offset - bytes.size(), '\0');
	}
	foreach_const (ValueList, pValue, context.GetList(1)) {
		size_t sizeEach = pValue->GetBytes().size();
		if (offset >= bytes.size()) {
			bytes += pValue->GetBytes();
		} else {
			bytes.replace(offset, sizeEach, pValue->GetBytes());
		}
		offset += sizeEach;
	}
	return Value::Null;
}

// Assignment
Class_Bytes::Class_Bytes(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(Bytes, len);
	AScript_AssignMethod(Bytes, each);
	AScript_AssignMethod(Bytes, pointer);
	AScript_AssignMethod(Bytes, unpack);
	AScript_AssignMethod(Bytes, unpacks);
	AScript_AssignMethod(Bytes, dump);
	AScript_AssignMethodEx(Bytes, add_X, "add!");
	AScript_AssignMethodEx(Bytes, store_X, "store!");
}

Object *Class_Bytes::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_Bytes((pClass == NULL)? this : pClass);
}

//-----------------------------------------------------------------------------
// Object_BytesPtr
//-----------------------------------------------------------------------------
Object_BytesPtr::~Object_BytesPtr()
{
	Object::Delete(_pObjBytes);
}

Object *Object_BytesPtr::Clone() const
{
	return new Object_BytesPtr(*this);
}

String Object_BytesPtr::ToString(Signal sig, bool exprFlag)
{
	char buff[64];
	::sprintf(buff, "<bytesptr:%d>", _offset);
	return String(buff);
}

bool Object_BytesPtr::UnpackForward(Signal sig, int distance, bool exceedErrorFlag)
{
	return _pObjBytes->UnpackForward(sig, _offset, distance, exceedErrorFlag);
}

Value Object_BytesPtr::Unpack(Signal sig,
					bool forwardFlag, const char *format, bool exceedErrorFlag)
{
	size_t offset = _offset;
	Value value = _pObjBytes->Unpack(sig, offset, format, exceedErrorFlag);
	if (forwardFlag) _offset = offset;
	return value;
}

bool Object_BytesPtr::Pack(Signal sig,
					bool forwardFlag, const char *format, const ValueList &valList)
{
	size_t offset = _offset;
	if (!_pObjBytes->Pack(sig, offset, format, valList)) return false;
	if (forwardFlag) _offset = offset;
	return true;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_BytesPtr
//-----------------------------------------------------------------------------
// bytesptr#reset()
AScript_DeclareMethod(BytesPtr, reset)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(BytesPtr, reset)
{
	Object_BytesPtr *pSelf = Object_BytesPtr::GetSelfObj(context);
	pSelf->Reset();
	return Value::Null;
}

// bytesptr#pack(format:string, value+):[stay]
AScript_DeclareMethod(BytesPtr, pack)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_OnceOrMore);
	DeclareAttr(AScript_Symbol(stay));
}

AScript_ImplementMethod(BytesPtr, pack)
{
	Object_BytesPtr *pSelf = Object_BytesPtr::GetSelfObj(context);
	bool forwardFlag = !context.IsSet(AScript_Symbol(stay));
	pSelf->Pack(sig, forwardFlag, context.GetString(0), context.GetList(1));
	return context.GetSelf();
}

// bytesptr#unpack(format:string):[stay]
AScript_DeclareMethod(BytesPtr, unpack)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareAttr(AScript_Symbol(stay));
}

AScript_ImplementMethod(BytesPtr, unpack)
{
	Object_BytesPtr *pSelf = Object_BytesPtr::GetSelfObj(context);
	bool forwardFlag = !context.IsSet(AScript_Symbol(stay));
	return pSelf->Unpack(sig, forwardFlag, context.GetString(0), false);
}

// bytesptr#unpacks(format:string, cnt?:number)
AScript_DeclareMethod(BytesPtr, unpacks)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "cnt", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(BytesPtr, unpacks)
{
	Object_BytesPtr *pSelf = Object_BytesPtr::GetSelfObj(context);
	Object_Bytes *pObj = dynamic_cast<Object_Bytes *>(pSelf->GetBytesObj()->IncRef());
	const char *format = context.GetString(0);
	int cntMax = context.IsNumber(1)? context.GetInt(1) : -1;
	Iterator *pIterator =
		new Object_Bytes::IteratorUnpack(pObj, format, pSelf->GetOffset(), cntMax);
	return ReturnIterator(env, sig, context, pIterator);
}

// bytesptr#forward(distance:number)
AScript_DeclareMethod(BytesPtr, forward)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "distance", VTYPE_Number);
}

AScript_ImplementMethod(BytesPtr, forward)
{
	Object_BytesPtr *pSelf = Object_BytesPtr::GetSelfObj(context);
	bool exeedErrorFlag = true;
	pSelf->UnpackForward(sig, context.GetInt(0), exeedErrorFlag);
	return context.GetSelf();
}

// Assignment
Class_BytesPtr::Class_BytesPtr(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(BytesPtr, reset);
	AScript_AssignMethod(BytesPtr, pack);
	AScript_AssignMethod(BytesPtr, unpack);
	AScript_AssignMethod(BytesPtr, unpacks);
	AScript_AssignMethod(BytesPtr, forward);
}

Object *Class_BytesPtr::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return NULL;
}

}
