//
// Object_Matrix
//

#include "Object_Matrix.h"
#include "Expr.h"
#include "Algorithm.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Matrix
//-----------------------------------------------------------------------------
Object_Matrix::Object_Matrix(Class *pClass, size_t nRows, size_t nCols) :
	Object(pClass), _pObjList(new Object_List(pClass->LookupClass(VTYPE_List))),
	_iRowOff(0), _iColOff(0), _nRows(nRows), _nCols(nCols), _nFold(nCols)
{
	ValueList &valListMat = _pObjList->GetList();
	valListMat.reserve(_nRows * _nCols);
}

Object_Matrix::Object_Matrix(Class *pClass,
						size_t nRows, size_t nCols, const Value &valueElem) :
	Object(pClass),
	_pObjList(new Object_List(pClass->LookupClass(VTYPE_List), nRows * nCols, valueElem)),
	_iRowOff(0), _iColOff(0), _nRows(nRows), _nCols(nCols), _nFold(nCols)
{
}

Object_Matrix::~Object_Matrix()
{
	Object::Delete(_pObjList);
}

Object *Object_Matrix::Clone() const
{
	return new Object_Matrix(*this);
}

Value Object_Matrix::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (!valueIdx.IsNumber()) {
		sig.SetError(ERR_IndexError, "index must be a number for list");
		return Value::Null;
	}
	size_t idx = valueIdx.GetSizeT();
	if (RowSize() == 1) {
		if (idx >= ColSize()) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		return *GetPointer(0, idx);
	} else {
		if (idx >= RowSize()) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		if (ColSize() == 1) {
			return *GetPointer(idx, 0);
		} else {
			return GetSub(idx, 0, 1, ColSize());
		}
	}
}

void Object_Matrix::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (!valueIdx.IsNumber()) {
		sig.SetError(ERR_IndexError, "index must be a number for list");
		return;
	}
	size_t idx = valueIdx.GetSizeT();
	if (RowSize() == 1) {
		if (idx >= ColSize()) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return;
		}
		*GetPointer(0, idx) = value;
	} else {
		if (idx >= RowSize()) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return;
		}
		if (ColSize() == 1) {
			*GetPointer(idx, 0) = value;
		} else if (value.IsMatrix()) {
			
			// todo
			
		} else {
			Iterator *pIterator = (value.IsList() || value.IsIterator())?
					value.CreateIterator(sig) : new Iterator_Constant(value);
			if (sig.IsSignalled()) return;
			SetRow(sig, idx, pIterator);
			Iterator::Delete(pIterator);
		}
	}
}

String Object_Matrix::ToString(Signal sig, bool exprFlag)
{
	String rtn;
	if (_nRows == 1 && _nCols == 1) {
		ValueList::const_iterator pValueElem = GetPointer(0, 0);
		rtn += pValueElem->ToString(sig, exprFlag);
	} else if (_nCols == 1) {
		rtn += "@@{";
		ValueList::const_iterator pValueElem = GetPointer(0, 0);
		for (size_t iRow = 0; iRow < _nRows; iRow++, pValueElem += FoldSize()) {
			if (iRow > 0) rtn += ", ";
			rtn += pValueElem->ToString(sig, exprFlag);
		}
		rtn += "}";
	} else {
		rtn += "@@{";
		for (size_t iRow = 0; iRow < _nRows; iRow++) {
			if (iRow > 0) rtn += ", ";
			rtn += "{";
			ValueList::const_iterator pValueElem = GetPointer(iRow, 0);
			for (size_t iCol = 0; iCol < _nCols; iCol++, pValueElem++) {
				if (iCol > 0) rtn += ", ";
				rtn += pValueElem->ToString(sig, exprFlag);
			}
			rtn += "}";
		}
		rtn += "}";
	}
	return rtn;
}

void Object_Matrix::ToList(ValueList &valList, bool transposeFlag)
{
	Environment &env = *this;
	if (_nRows == 1) {
		valList.reserve(_nCols);
		ValueList::const_iterator pValueElem = GetPointer(0, 0);
		for (size_t iCol = 0; iCol < _nCols; iCol++, pValueElem++) {
			valList.push_back(*pValueElem);
		}
	} else if (_nCols == 1) {
		valList.reserve(_nRows);
		ValueList::const_iterator pValueElem = GetPointer(0, 0);
		for (size_t iRow = 0; iRow < _nRows; iRow++, pValueElem += FoldSize()) {
			valList.push_back(*pValueElem);
		}
	} else if (transposeFlag) {
		valList.reserve(_nCols);
		for (size_t iCol = 0; iCol < _nCols; iCol++) {
			ValueList::const_iterator pValueElem = GetPointer(0, iCol);
			Value valueCol;
			ValueList &valListCol = valueCol.InitAsList(env);
			valList.push_back(valueCol);
			for (size_t iRow = 0; iRow < _nRows; iRow++, pValueElem += FoldSize()) {
				valListCol.push_back(*pValueElem);
			}
		}
	} else {
		valList.reserve(_nRows);
		for (size_t iRow = 0; iRow < _nRows; iRow++) {
			ValueList::const_iterator pValueElem = GetPointer(iRow, 0);
			Value valueRow;
			ValueList &valListRow = valueRow.InitAsList(env);
			valList.push_back(valueRow);
			for (size_t iCol = 0; iCol < _nCols; iCol++, pValueElem++) {
				valListRow.push_back(*pValueElem);
			}
		}
	}
}

Value Object_Matrix::GetSub(size_t iRow, size_t iCol, size_t nRows, size_t nCols)
{
	Environment &env = *this;
	Value result;
	Object_Matrix *pObj = new Object_Matrix(env.LookupClass(VTYPE_Matrix),
				dynamic_cast<Object_List *>(_pObjList->IncRef()),
				iRow, iCol, nRows, nCols, _nFold);
	result.InitAsObject(pObj, VTYPE_Matrix);
	return result;
}

bool Object_Matrix::Set(Signal sig, Iterator *pIterator)
{
	for (size_t iRow = 0; iRow < _nRows; iRow++) {
		ValueList::iterator pValueElem = GetPointer(iRow, 0);
		Value value;
		for (size_t iCol = 0; iCol < _nCols && pIterator->Next(sig, value);
														iCol++, pValueElem++) {
			*pValueElem = value;
		}
		if (sig.IsSignalled()) return false;
	}
	return true;
}

bool Object_Matrix::SetRow(Signal sig, size_t iRow, Iterator *pIterator)
{
	if (iRow >= _nRows) {
		sig.SetError(ERR_ValueError, "index exceeds the matrix size");
		return false;
	}
	Value value;
	ValueList::iterator pValueElem = GetPointer(iRow, 0);
	for (size_t iCol = 0; iCol < _nCols && pIterator->Next(sig, value);
												iCol++, pValueElem++) {
		*pValueElem = value;
	}
	return true;
}

bool Object_Matrix::SetCol(Signal sig, size_t iCol, Iterator *pIterator)
{
	if (iCol >= _nCols) {
		sig.SetError(ERR_ValueError, "index exceeds the matrix size");
		return false;
	}
	Value value;
	ValueList::iterator pValueElem = GetPointer(0, iCol);
	for (size_t iRow = 0; iRow < _nRows && pIterator->Next(sig, value);
												iRow++, pValueElem += _nFold) {
		*pValueElem = value;
	}
	return true;
}

Value Object_Matrix::Transpose(Environment &env, Signal sig)
{
	Value result;
	size_t nRows = RowSize(), nCols = ColSize();
	size_t nFold = FoldSize();
	Object_Matrix *pObjMatRtn = new Object_Matrix(
								env.LookupClass(VTYPE_Matrix), nCols, nRows);
	result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
	ValueList &valList = pObjMatRtn->GetList();
	for (size_t iCol = 0; iCol < nCols; iCol++) {
		ValueList::const_iterator pValueElem = GetPointer(0, iCol);
		for (size_t iRow = 0; iRow < nRows; iRow++, pValueElem += nFold) {
			valList.push_back(*pValueElem);
		}
	}
	return result;
}

Value Object_Matrix::Inverse(Environment &env, Signal sig)
{
	size_t nCols = ColSize(), nRows = RowSize();
	if (nCols != nRows) {
		sig.SetError(ERR_ValueError, "inversion can only be applied to square matrix");
		return Value::Null;
	}
	ValueType valType = CheckValueType();
	if (valType == VTYPE_Number) {
		NumberList mat;
		mat.reserve(nCols * 2 * nRows);
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			ValueList::const_iterator pValueElem = GetPointer(iRow, 0);
			for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
				mat.push_back(pValueElem->GetNumber());
			}
			for (size_t iCol = 0; iCol < nCols; iCol++) {
				mat.push_back((iCol == iRow)? 1. : 0.);
			}
		}
		Number det;
		if (!InvertMatrix(mat, nCols, det)) {
			sig.SetError(ERR_ValueError, "failed to calculate inverse matrix");
			return Value::Null;
		}
		Value result;
		Object_Matrix *pObjMatRtn = new Object_Matrix(
									env.LookupClass(VTYPE_Matrix), nCols, nRows);
		result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
		ValueList &valList = pObjMatRtn->GetList();
		NumberList::iterator pResult = mat.begin() + nCols;
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			for (size_t iCol = 0; iCol < nCols; iCol++, pResult++) {
				valList.push_back(*pResult);
			}
			pResult += nCols;
		}
		return result;
	} else if (valType == VTYPE_Complex) {
		ComplexList mat;
		mat.reserve(nCols * 2 * nRows);
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			ValueList::const_iterator pValueElem = GetPointer(iRow, 0);
			for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
				mat.push_back(pValueElem->GetComplex());
			}
			for (size_t iCol = 0; iCol < nCols; iCol++) {
				mat.push_back((iCol == iRow)? 1. : 0.);
			}
		}
		Complex det;
		if (!InvertMatrix(mat, nCols, det)) {
			sig.SetError(ERR_ValueError, "failed to calculate inverse matrix");
			return Value::Null;
		}
		Value result;
		Object_Matrix *pObjMatRtn = new Object_Matrix(
									env.LookupClass(VTYPE_Matrix), nCols, nRows);
		result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
		ValueList &valList = pObjMatRtn->GetList();
		ComplexList::iterator pResult = mat.begin() + nCols;
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			for (size_t iCol = 0; iCol < nCols; iCol++, pResult++) {
				valList.push_back(*pResult);
			}
			pResult += nCols;
		}
		return result;
	}
	sig.SetError(ERR_ValueError, "failed to calculate inverse matrix");
	return Value::Null;
}

bool Object_Matrix::GetElemIndex(Environment &env, Signal sig,
		const Expr *pExprIdx, size_t nElemsSrc, size_t &iElem, size_t &nElems)
{
	Value valueIdx = pExprIdx->Exec(env, sig);
	if (sig.IsSignalled()) return false;
	if (valueIdx.IsNumber()) {
		iElem = valueIdx.GetSizeT();
	} else if (valueIdx.IsIterator()) {
		Iterator *pIterator = valueIdx.GetIterator();
		if (pIterator->IsSequence()) {
			Iterator_Sequence *pIteratorEx =
							dynamic_cast<Iterator_Sequence *>(pIterator);
			iElem = static_cast<size_t>(pIteratorEx->GetBegin());
			nElems = static_cast<size_t>(
						pIteratorEx->GetEnd() - pIteratorEx->GetBegin()) + 1;
		} else if (pIterator->IsSequenceInf()) {
			Iterator_SequenceInf *pIteratorEx =
							dynamic_cast<Iterator_SequenceInf *>(pIterator);
			iElem = static_cast<size_t>(pIteratorEx->GetBegin());
			nElems = nElemsSrc - iElem;
		} else {
			sig.SetError(ERR_ValueError, "invalid index expression");
			return false;
		}
	} else {
		sig.SetError(ERR_ValueError, "invalid index expression");
		return false;
	}
	if (iElem >= nElemsSrc || iElem + nElems > nElemsSrc) {
		sig.SetError(ERR_ValueError, "index exceeds the matrix size");
		return false;
	}
	return true;
}

Value Object_Matrix::OperatorNeg(Environment &env, Signal sig,
											const Object_Matrix *pObjMat)
{
	Value result;
	size_t nRows = pObjMat->RowSize(), nCols = pObjMat->ColSize();
	Object_Matrix *pObjMatRtn = new Object_Matrix(
								env.LookupClass(VTYPE_Matrix), nRows, nCols);
	result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
	ValueList &valList = pObjMatRtn->GetList();
	const Function &func = env.GetFunc_Neg();
	for (size_t iRow = 0; iRow < nRows; iRow++) {
		ValueList::const_iterator pValueElem = pObjMat->GetPointer(iRow, 0);
		for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
			ValueList valListArg(*pValueElem);
			Context context(valListArg);
			Value resultElem = func.Eval(env, sig, context);
			if (sig.IsSignalled()) return Value::Null;
			valList.push_back(resultElem);
		}
	}
	return result;
}

void SetError_MatrixSizeMismatch(Signal sig)
{
	sig.SetError(ERR_ValueError, "matrix size mismatches");
}

Value Object_Matrix::OperatorPlusMinus(Environment &env, Signal sig, const Function &func,
				const Object_Matrix *pObjMat1, const Object_Matrix *pObjMat2)
{
	Value result;
	size_t nRows = pObjMat1->RowSize(), nCols = pObjMat1->ColSize();
	if (!(nRows == pObjMat2->RowSize() && nCols == pObjMat2->ColSize())) {
		SetError_MatrixSizeMismatch(sig);
		return Value::Null;
	}
	Object_Matrix *pObjMatRtn = new Object_Matrix(
								env.LookupClass(VTYPE_Matrix), nRows, nCols);
	result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
	ValueList &valList = pObjMatRtn->GetList();
	for (size_t iRow = 0; iRow < nRows; iRow++) {
		ValueList::const_iterator pValueElem1 = pObjMat1->GetPointer(iRow, 0);
		ValueList::const_iterator pValueElem2 = pObjMat2->GetPointer(iRow, 0);
		for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem1++, pValueElem2++) {
			ValueList valListArg(*pValueElem1, *pValueElem2);
			Context context(valListArg);
			Value resultElem = func.Eval(env, sig, context);
			if (sig.IsSignalled()) return Value::Null;
			valList.push_back(resultElem);
		}
	}
	return result;
}

Value Object_Matrix::OperatorMultiply(Environment &env, Signal sig,
				const Object_Matrix *pObjMat1, const Object_Matrix *pObjMat2)
{
	Value result;
	size_t nRows = pObjMat1->RowSize(), nCols = pObjMat2->ColSize();
	size_t nElems = pObjMat1->ColSize();
	size_t nFold = pObjMat2->FoldSize();
	if (nElems != pObjMat2->RowSize()) {
		SetError_MatrixSizeMismatch(sig);
		return Value::Null;
	}
	Object_Matrix *pObjMatRtn = new Object_Matrix(
								env.LookupClass(VTYPE_Matrix), nRows, nCols);
	result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
	ValueList &valList = pObjMatRtn->GetList();
	const Function &funcMultiply = env.GetFunc_Multiply();
	const Function &funcPlus = env.GetFunc_Plus();
	for (size_t iRow = 0; iRow < nRows; iRow++) {
		for (size_t iCol = 0; iCol < nCols; iCol++) {
			ValueList::const_iterator pValueElem1 = pObjMat1->GetPointer(iRow, 0);
			ValueList::const_iterator pValueElem2 = pObjMat2->GetPointer(0, iCol);
			Value valueAccum(0);
			for (size_t iElem = 0; iElem < nElems;
									iElem++, pValueElem1++, pValueElem2 += nFold) {
				Value valueElem;
				do {
					ValueList valListArg(*pValueElem1, *pValueElem2);
					Context context(valListArg);
					valueElem = funcMultiply.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
				do {
					ValueList valListArg(valueAccum, valueElem);
					Context context(valListArg);
					valueAccum = funcPlus.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
			}
			valList.push_back(valueAccum);
		}
	}
	return result;
}

Value Object_Matrix::OperatorMultiply(Environment &env, Signal sig,
							const Object_Matrix *pObjMat, const Value &value)
{
	Value result;
	size_t nRows = pObjMat->RowSize(), nCols = pObjMat->ColSize();
	if (value.IsList()) {
		const ValueList &valList = value.GetList();
		if (nCols != valList.size()) {
			SetError_MatrixSizeMismatch(sig);
			return Value::Null;
		}
		const Function &funcMultiply = env.GetFunc_Multiply();
		const Function &funcPlus = env.GetFunc_Plus();
		ValueList &valListResult = result.InitAsList(env);
		valListResult.reserve(nRows);
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			ValueList::const_iterator pValueElem = pObjMat->GetPointer(iRow, 0);
			Value valueAccum(0);
			foreach_const (ValueList, pValue, valList) {
				Value valueElem;
				do {
					ValueList valListArg(*pValueElem, *pValue);
					Context context(valListArg);
					valueElem = funcMultiply.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
				do {
					ValueList valListArg(valueAccum, valueElem);
					Context context(valListArg);
					valueAccum = funcPlus.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
				pValueElem++;
			}
			valListResult.push_back(valueAccum);
		}
		if (valListResult.size() == 1) return valListResult.front();
	} else {
		Object_Matrix *pObjMatRtn = new Object_Matrix(
									env.LookupClass(VTYPE_Matrix), nRows, nCols);
		result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
		ValueList &valListResult = pObjMatRtn->GetList();
		const Function &func = env.GetFunc_Multiply();
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			ValueList::const_iterator pValueElem = pObjMat->GetPointer(iRow, 0);
			for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
				ValueList valListArg(*pValueElem, value);
				Context context(valListArg);
				Value resultElem = func.Eval(env, sig, context);
				if (sig.IsSignalled()) return Value::Null;
				valListResult.push_back(resultElem);
			}
		}
		if (valListResult.size() == 1) return valListResult.front();
	}
	return result;
}

Value Object_Matrix::OperatorMultiply(Environment &env, Signal sig,
							const Value &value, const Object_Matrix *pObjMat)
{
	Value result;
	size_t nRows = pObjMat->RowSize(), nCols = pObjMat->ColSize();
	size_t nFold = pObjMat->FoldSize();
	if (value.IsList()) {
		const ValueList &valList = value.GetList();
		if (nRows != valList.size()) {
			SetError_MatrixSizeMismatch(sig);
			return Value::Null;
		}
		const Function &funcMultiply = env.GetFunc_Multiply();
		const Function &funcPlus = env.GetFunc_Plus();
		ValueList &valListResult = result.InitAsList(env);
		valListResult.reserve(nCols);
		for (size_t iCol = 0; iCol < nCols; iCol++) {
			ValueList::const_iterator pValueElem = pObjMat->GetPointer(0, iCol);
			Value valueAccum(0);
			foreach_const (ValueList, pValue, valList) {
				Value valueElem;
				do {
					ValueList valListArg(*pValueElem, *pValue);
					Context context(valListArg);
					valueElem = funcMultiply.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
				do {
					ValueList valListArg(valueAccum, valueElem);
					Context context(valListArg);
					valueAccum = funcPlus.Eval(env, sig, context);
					if (sig.IsSignalled()) return Value::Null;
				} while (0);
				pValueElem += nFold;
			}
			valListResult.push_back(valueAccum);
		}
		if (valListResult.size() == 1) return valListResult.front();
	} else {
		Object_Matrix *pObjMatRtn = new Object_Matrix(
									env.LookupClass(VTYPE_Matrix), nRows, nCols);
		result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
		ValueList &valListResult = pObjMatRtn->GetList();
		const Function &func = env.GetFunc_Multiply();
		for (size_t iRow = 0; iRow < nRows; iRow++) {
			ValueList::const_iterator pValueElem = pObjMat->GetPointer(iRow, 0);
			for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
				ValueList valListArg(value, *pValueElem);
				Context context(valListArg);
				Value resultElem = func.Eval(env, sig, context);
				if (sig.IsSignalled()) return Value::Null;
				valListResult.push_back(resultElem);
			}
		}
		if (valListResult.size() == 1) return valListResult.front();
	}
	return result;
}

Value Object_Matrix::OperatorDivide(Environment &env, Signal sig,
							const Object_Matrix *pObjMat, const Value &value)
{
	Value result;
	size_t nRows = pObjMat->RowSize(), nCols = pObjMat->ColSize();
	Object_Matrix *pObjMatRtn = new Object_Matrix(
								env.LookupClass(VTYPE_Matrix), nRows, nCols);
	result.InitAsObject(pObjMatRtn, VTYPE_Matrix);
	ValueList &valList = pObjMatRtn->GetList();
	const Function &func = env.GetFunc_Divide();
	for (size_t iRow = 0; iRow < nRows; iRow++) {
		ValueList::const_iterator pValueElem = pObjMat->GetPointer(iRow, 0);
		for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
			ValueList valListArg(*pValueElem, value);
			Context context(valListArg);
			Value resultElem = func.Eval(env, sig, context);
			if (sig.IsSignalled()) return Value::Null;
			valList.push_back(resultElem);
		}
	}
	return result;
}

ValueType Object_Matrix::CheckValueType() const
{
	ValueType valType = VTYPE_Invalid;
	size_t nCols = ColSize(), nRows = RowSize();
	for (size_t iRow = 0; iRow < nRows; iRow++) {
		ValueList::const_iterator pValueElem = GetPointer(iRow, 0);
		for (size_t iCol = 0; iCol < nCols; iCol++, pValueElem++) {
			if (pValueElem->IsNumber()) {
				if (valType == VTYPE_Invalid) valType = VTYPE_Number;
			} else if (pValueElem->IsComplex()) {
				valType = VTYPE_Complex;
			} else {
				return VTYPE_Invalid;
			}
		}
	}
	return valType;
}

//-----------------------------------------------------------------------------
// Object_Matrix::IteratorEach
//-----------------------------------------------------------------------------
Object_Matrix::IteratorEach::~IteratorEach()
{
	Object::Delete(_pObj);
}

bool Object_Matrix::IteratorEach::DoNext(Signal sig, Value &value)
{
	if (_transposeFlag) {
		if (_iCol >= _pObj->ColSize()) return false;
		value = *_pObj->GetPointer(_iRow, _iCol);
		_iRow++;
		if (_iRow >= _pObj->RowSize()) {
			_iRow = 0, _iCol++;
		}
	} else {
		if (_iRow >= _pObj->RowSize()) return false;
		value = *_pObj->GetPointer(_iRow, _iCol);
		_iCol++;
		if (_iCol >= _pObj->ColSize()) {
			_iCol = 0, _iRow++;
		}
	}
	return true;
}

String Object_Matrix::IteratorEach::ToString(Signal sig) const
{
	return String("<iterator:matrix#each>");
}

//-----------------------------------------------------------------------------
// Object_Matrix::IteratorEachRow
//-----------------------------------------------------------------------------
Object_Matrix::IteratorEachRow::~IteratorEachRow()
{
	Object::Delete(_pObj);
}

bool Object_Matrix::IteratorEachRow::DoNext(Signal sig, Value &value)
{
	if (_iRow >= _pObj->RowSize()) return false;
	value = _pObj->GetSub(_iRow, 0, 1, _pObj->ColSize());
	_iRow++;
	return true;
}

String Object_Matrix::IteratorEachRow::ToString(Signal sig) const
{
	return String("<iterator:matrix#eachrow>");
}

//-----------------------------------------------------------------------------
// Object_Matrix::IteratorEachCol
//-----------------------------------------------------------------------------
Object_Matrix::IteratorEachCol::~IteratorEachCol()
{
	Object::Delete(_pObj);
}

bool Object_Matrix::IteratorEachCol::DoNext(Signal sig, Value &value)
{
	if (_iCol >= _pObj->ColSize()) return false;
	value = _pObj->GetSub(0, _iCol, _pObj->RowSize(), 1);
	_iCol++;
	return true;
}

String Object_Matrix::IteratorEachCol::ToString(Signal sig) const
{
	return String("<iterator:matrix#eachcol>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Matrix
//-----------------------------------------------------------------------------
// matrix#rowsize()
AScript_DeclareMethod(Matrix, rowsize)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, rowsize)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return Value(pSelf->RowSize());
}

// matrix#colsize()
AScript_DeclareMethod(Matrix, colsize)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, colsize)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return Value(pSelf->ColSize());
}

// matrix#issquare()
AScript_DeclareMethod(Matrix, issquare)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, issquare)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return Value(pSelf->RowSize() == pSelf->ColSize());
}

// matrix#row(row:number):map
AScript_DeclareMethod(Matrix, row)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "row", VTYPE_Number);
}

AScript_ImplementMethod(Matrix, row)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return pSelf->GetSub(context.GetSizeT(0), 0, 1, pSelf->ColSize());
}

// matrix#col(col:number):map
AScript_DeclareMethod(Matrix, col)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "col", VTYPE_Number);
}

AScript_ImplementMethod(Matrix, col)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return pSelf->GetSub(0, context.GetSizeT(0), pSelf->RowSize(), 1);
}

// matrix#submat(row:number, col:number, nrows:number, ncols:number):map
AScript_DeclareMethod(Matrix, submat)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "row", VTYPE_Number);
	DeclareArg(env, "col", VTYPE_Number);
	DeclareArg(env, "nrows", VTYPE_Number);
	DeclareArg(env, "ncols", VTYPE_Number);
}

AScript_ImplementMethod(Matrix, submat)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	return pSelf->GetSub(context.GetSizeT(0), context.GetSizeT(1),
						context.GetSizeT(2), context.GetSizeT(3));
}

// matrix#each():[transpose]
AScript_DeclareMethod(Matrix, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(transpose));
}

AScript_ImplementMethod(Matrix, each)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Object_Matrix *pObj = dynamic_cast<Object_Matrix *>(pSelf->IncRef());
	Iterator *pIterator = new Object_Matrix::IteratorEach(pObj,
								context.IsSet(AScript_Symbol(transpose)));
	return ReturnIterator(env, sig, context, pIterator);
}

// matrix#eachrow()
AScript_DeclareMethod(Matrix, eachrow)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, eachrow)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Object_Matrix *pObj = dynamic_cast<Object_Matrix *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_Matrix::IteratorEachRow(pObj));
}

// matrix#eachcol()
AScript_DeclareMethod(Matrix, eachcol)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, eachcol)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Object_Matrix *pObj = dynamic_cast<Object_Matrix *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_Matrix::IteratorEachCol(pObj));
}

// matrix#tolist():[transpose]
AScript_DeclareMethod(Matrix, tolist)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(transpose));
}

AScript_ImplementMethod(Matrix, tolist)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Value result;
	ValueList &valList = result.InitAsList(env);
	pSelf->ToList(valList, context.IsSet(AScript_Symbol(transpose)));
	return result;
}

// matrix#set!(value)
AScript_DeclareMethodEx(Matrix, set_X, "set!")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
}

AScript_ImplementMethod(Matrix, set_X)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Iterator *pIterator = (context.IsList(0) || context.IsIterator(0))?
								context.GetValue(0).CreateIterator(sig) :
								new Iterator_Constant(context.GetValue(0));
	if (sig.IsSignalled()) return Value::Null;
	pSelf->Set(sig, pIterator);
	Iterator::Delete(pIterator);
	return Value::Null;
}

// matrix#setrow!(row:number, value)
AScript_DeclareMethodEx(Matrix, setrow_X, "setrow!")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "row", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Any);
}

AScript_ImplementMethod(Matrix, setrow_X)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Iterator *pIterator = (context.IsList(1) || context.IsIterator(1))?
								context.GetValue(1).CreateIterator(sig) :
								new Iterator_Constant(context.GetValue(1));
	if (sig.IsSignalled()) return Value::Null;
	pSelf->SetRow(sig, context.GetSizeT(0), pIterator);
	Iterator::Delete(pIterator);
	return Value::Null;
}

// matrix#setcol!(col:number, value)
AScript_DeclareMethodEx(Matrix, setcol_X, "setcol!")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "col", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Any);
}

AScript_ImplementMethod(Matrix, setcol_X)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	Iterator *pIterator = (context.IsList(1) || context.IsIterator(1))?
								context.GetValue(1).CreateIterator(sig) :
								new Iterator_Constant(context.GetValue(1));
	if (sig.IsSignalled()) return Value::Null;
	pSelf->SetCol(sig, context.GetSizeT(0), pIterator);
	Iterator::Delete(pIterator);
	return Value::Null;
}

// matrix#transpose()
AScript_DeclareMethod(Matrix, transpose)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, transpose)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	if (sig.IsSignalled()) return Value::Null;
	return pSelf->Transpose(env, sig);
}

// matrix#inverse()
AScript_DeclareMethod(Matrix, inverse)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Matrix, inverse)
{
	Object_Matrix *pSelf = Object_Matrix::GetSelfObj(context);
	if (sig.IsSignalled()) return Value::Null;
	return pSelf->Inverse(env, sig);
}

// Assignment
Class_Matrix::Class_Matrix(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(Matrix, rowsize);
	AScript_AssignMethod(Matrix, colsize);
	AScript_AssignMethod(Matrix, issquare);
	AScript_AssignMethod(Matrix, row);
	AScript_AssignMethod(Matrix, col);
	AScript_AssignMethod(Matrix, submat);
	AScript_AssignMethod(Matrix, each);
	AScript_AssignMethod(Matrix, eachrow);
	AScript_AssignMethod(Matrix, eachcol);
	AScript_AssignMethod(Matrix, tolist);
	AScript_AssignMethod(Matrix, set_X);
	AScript_AssignMethod(Matrix, setrow_X);
	AScript_AssignMethod(Matrix, setcol_X);
	AScript_AssignMethod(Matrix, transpose);
	AScript_AssignMethod(Matrix, inverse);
}

Object *Class_Matrix::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return NULL;
}

}
