#ifndef __STRING_H__
#define __STRING_H__

#include "Common.h"
#include "Signal.h"
#include "Codec.h"

namespace AScript {

class Symbol;
class SymbolSet;
class Value;
class ValueList;
class Environment;

//-----------------------------------------------------------------------------
// String operations
//-----------------------------------------------------------------------------
char GetEscaped(char ch);
const char *FindString(const char *str, const char *sub, bool ignoreCaseFlag);
String::const_iterator FindString(String::const_iterator str,
		String::const_iterator strEnd, const String &sub, bool ignoreCaseFlag);

String PickChar(const String &str, size_t idx);
String::const_iterator NextChar(const String &str, String::const_iterator p);
void EscapeString(String &strDst, const char *str);
String EscapeString(const char *str);
String NumberToString(Number num);
String Capitalize(const char *str);
String Lower(const char *str);
String Upper(const char *str);
String Strip(const char *str, const SymbolSet &attrs);
size_t Length(const char *str);
String Center(const char *str, size_t len, const char *padding);
String LJust(const char *str, size_t len, const char *padding);
String RJust(const char *str, size_t len, const char *padding);
String Left(const char *str, size_t len);
String Right(const char *str, size_t len);
String Middle(const char *str, int start, int len);
String Join(const ValueList &valList, const char *str);
String Replace(const char *str, const char *sub, const char *replace,
									int nMaxReplace, const SymbolSet &attrs);
void SplitPathList(Environment &env, const char *str, ValueList &valList);
Value FindString(Environment &env, Signal sig,
		const char *str, const char *sub, int start, const SymbolSet &attrs);

extern const unsigned short __ctypeTbl[];

#define CTYPE_Alpha (1 << 0)
inline bool IsAlpha(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_Alpha) != 0;
}

#define CTYPE_Digit (1 << 1)
inline bool IsDigit(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_Digit) != 0;
}

#define CTYPE_HexDigit (1 << 2)
inline bool IsHexDigit(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_HexDigit) != 0;
}

#define CTYPE_OctDigit (1 << 3)
inline bool IsOctDigit(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_OctDigit) != 0;
}

#define CTYPE_White (1 << 4)
inline bool IsWhite(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_White) != 0;
}

#define CTYPE_Space (1 << 5)
inline bool IsSpace(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_Space) != 0;
}

#define CTYPE_SJISFirst (1 << 6)
inline bool IsSJISFirst(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_SJISFirst) != 0;
}

#define CTYPE_SJISSecond (1 << 7)
inline bool IsSJISSecond(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_SJISSecond) != 0;
}

#define CTYPE_UTF8First (1 << 8)
inline bool IsUTF8First(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_UTF8First) != 0;
}

#define CTYPE_UTF8Follower (1 << 9)
inline bool IsUTF8Follower(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_UTF8Follower) != 0;
}

#define CTYPE_SymbolExtra (1 << 10)
inline bool IsSymbolExtra(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_SymbolExtra) != 0;
}

#define CTYPE_Lower (1 << 11)
inline bool IsLower(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_Lower) != 0;
}

#define CTYPE_Upper (1 << 12)
inline bool IsUpper(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] & CTYPE_Upper) != 0;
}

inline bool IsSymbolFirstChar(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] &
				(CTYPE_Alpha | CTYPE_SymbolExtra | CTYPE_UTF8First)) != 0;
}

inline bool IsSymbolChar(char ch) {
	return (__ctypeTbl[static_cast<unsigned char>(ch)] &
				(CTYPE_Alpha | CTYPE_SymbolExtra | CTYPE_Digit |
				 CTYPE_UTF8First | CTYPE_UTF8Follower)) != 0;
}

inline char ConvHexDigit(char ch) {
	return
		('0' <= ch && ch <= '9')? ch - '0' :
		('A' <= ch && ch <= 'F')? ch - 'A' + 10 :
		('a' <= ch && ch <= 'f')? ch - 'a' + 10 : 0;
}

inline char ConvOctDigit(char ch) {
	return ('0' <= ch && ch <= '7')? ch - '0' : 0;
}

inline char ToUpper(char ch) {
	return ('a' <= ch && ch <= 'z')? ch - 'a' + 'A' : ch;
}

inline char ToLower(char ch) {
	return ('A' <= ch && ch <= 'Z')? ch - 'A' + 'a' : ch;
}

inline int CompareChar(char ch1, char ch2, bool ignoreCaseFlag) {
	return ignoreCaseFlag?
		static_cast<int>(ToUpper(ch1)) - static_cast<int>(ToUpper(ch2)) :
		static_cast<int>(ch1) - static_cast<int>(ch2);
}

#if defined(HAVE_WINDOWS_H)
inline bool IsPathSeparator(char ch) { return ch == ';'; }
#else
inline bool IsPathSeparator(char ch) { return ch == ';' || ch == ':'; }
#endif

//-----------------------------------------------------------------------------
// StringList
//-----------------------------------------------------------------------------
class StringList : public std::vector<String> {
public:
	static const StringList Null;
	inline StringList() {}
private:
	inline StringList(const StringList &stringList) {}
	inline void operator=(const StringList &stringList) {}
};

}

#endif
