#ifndef __VALUE_H__
#define __VALUE_H__

#include "Common.h"
#include "Symbol.h"
#include "File.h"

namespace AScript {

class Expr;
class ObjectBase;
class Module;
class Class;
class Object;
class Object_List;
class Object_Matrix;
class Object_Dict;
class Object_File;
class Object_String;
class Object_Bytes;
class Object_BytesPtr;
class Object_Function;
class Environment;
class Function;

class ValueList;
class ValueDict;

class Iterator;

//-----------------------------------------------------------------------------
// ValueType
//-----------------------------------------------------------------------------
typedef int ValueType;

#define AScript_VTYPE(name) \
(ValueTypePool::GetInstance()->_pValueTypeInfo_##name->GetValueType())

#define AScript_VTYPEInfo(name) \
(ValueTypePool::GetInstance()->_pValueTypeInfo_##name)

#define AScript_DeclareVTYPE(name) \
ValueTypeInfo *_pValueTypeInfo_##name

#define AScript_RealizeVTYPE(name, str) \
env.AssignValueType(_pValueTypeInfo_##name = \
			ValueTypePool::GetInstance()->Add(Symbol::Add(str), NULL))

#define AScript_PrivClass(name) \
(_pPrivValueTypeInfo_##name->GetClass())

#define AScript_PrivVTYPE(name) \
(_pPrivValueTypeInfo_##name->GetValueType())

#define AScript_DeclarePrivClass(name) \
class Class_##name : public Class { \
public: \
	Class_##name(Environment &env); \
}; \
static ValueTypeInfo *_pPrivValueTypeInfo_##name = NULL

#define AScript_ImplementPrivClass(name) \
Class_##name::Class_##name(Environment &env) : \
			Class(env.LookupClass(VTYPE_Object), Symbol::Add(#name))

#define AScript_RealizePrivClass(name, str) \
env.AssignValueType(_pPrivValueTypeInfo_##name = \
			ValueTypePool::GetInstance()->Add(Symbol::Add(str), NULL)); \
_pPrivValueTypeInfo_##name->SetClass(new Class_##name(env))

#define VTYPE_Invalid		static_cast<ValueType>(0)
#define VTYPE_Quote			AScript_VTYPE(Quote)	// this type is used in declaration
#define VTYPE_Any			AScript_VTYPE(Any)		// this type is used in declaration
#define VTYPE_Symbol		AScript_VTYPE(Symbol)
#define VTYPE_Boolean		AScript_VTYPE(Boolean)
#define VTYPE_Number		AScript_VTYPE(Number)
#define VTYPE_Complex		AScript_VTYPE(Complex)
#define VTYPE_Module		AScript_VTYPE(Module)
#define VTYPE_Class			AScript_VTYPE(Class)
#define VTYPE_Object		AScript_VTYPE(Object)
#define VTYPE_Function		AScript_VTYPE(Function)
#define VTYPE_String		AScript_VTYPE(String)
#define VTYPE_StringEmb		AScript_VTYPE(StringEmb)
#define VTYPE_Bytes			AScript_VTYPE(Bytes)
#define VTYPE_BytesEmb		AScript_VTYPE(BytesEmb)
#define VTYPE_BytesPtr		AScript_VTYPE(BytesPtr)
#define VTYPE_List			AScript_VTYPE(List)
#define VTYPE_Matrix		AScript_VTYPE(Matrix)
#define VTYPE_Dict			AScript_VTYPE(Dict)
#define VTYPE_File			AScript_VTYPE(File)
#define VTYPE_FileStat		AScript_VTYPE(FileStat)
#define VTYPE_DateTime		AScript_VTYPE(DateTime)
#define VTYPE_Iterator		AScript_VTYPE(Iterator)
#define VTYPE_Expr			AScript_VTYPE(Expr)
#define VTYPE_Environment	AScript_VTYPE(Environment)
#define VTYPE_Error			AScript_VTYPE(Error)
#define VTYPE_Semaphore		AScript_VTYPE(Semaphore)
#define VTYPE_Struct		AScript_VTYPE(Struct)

//-----------------------------------------------------------------------------
// ValueTypeInfo
//-----------------------------------------------------------------------------
class ValueTypeInfo {
private:
	ValueType _valType;
	const Symbol *_pSymbol;
	Class *_pClass;
public:
	inline ValueTypeInfo(ValueType valType, const Symbol *pSymbol, Class *pClass) :
					_valType(valType), _pSymbol(pSymbol), _pClass(pClass) {}
	~ValueTypeInfo();
	inline static void Delete(ValueTypeInfo *pValTypeInfo) { delete pValTypeInfo; }
	inline const Symbol *GetSymbol() const { return _pSymbol; }
	inline ValueType GetValueType() const { return _valType; }
	inline Class *GetClass() const { return _pClass; }
	void SetClass(Class *pClass);
};

//-----------------------------------------------------------------------------
// ValueTypeMap
//-----------------------------------------------------------------------------
typedef std::map<const Symbol *, const ValueTypeInfo *,
									Symbol::KeyCompare_UniqNumber> ValueTypeMap;

//-----------------------------------------------------------------------------
// ValueTypeList
//-----------------------------------------------------------------------------
typedef std::vector<ValueTypeInfo *> ValueTypeList;

//-----------------------------------------------------------------------------
// ValueTypePool
//-----------------------------------------------------------------------------
class ValueTypePool {
public:
	AScript_DeclareVTYPE(Invalid);
	AScript_DeclareVTYPE(Quote);		// this type is used in declaration
	AScript_DeclareVTYPE(Any);			// this type is used in declaration
	AScript_DeclareVTYPE(Symbol);
	AScript_DeclareVTYPE(Boolean);
	AScript_DeclareVTYPE(Number);
	AScript_DeclareVTYPE(Complex);
	AScript_DeclareVTYPE(Module);
	AScript_DeclareVTYPE(Class);
	// object types
	AScript_DeclareVTYPE(Object);
	AScript_DeclareVTYPE(Function);
	AScript_DeclareVTYPE(String);
	AScript_DeclareVTYPE(StringEmb);
	AScript_DeclareVTYPE(Bytes);
	AScript_DeclareVTYPE(BytesEmb);
	AScript_DeclareVTYPE(BytesPtr);
	AScript_DeclareVTYPE(List);
	AScript_DeclareVTYPE(Matrix);
	AScript_DeclareVTYPE(Dict);
	AScript_DeclareVTYPE(File);
	AScript_DeclareVTYPE(FileStat);
	AScript_DeclareVTYPE(DateTime);
	AScript_DeclareVTYPE(Iterator);
	AScript_DeclareVTYPE(Expr);
	AScript_DeclareVTYPE(Environment);
	AScript_DeclareVTYPE(Error);
	AScript_DeclareVTYPE(Semaphore);
	AScript_DeclareVTYPE(Struct);
private:
	static ValueTypePool *_pInst;
	ValueTypeList _valueTypeList;
private:
	inline ValueTypePool() {}
public:
	~ValueTypePool();
	static void Initialize(Environment &env);
	inline static void SetInstance(ValueTypePool *pInst) { _pInst = pInst; }
	inline static ValueTypePool *GetInstance() { return _pInst; }
	inline ValueTypeList &GetList() { return _valueTypeList; }
	inline const ValueTypeList &GetList() const { return _valueTypeList; }
	ValueTypeInfo *Add(const Symbol *pSymbol, Class *pClass);
	ValueTypeInfo *Lookup(ValueType valType);
private:
	void _Initialize(Environment &env);
private:
	inline ValueTypePool(const ValueTypePool &valTypeInfoPool) {}
	inline void operator=(const ValueTypePool &valTypeInfoPool) {}
};

inline const Symbol *GetValueTypeSymbol(ValueType valType)
{
	return ValueTypePool::GetInstance()->Lookup(valType)->GetSymbol();
}

//-----------------------------------------------------------------------------
// ValueVisitor
//-----------------------------------------------------------------------------
class ValueVisitor {
public:
	virtual void Visit(Signal sig, const Value &value) = 0;
};

//-----------------------------------------------------------------------------
// Value
//-----------------------------------------------------------------------------
class Value {
public:
	struct KeyCompare {
		inline bool operator()(const Value &value1, const Value &value2) const {
			return Compare(value1, value2) < 0;
		}
	};
private:
	short _valType;
	char _ownFlag;
	union {
		Number num;				// VTYPE_Number
		bool flag;				// VTYPE_Boolean
		const Symbol *pSymbol;	// VTYPE_Symbol
		Complex *pComp;			// VTYPE_Complex
		Module *pModule;		// VTYPE_Module
		Class *pClass;			// VTYPE_Class
		Object *pObj;			// VTYPE_Object,Function,String,List,Dict,
								// VTYPE_File,FileStat,DateTime,Iterator, Expr
								// VTYPE_Environment,Error
	} _u;
private:
	static const Complex _compZero;
public:
	static const Value Null;
public:
	inline Value() : _valType(static_cast<short>(VTYPE_Invalid)), _ownFlag(true) {
	}
	inline Value(Number num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = num;
	}
	inline Value(int num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(unsigned int num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(char num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(unsigned char num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(short num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(unsigned short num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(long num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(unsigned long num) : _valType(static_cast<short>(VTYPE_Number)), _ownFlag(true) {
		_u.num = static_cast<Number>(num);
	}
	inline Value(bool flag) : _valType(static_cast<short>(VTYPE_Boolean)), _ownFlag(true) {
		_u.flag = flag;
	}
	inline Value(const Symbol *pSymbol) : _valType(static_cast<short>(VTYPE_Symbol)), _ownFlag(true) {
		_u.pSymbol = pSymbol;
	}
	Value(Environment &env, const char *str);
	Value(Environment &env, const char *str, size_t len);
	Value(Environment &env, ValueType valType, const char *str);
	Value(Environment &env, ValueType valType, const char *str, size_t len);
	Value(Environment &env, Iterator *pIterator);
	Value(Object *pObj, ValueType valType, bool ownFlag = true);
	inline Value(const Complex &comp) : _valType(static_cast<short>(VTYPE_Complex)), _ownFlag(true) {
		_u.pComp = new Complex(comp);
	}
	Value(const Value &value);
	~Value();
	Value &operator=(const Value &value);
	const char *GetTypeName() const;
	inline bool IsObject() const { return _valType >= VTYPE_Object; }
	//inline bool IsObject() const { return GetValueTypeClass(_valType) != NULL; }
	inline ValueType GetType() const { return static_cast<ValueType>(_valType); }
	inline bool IsType(ValueType valType) const {
		return _valType == static_cast<char>(valType);
	}
	inline bool IsInvalid() const			{ return IsType(VTYPE_Invalid);		}
	inline bool IsValid() const				{ return !IsType(VTYPE_Invalid);	}
	inline bool IsNumber() const			{ return IsType(VTYPE_Number);		}
	inline bool IsBoolean() const			{ return IsType(VTYPE_Boolean);		}
	inline bool IsSymbol() const			{ return IsType(VTYPE_Symbol);		}
	inline bool IsString() const			{ return IsType(VTYPE_String);		}
	inline bool IsStringEmb() const			{ return IsType(VTYPE_StringEmb);	}
	inline bool IsBytes() const				{ return IsType(VTYPE_Bytes);		}
	inline bool IsBytesEmb() const			{ return IsType(VTYPE_BytesEmb);	}
	inline bool IsBytesPtr() const			{ return IsType(VTYPE_BytesPtr);	}
	inline bool IsComplex() const			{ return IsType(VTYPE_Complex);		}
	inline bool IsModule() const			{ return IsType(VTYPE_Module);		}
	inline bool IsClass() const				{ return IsType(VTYPE_Class);		}
	inline bool IsGenericObject() const		{ return IsType(VTYPE_Object);		}
	inline bool IsFunction() const			{ return IsType(VTYPE_Function);	}
	inline bool IsList() const				{ return IsType(VTYPE_List);		}
	inline bool IsMatrix() const			{ return IsType(VTYPE_Matrix);		}
	inline bool IsDict() const				{ return IsType(VTYPE_Dict);		}
	inline bool IsFile() const				{ return IsType(VTYPE_File);		}
	inline bool IsFileStat() const			{ return IsType(VTYPE_FileStat);	}
	inline bool IsDateTime() const			{ return IsType(VTYPE_DateTime);	}
	inline bool IsIterator() const			{ return IsType(VTYPE_Iterator);	}
	inline bool IsExpr() const				{ return IsType(VTYPE_Expr);		}
	inline bool IsEnvironment() const		{ return IsType(VTYPE_Environment);	}
	inline bool IsError() const				{ return IsType(VTYPE_Error);		}
	inline bool IsSemaphore() const			{ return IsType(VTYPE_Semaphore);	}
	inline bool IsNumberOrComplex() const	{ return IsNumber() || IsComplex(); }
	inline void SetNumber(Number num) {
		FreeResource(); _valType = static_cast<char>(VTYPE_Number), _u.num = num;
	}
	inline void SetBoolean(bool flag) {
		FreeResource(); _valType = static_cast<char>(VTYPE_Boolean), _u.flag = flag;
	}
	inline void SetSymbol(const Symbol *pSymbol) {
		FreeResource(); _valType = static_cast<char>(VTYPE_Symbol), _u.pSymbol = pSymbol;
	}
	inline void SetComplex(const Complex &comp) {
		FreeResource();
		if (comp.imag() == 0.) {
			 _valType = static_cast<char>(VTYPE_Number), _u.num = comp.real();
		} else {
			 _valType = static_cast<char>(VTYPE_Complex), _u.pComp = new Complex(comp);
		}
	}
	inline Number GetNumber() const {
		return IsNumber()? _u.num :
				IsBoolean()? (_u.flag? 1. : 0.) :
				IsString()? ::strtod(GetString(), NULL) : 0.;
	}
	inline int GetInt() const { return static_cast<int>(GetNumber()); }
	inline unsigned int GetUInt() const { return static_cast<unsigned int>(GetNumber()); }
	inline size_t GetSizeT() const { return static_cast<size_t>(GetNumber()); }
	inline char GetChar() const { return static_cast<char>(GetNumber()); }
	inline unsigned char GetUChar() const { return static_cast<unsigned char>(GetNumber()); }
	inline short GetShort() const { return static_cast<short>(GetNumber()); }
	inline unsigned short GetUShort() const { return static_cast<unsigned short>(GetNumber()); }
	inline long GetLong() const { return static_cast<long>(GetNumber()); }
	inline unsigned long GetULong() const { return static_cast<unsigned long>(GetNumber()); }
	inline bool GetBoolean() const {
		return IsBoolean()? _u.flag : IsValid();
	}
	inline const Symbol *GetSymbol() const {
		return IsSymbol()? _u.pSymbol : NULL;
	}
	const char *GetString() const;
	const String &_GetString() const;
	const Bytes &GetBytes() const;
	Object_BytesPtr *GetBytesPtr() const;
	const DateTime &GetDateTime() const;
	ErrorType GetErrorType() const;
	Object_String *GetStringObj();
	const Object_String *GetStringObj() const;
	Object_File *GetFileObj();
	const Object_File *GetFileObj() const;
	inline Complex GetComplex() const {
		return IsComplex()? *_u.pComp : IsNumber()? Complex(GetNumber()) : _compZero;
	}
	bool IsFlatList() const;
	bool IsInstanceOf(ValueType valType);
	ObjectBase *ExtractObject(Signal sig);
	Object_List *GetListObj();
	const Object_List *GetListObj() const;
	Object_Matrix *GetMatrixObj();
	const Object_Matrix *GetMatrixObj() const;
	ValueList &GetList();
	const ValueList &GetList() const;
	ValueDict &GetDict();
	const ValueDict &GetDict() const;
	Iterator *GetIterator() const;
	File &GetFile();
	const Expr *GetExpr() const;
	Expr *CloneExpr() const;
	Function *GetFunction();
	Object_Function *GetFunctionObj();
	inline const Function *GetFunction() const {
		return const_cast<Value *>(this)->GetFunction();
	}
	inline const Object_Function *GetFunctionObj() const {
		return const_cast<Value *>(this)->GetFunctionObj();
	}
	inline Module *GetModule() { return IsModule()? _u.pModule : NULL; }
	inline const Module *GetModule() const { return IsModule()? _u.pModule : NULL; }
	inline Class *GetClass() { return IsClass()? _u.pClass : NULL; }
	inline const Class *GetClass() const { return IsClass()? _u.pClass : NULL; }
	inline Object *GetObject() { return IsObject()? _u.pObj : NULL; }
	inline const Object *GetObject() const { return IsObject()? _u.pObj : NULL; }
	Iterator *CreateIterator(Signal sig) const;
	String ToString(Signal sig, bool exprFlag = true) const;
	Number ToNumber(bool allowPartFlag, bool &successFlag) const;
	void Accept(Signal sig, ValueVisitor &visitor) const;
	void InitAsModule(Module *pModule);
	void InitAsClass(Class *pClass);
	void InitAsObject(Object *pObj, ValueType valType, bool ownFlag = true);
	ValueList &InitAsList(Environment &env);
	ValueList &InitAsList(Environment &env, size_t n, const Value &value);
	Object_Matrix *InitAsMatrix(Environment &env,
								int nRows, int nCols, const Value &valueElem);
	Object_Matrix *InitAsMatrix(Environment &env, Signal sig, const ValueList &valList);
	ValueDict &InitAsDict(Environment &env);
	Object_File *InitAsFile(Environment &env);
	Object_Bytes *InitAsBytes(Environment &env, ValueType valType = VTYPE_Bytes);
	Object_Bytes *InitAsBytes(Environment &env,
							const Bytes &bytes, ValueType valType = VTYPE_Bytes);
	Object_BytesPtr *InitAsBytesPtr(Environment &env,
							Object_Bytes *pObjBytes, size_t offset);
	void InitAsExpr(Environment &env, Expr *pExpr);
	void InitAsIterator(Environment &env, Iterator *pIterator);
	void InitAsError(Environment &env,
						ErrorType errType, const char *text, const Value &value);
	void InitAsFunction(Environment &env, Function *pFunc);
	void InitAsEnvironment(Environment &env);
public:
	static int Compare(const Value &value1, const Value &value2);
	inline bool operator<(const Value &value) const {
		return Compare(*this, value) < 0;
	}
private:
	void FreeResource();
};

//-----------------------------------------------------------------------------
// ValueList
//-----------------------------------------------------------------------------
class ValueList : public std::vector<Value> {
public:
	static const ValueList Null;
public:
	inline ValueList() {}
	inline ValueList(size_t n) : std::vector<Value>(n) {}
	inline ValueList(size_t n, const Value &value) : std::vector<Value>(n, value) {}
	inline ValueList(const Value &value) {
		reserve(1);
		push_back(value);
	}
	inline ValueList(const Value &value1, const Value &value2) {
		reserve(2);
		push_back(value1), push_back(value2);
	}
	inline ValueList(const Value &value1, const Value &value2, const Value &value3) {
		reserve(3);
		push_back(value1), push_back(value2), push_back(value3);
	}
	ValueList(const ValueList &valList);
	bool IsTypeMixed() const;
	bool IsFlat() const;
	bool IsContain(const Value &value) const;
	bool IsContainIterator() const;
	bool CheckMatrix(size_t *pnRow, size_t *pnCol) const;
	void Append(const ValueList &valList);
	void Print(Signal sig, int indentLevel = 0) const;
	static bool AssumeSameLength(Signal sig,
					const ValueList &valList1, const ValueList &valList2);
};

//-----------------------------------------------------------------------------
// ValuePtrList
//-----------------------------------------------------------------------------
typedef std::vector<const Value *> ValuePtrList;

//-----------------------------------------------------------------------------
// ValueMap
//-----------------------------------------------------------------------------
class ValueMap : public std::map<const Symbol *, Value, Symbol::KeyCompare_UniqNumber> {
public:
	static const ValueMap Null;
public:
	inline bool IsSet(const Symbol *pSymbol) const {
		return find(pSymbol) != const_cast<ValueMap *>(this)->end();
	}
	inline void Insert(const Symbol *pSymbol) {
		insert(value_type(pSymbol, Value::Null));
	}
	inline void Insert(const Symbol *pSymbol, const Value &value) {
		insert(value_type(pSymbol, value));
	}
};

//-----------------------------------------------------------------------------
// ValueDeque
//-----------------------------------------------------------------------------
typedef std::deque<Value> ValueDeque;

//-----------------------------------------------------------------------------
// ValueDict
//-----------------------------------------------------------------------------
class ValueDict : public std::map<Value, Value, Value::KeyCompare> {
public:
	static const ValueDict Null;
public:
	bool Store(Signal sig, const ValueList &valList);
	bool Store(Signal sig, const Value &valueIdx, const Value &value);
	inline static bool IsValidKey(const Value &value) {
		return value.IsNumber() || value.IsString() || value.IsSymbol();
	}
};

inline const char *GetNumberFormat() { return "%g"; }

}

#endif
