#include "Iterator.h"
#include "Object.h"
#include "Expr.h"
#include "OAL.h"
#include "Function.h"
#include "Object_Function.h"
#include "Object_Bytes.h"

namespace AScript {


//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
Iterator::~Iterator()
{
	delete _pShare;
}

bool Iterator::IsSequence() const { return false; }
bool Iterator::IsSequenceInf() const { return false; }

Iterator *Iterator::Clone()
{
	if (_pShare == NULL) _pShare = new Share();
	int id = _pShare->Register();
	return new Iterator_GenericClone(IncRef(), id);
}

bool Iterator::NextShared(int id, Signal sig, Value &value)
{
	if (_pShare == NULL) return DoNext(sig, value);
	if (!_pShare->Next(id, value)) {
		if (_pShare->IsDone()) return false;
		if (!DoNext(sig, value)) {
			_pShare->SetDone();
			return false;
		}
		_pShare->Store(id, value);
	}
	return true;
}

Value Iterator::ToList(Environment &env, Signal sig)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value result;
	ValueList &valList = result.InitAsList(env);
	Value value;
	while (Next(sig, value)) {
		valList.push_back(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &env, Signal sig, Context &context)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(env, context, result);
	while (Next(sig, value)) {
		resultListComposer.Store(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &env, Signal sig,
								Context &context, const Function *pFuncBlock)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	std::auto_ptr<Function::ResultListComposer> pResultListComposer;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		pResultListComposer.reset(
					new Function::ResultListComposer(env, context, result));
	}
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		if (pResultListComposer.get() == NULL) {
			result = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} else {
			Value resultElem = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			pResultListComposer->Store(resultElem);
		}
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Reduce(Environment &env, Signal sig,
									Value valueAccum, const Function *pFuncBlock)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value;
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, valueAccum);
		Context contextSub(valListArg);
		valueAccum = pFuncBlock->Eval(env, sig, contextSub);
		AScript_BlockSignalHandlerInLoop(sig, valueAccum, Value::Null)
	}
	if (sig.IsSignalled()) return Value::Null;
	return valueAccum;
}

Value Iterator::MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs)
{
	Value valueHit;
	if (!Next(sig, valueHit)) return Value::Null;
	Value result;
	if (attrs.IsSet(AScript_Symbol(index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(last_index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp >= 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(indices))) {
		ValueList &resultList = result.InitAsList(env);
		resultList.push_back(Value(static_cast<Number>(0)));
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				resultList.clear();
				resultList.push_back(Value(static_cast<Number>(idx)));
			} else if (cmp == 0) {
				resultList.push_back(Value(static_cast<Number>(idx)));
			}
		}
		if (sig.IsSignalled()) return Value::Null;
	} else {
		Value value;
		while (Next(sig, value)) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result = valueHit;
	}
	return result;
}

Value Iterator::Sum(Environment &env, Signal sig)
{
	Value value;
	if (!Next(sig, value)) return Value::Null;
	Value result = value;
	while (Next(sig, value)) {
		ValueList valListArg(result, value);
		Context contextSub(valListArg);
		result = env.GetFunc_Plus().Eval(env, sig, contextSub);
		if (sig.IsSignalled()) return Value::Null;
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Average(Environment &env, Signal sig)
{
	int cnt = 0;
	Value value;
	if (!Next(sig, value)) return Value::Null;
	Value result = value;
	for (cnt = 1; Next(sig, value); cnt++) {
		ValueList valListArg(result, value);
		Context contextSub(valListArg);
		result = env.GetFunc_Plus().Eval(env, sig, contextSub);
		if (sig.IsSignalled()) return Value::Null;
	}
	if (sig.IsSignalled()) return Value::Null;
	ValueList valListArg(result, Value(static_cast<Number>(cnt)));
	Context contextSub(valListArg);
	return env.GetFunc_Divide().Eval(env, sig, contextSub);
}

Value Iterator::Variance(Environment &env, Signal sig)
{
	Value valueAve = Clone()->Average(env, sig);
	if (valueAve.IsInvalid()) return Value::Null;
	size_t cnt;
	Value value, valueSum(0);
	for (cnt = 0; Next(sig, value); cnt++) {
		Value valueDiff;
		do {
			ValueList valListArg(value, valueAve);
			Context contextSub(valListArg);
			valueDiff = env.GetFunc_Minus().Eval(env, sig, contextSub);
			if (sig.IsSignalled()) return Value::Null;
		} while (0);
		Value valueDiff2;
		do {
			ValueList valListArg(valueDiff, valueDiff);
			Context contextSub(valListArg);
			valueDiff2 = env.GetFunc_Multiply().Eval(env, sig, contextSub);
			if (sig.IsSignalled()) return Value::Null;
		} while (0);
		do {
			ValueList valListArg(valueSum, valueDiff2);
			Context contextSub(valListArg);
			valueSum = env.GetFunc_Plus().Eval(env, sig, contextSub);
			if (sig.IsSignalled()) return Value::Null;
		} while (0);
	}
	Value result;
	do {
		ValueList valListArg(valueSum, Value(cnt));
		Context contextSub(valListArg);
		result = env.GetFunc_Divide().Eval(env, sig, contextSub);
		if (sig.IsSignalled()) return Value::Null;
	} while (0);
	return result;
}

Value Iterator::And(Environment &env, Signal sig)
{
	Value value;
	if (!Next(sig, value)) return Value::Null;
	if (!value.GetBoolean()) return Value(false);
	while (Next(sig, value)) {
		if (!value.GetBoolean()) return Value(false);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(true);
}

Value Iterator::Or(Environment &env, Signal sig)
{
	Value value;
	if (!Next(sig, value)) return Value::Null;
	if (value.GetBoolean()) return Value(true);
	while (Next(sig, value)) {
		if (value.GetBoolean()) return Value(true);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(false);
}

size_t Iterator::Count(Environment &env, Signal sig, const Value &criteria)
{
	size_t cnt = 0;
	if (criteria.IsFunction()) {
		const Function *pFunc = criteria.GetFunction();
		Value value;
		while (Next(sig, value)) {
			ValueList valListArg(value);
			Context context(valListArg);
			Value valueFlag = pFunc->Eval(env, sig, context);
			if (sig.IsSignalled()) return 0;
			if (valueFlag.GetBoolean()) cnt++;
		}
		if (sig.IsSignalled()) return 0;
	} else {
		Value value;
		while (Next(sig, value)) {
			if (Value::Compare(value, criteria) == 0) cnt++;
		}
	}
	return cnt;
}

Iterator *Iterator::Filter(Environment &env, Signal sig, const Value &criteria)
{
	if (criteria.IsFunction()) {
		Object_Function *pFuncObjCriteria =
			dynamic_cast<Object_Function *>(criteria.GetFunctionObj()->IncRef());
		return new Iterator_Filter(env, this, pFuncObjCriteria);
	} else if (criteria.IsList() || criteria.IsIterator()) {
		Iterator *pIteratorCriteria = criteria.CreateIterator(sig);
		if (sig.IsSignalled()) return NULL;
		return new Iterator_FilterEach(this, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for filter");
		return NULL;
	}
}

Iterator *Iterator::While(Environment &env, Signal sig, const Value &criteria)
{
	if (criteria.IsFunction()) {
		Object_Function *pFuncObjCriteria = 
			dynamic_cast<Object_Function *>(criteria.GetFunctionObj()->IncRef());
		return new Iterator_While(env, this, pFuncObjCriteria);
	} else if (criteria.IsList() || criteria.IsIterator()) {
		Iterator *pIteratorCriteria = criteria.CreateIterator(sig);
		if (sig.IsSignalled()) return NULL;
		return new Iterator_WhileEach(this, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for while");
		return NULL;
	}
}

bool Iterator::IsContain(Signal sig, const Value &value)
{
	Value valueToFind;
	while (Next(sig, valueToFind)) {
		if (Value::Compare(value, valueToFind) == 0) return true;
	}
	return false;
}

String Iterator::ToString(Signal sig) const
{
	return String("<iterator>");
}

//-----------------------------------------------------------------------------
// Iterator::Share
//-----------------------------------------------------------------------------
Iterator::Share::Share() : _doneFlag(false), _indexMin(0)
{
	_indexList.push_back(_indexMin);
}

int Iterator::Share::Register()
{
	int id = static_cast<int>(_indexList.size());
	_indexList.push_back(_indexMin);
	return id;
}

bool Iterator::Share::Next(int id, Value &value)
{
	size_t iElem = _indexList[id] - _indexMin;
	if (iElem >= _valueDeque.size()) return false;
	value = _valueDeque[iElem];
	_indexList[id] = _indexMin + iElem + 1;
	size_t indexMinCur = _indexMin;
	foreach (IndexList, pIndex, _indexList) {
		if (indexMinCur > *pIndex) indexMinCur = *pIndex;
	}
	for (size_t cnt = indexMinCur - _indexMin; cnt > 0; cnt--) {
		_valueDeque.pop_front();
	}
	_indexMin = indexMinCur;
	return true;
}

void Iterator::Share::Store(int id, const Value &value)
{
	_valueDeque.push_back(value);
	_indexList[id] = _indexMin + _valueDeque.size();
}

//-----------------------------------------------------------------------------
// Iterator_GenericClone
//-----------------------------------------------------------------------------
Iterator_GenericClone::~Iterator_GenericClone()
{
}

bool Iterator_GenericClone::DoNext(Signal sig, Value &value)
{
	return _pIterator->NextShared(_id, sig, value);
}

String Iterator_GenericClone::ToString(Signal sig) const
{
	return _pIterator->ToString(sig);
}

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
Iterator_Constant::~Iterator_Constant()
{
}

Iterator *Iterator_Constant::Clone()
{
	return new Iterator_Constant(*this);
}

bool Iterator_Constant::DoNext(Signal sig, Value &value)
{
	value = _value;
	return true;
}

String Iterator_Constant::ToString(Signal sig) const
{
	String rtn = "<iterator:constant:";
	rtn += _value.ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Fill
//-----------------------------------------------------------------------------
Iterator_Fill::~Iterator_Fill()
{
}

Iterator *Iterator_Fill::Clone()
{
	return new Iterator_Fill(*this);
}

bool Iterator_Fill::DoNext(Signal sig, Value &value)
{
	if (_idx >= _cnt) return false;
	if (_cnt > 0) _idx++;
	value = _value;
	return true;
}

String Iterator_Fill::ToString(Signal sig) const
{
	String rtn = "<iterator:fill(";
	rtn += NumberToString(_cnt);
	if (_value.IsValid()) {
		rtn += ", ";
		rtn += _value.ToString(sig, true);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Rand
//-----------------------------------------------------------------------------
Iterator_Rand::~Iterator_Rand()
{
}

bool Iterator_Rand::DoNext(Signal sig, Value &value)
{
	if (_idx >= _cnt) return false;
	if (_cnt > 0) _idx++;
	value = Value((_range > 0)?
		static_cast<Number>(static_cast<int>(::genrand_real2() * _range)) :
		static_cast<Number>(::genrand_real2()));
	return true;
}

String Iterator_Rand::ToString(Signal sig) const
{
	String rtn = "<iterator:rands(";
	rtn += NumberToString(_cnt);
	if (_range > 0) {
		rtn += ", ";
		rtn += NumberToString(_range);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Range
//-----------------------------------------------------------------------------
Iterator_Range::~Iterator_Range()
{
}

Iterator *Iterator_Range::Clone()
{
	return new Iterator_Range(*this);
}

bool Iterator_Range::DoNext(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num < _numEnd) : (_num > _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Range::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1 || _numStep == -1) {
		if (_numBegin == 0) {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		} else {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numBegin);
			rtn += ", ";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		}
	} else {
		rtn = "<iterator:range(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Sequence
//-----------------------------------------------------------------------------
bool Iterator_Sequence::IsSequence() const { return true; }
Iterator_Sequence::~Iterator_Sequence()
{
}

Iterator *Iterator_Sequence::Clone()
{
	return new Iterator_Sequence(*this);
}

bool Iterator_Sequence::DoNext(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num <= _numEnd) : (_num >= _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Sequence::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1. || _numStep == -1.) {
		rtn = "<iterator:";
		rtn += NumberToString(_numBegin);
		rtn += "..";
		rtn += NumberToString(_numEnd);
		rtn += ">";
	} else {
		rtn = "<iterator:seq(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SequenceInf
//-----------------------------------------------------------------------------
bool Iterator_SequenceInf::IsSequenceInf() const { return true; }
Iterator_SequenceInf::~Iterator_SequenceInf()
{
}

Iterator *Iterator_SequenceInf::Clone()
{
	return new Iterator_SequenceInf(*this);
}

bool Iterator_SequenceInf::DoNext(Signal sig, Value &value)
{
	value = Value(_num);
	_num += 1;
	return true;
}

String Iterator_SequenceInf::ToString(Signal sig) const
{
	String rtn;
	rtn = "<iterator:";
	rtn += NumberToString(_numBegin);
	rtn += "..>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Interval
//-----------------------------------------------------------------------------
Iterator_Interval::~Iterator_Interval()
{
}

Iterator *Iterator_Interval::Clone()
{
	return new Iterator_Interval(*this);
}

bool Iterator_Interval::DoNext(Signal sig, Value &value)
{
	Number num;
	if (!_DoNext(num)) return false;
	value = Value(num);
	return true;
}

String Iterator_Interval::ToString(Signal sig) const
{
	return String("<iterator:interval>");
}

//-----------------------------------------------------------------------------
// Iterator_Map
//-----------------------------------------------------------------------------
Iterator_Map::~Iterator_Map()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_Map::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	ValueList valList(value);
	Context context(valList);
	value = _pObjFunc->Eval(_env, sig, context);
	return true;
}

String Iterator_Map::ToString(Signal sig) const
{
	return String("<iterator:map>");
}

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
Iterator_Mapping::Iterator_Mapping(Environment &env, Signal sig,
		Function *pFunc, const Value &valueSelf, const ValueList &valListArg) :
	Iterator(false), _env(env), _pFunc(pFunc), _valueSelf(valueSelf)
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, valListArg);
}

Iterator_Mapping::~Iterator_Mapping()
{
	Function::Delete(_pFunc);
}

bool Iterator_Mapping::DoNext(Signal sig, Value &value)
{
	ValueList valList;
	if (!_iterOwner.Next(sig, valList)) return false;
	Context context(valList);
	context.SetSelf(_valueSelf);
	value = _pFunc->Eval(_env, sig, context);
	if (sig.IsSignalled()) return false;
	return true;
}

String Iterator_Mapping::ToString(Signal sig) const
{
	String str;
	str += "<iterator:mapping:";
	str += _pFunc->ToString();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Iterator_FuncBinder
//-----------------------------------------------------------------------------
Iterator_FuncBinder::Iterator_FuncBinder(Environment &env, Signal sig,
				Function *pFunc, const Value &valueSelf, Iterator *pIterator) :
		Iterator(false), _env(env), _pFunc(pFunc),
		_valueSelf(valueSelf), _pIterator(pIterator)
{
}

Iterator_FuncBinder::~Iterator_FuncBinder()
{
	Function::Delete(_pFunc);
	Iterator::Delete(_pIterator);
}

bool Iterator_FuncBinder::DoNext(Signal sig, Value &value)
{
	Value valueArg;
	if (!_pIterator->Next(sig, valueArg)) return false;
	if (valueArg.IsList()) {
		value = _pFunc->EvalFuncBinder(_env, sig, valueArg.GetList());
		if (sig.IsSignalled()) return false;
	} else {
		sig.SetError(ERR_TypeError, "invalid structure of arguments");
		return false;
	}
	return true;
}

String Iterator_FuncBinder::ToString(Signal sig) const
{
	String str;
	str += "<iterator:func_binder:";
	str += _pFunc->ToString();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
Iterator_MemberMapping::~Iterator_MemberMapping()
{
	Expr::Delete(_pExpr);
	Iterator::Delete(_pIterator);
}

bool Iterator_MemberMapping::DoNext(Signal sig, Value &value)
{
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
	if (sig.IsSignalled()) return false;
	Environment &env = *pObjEach;
	value = _pExpr->Exec(env, sig);
	if (value.IsFunction()) {
		Object *pObj = new Object_WrappedMethod(env.LookupClass(VTYPE_Function),
							value.GetFunction()->IncRef(), valueSelfEach);
		value = Value(pObj, VTYPE_Function, true);
	}
	return true;
}

String Iterator_MemberMapping::ToString(Signal sig) const
{
	return String("<iterator:member_mapping>");
}

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
Iterator_MethodMapping::~Iterator_MethodMapping()
{
	Expr::Delete(_pExprCaller);
	Iterator::Delete(_pIterator);
}

bool Iterator_MethodMapping::DoNext(Signal sig, Value &value)
{
	const Function *pFuncSuccRequester = NULL;
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	value = _pExprCaller->EvalEach(_env, sig, valueSelfEach, &pFuncSuccRequester);
	return true;
}

String Iterator_MethodMapping::ToString(Signal sig) const
{
	return String("<iterator:method_mapping>");
}

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
Iterator_Fork::Iterator_Fork(Environment &env, Signal sig,
							Function *pFunc, const ValueList &valListArg) :
	Iterator(false), _env(env), _pFunc(pFunc), _doneFlag(false)
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, valListArg);
	_pValListToWrite = &_valListA;
	_pValListToRead = &_valListB;
	_pValueRead = _pValListToRead->begin();
	_readBlock.blockedFlag = false;
	_writeBlock.blockedFlag = false;
}

Iterator_Fork::~Iterator_Fork()
{
	Function::Delete(_pFunc);
}

bool Iterator_Fork::DoNext(Signal sig, Value &value)
{
	if (_pValueRead == _pValListToRead->end()) {
		for (;;) {
			SwapList();
			if (_pValueRead != _pValListToRead->end()) {
				break;
			} else if (_doneFlag) {
				return false;
			} else {
				_readBlock.blockedFlag = true;
				_readBlock.event.Wait();
			}
		}
	}
	value = *_pValueRead++;
	return true;
}

String Iterator_Fork::ToString(Signal sig) const
{
	return String("<iterator:fork>");
}

void Iterator_Fork::SwapList()
{
	_semaphore.Wait();
	ValueList *pValList = _pValListToWrite;
	_pValListToWrite = _pValListToRead;
	_pValListToRead = pValList;
	_pValListToWrite->clear();
	_pValueRead = _pValListToRead->begin();
	if (_writeBlock.blockedFlag) {
		_writeBlock.blockedFlag = false;
		_writeBlock.event.Notify();
	}
	_semaphore.Release();
}

void Iterator_Fork::ForkProcess()
{
	Clone();
	Start();
}

void Iterator_Fork::Run()
{
	Signal sig;
	ValueList valList;
	while (_iterOwner.Next(sig, valList)) {
		Context context(valList);
		Value value = _pFunc->Eval(_env, sig, context);
		if (sig.IsSignalled()) break;
		_semaphore.Wait();
		_pValListToWrite->push_back(value);
		_semaphore.Release();
		if (_readBlock.blockedFlag) {
			_readBlock.blockedFlag = false;
			_readBlock.event.Notify();
		}
	}
	_doneFlag = true;
}

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
Iterator_Delay::~Iterator_Delay()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Delay::DoNext(Signal sig, Value &value)
{
	OAL::Sleep(_delay);
	return _pIterator->Next(sig, value);
}

String Iterator_Delay::ToString(Signal sig) const
{
	return String("<iterator:delay>");
}

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
Iterator_Skip::~Iterator_Skip()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Skip::DoNext(Signal sig, Value &value)
{
	bool flag = _pIterator->Next(sig, value);
	for (int i = 0; flag && i < _nSkip; i++) {
		Value valueTmp;
		flag = _pIterator->Next(sig, valueTmp);
	}
	return flag;
}

String Iterator_Skip::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:skip:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
Iterator_SkipInvalid::~Iterator_SkipInvalid()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_SkipInvalid::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		if (value.IsValid()) return true;
	}
	return false;
}

String Iterator_SkipInvalid::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:skip_invalid:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Filter
//-----------------------------------------------------------------------------
Iterator_Filter::~Iterator_Filter()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_Filter::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		ValueList valList(value);
		Context context(valList);
		Value valueCriteria = _pObjFunc->Eval(_env, sig, context);
		if (valueCriteria.GetBoolean()) return true;
	}
	return false;
}

String Iterator_Filter::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:filter:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_FilterEach
//-----------------------------------------------------------------------------
Iterator_FilterEach::~Iterator_FilterEach()
{
	Iterator::Delete(_pIterator);
	Iterator::Delete(_pIteratorCriteria);
}

bool Iterator_FilterEach::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		Value valueCriteria;
		if (!_pIteratorCriteria->Next(sig, valueCriteria)) break;
		if (valueCriteria.GetBoolean()) return true;
	}
	return false;
}

String Iterator_FilterEach::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:filter_each:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_While
//-----------------------------------------------------------------------------
Iterator_While::~Iterator_While()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_While::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	ValueList valList(value);
	Context context(valList);
	Value valueCriteria = _pObjFunc->Eval(_env, sig, context);
	return valueCriteria.GetBoolean();
}

String Iterator_While::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:while:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_WhileEach
//-----------------------------------------------------------------------------
Iterator_WhileEach::~Iterator_WhileEach()
{
	Iterator::Delete(_pIterator);
	Iterator::Delete(_pIteratorCriteria);
}

bool Iterator_WhileEach::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	Value valueCriteria;
	if (!_pIteratorCriteria->Next(sig, valueCriteria)) return false;
	return valueCriteria.GetBoolean();
}

String Iterator_WhileEach::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:while_each:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Replace
//-----------------------------------------------------------------------------
Iterator_Replace::~Iterator_Replace()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Replace::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	if (Value::Compare(value, _value) == 0) value = _valueReplace;
	return true;
}

String Iterator_Replace::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:replace:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_ReplaceInvalid
//-----------------------------------------------------------------------------
Iterator_ReplaceInvalid::~Iterator_ReplaceInvalid()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_ReplaceInvalid::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	if (value.IsInvalid()) value = _valueReplace;
	return true;
}

String Iterator_ReplaceInvalid::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:replace_invalid:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Format
//-----------------------------------------------------------------------------
Iterator_Format::~Iterator_Format()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Format::DoNext(Signal sig, Value &value)
{
	Value valueSrc;
	if (!_pIterator->Next(sig, valueSrc)) return false;
	String str;
	if (valueSrc.IsList()) {
		str = Formatter::Format(sig, _format.c_str(), valueSrc.GetList());
	} else {
		ValueList valList(valueSrc);
		str = Formatter::Format(sig, _format.c_str(), valList);
	}
	value = Value(_env, str.c_str());
	return true;
}

String Iterator_Format::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:format:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Pack
//-----------------------------------------------------------------------------
Iterator_Pack::~Iterator_Pack()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Pack::DoNext(Signal sig, Value &value)
{
	Value valueSrc;
	if (!_pIterator->Next(sig, valueSrc)) return false;
	Object_Bytes *pObjBytes = value.InitAsBytes(_env);
	size_t offset = 0;
	if (valueSrc.IsList()) {
		pObjBytes->Pack(sig, offset, _format.c_str(), valueSrc.GetList());
	} else {
		ValueList valList(valueSrc);
		pObjBytes->Pack(sig, offset, _format.c_str(), valList);
	}
	if (sig.IsSignalled()) {
		value = Value::Null;
		return false;
	}
	return true;
}

String Iterator_Pack::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:pack:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Zip
//-----------------------------------------------------------------------------
Iterator_Zip::~Iterator_Zip()
{
}

bool Iterator_Zip::DoNext(Signal sig, Value &value)
{
	ValueList &valList = value.InitAsList(_env);
	return _iterOwner.Next(sig, valList);
}

String Iterator_Zip::ToString(Signal sig) const
{
	return String("<iterator:zip>");
}

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
Iterator_Align::~Iterator_Align()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Align::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	if (!_pIterator->Next(sig, value)) value = _valueFill;
	return true;
}

String Iterator_Align::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:align:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Head
//-----------------------------------------------------------------------------
Iterator_Head::~Iterator_Head()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Head::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	return _pIterator->Next(sig, value);
}

String Iterator_Head::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:head:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_EachFold
//-----------------------------------------------------------------------------
Iterator_EachFold::~Iterator_EachFold()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_EachFold::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	value = _valueNext;
	_cnt--;
	if (_cnt > 0 && !_pIterator->Next(sig, _valueNext)) _cnt = 0;
	return true;
}

String Iterator_EachFold::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:each_fold:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Fold
//-----------------------------------------------------------------------------
Iterator_Fold::~Iterator_Fold()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Fold::DoNext(Signal sig, Value &value)
{
	Value valueNext;
	if (!_pIterator->Next(sig, valueNext)) return false;
	value.InitAsIterator(_env,
				new Iterator_EachFold(_pIterator->IncRef(), _cnt, valueNext));
	return true;
}

String Iterator_Fold::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:fold:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Concat
//-----------------------------------------------------------------------------
Iterator_Concat::~Iterator_Concat()
{
}

bool Iterator_Concat::DoNext(Signal sig, Value &value)
{
	for ( ; _ppIterator != _iterOwner.end(); _ppIterator++) {
		Iterator *pIterator = *_ppIterator;
		if (pIterator->Next(sig, value)) return true;
	}
	return false;
}

String Iterator_Concat::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:concat>";
	return rtn;
}

void Iterator_Concat::Add(Iterator *pIterator)
{
	_iterOwner.push_back(pIterator);
	_ppIterator = _iterOwner.begin();
	if (pIterator->IsInfinite()) _infiniteFlag = true;
}

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
IteratorOwner::IteratorOwner(const IteratorOwner &iterOwner)
{
	reserve(iterOwner.size());
	foreach_const (IteratorOwner, ppIterator, iterOwner) {
		push_back((*ppIterator)->Clone());
	}
}

IteratorOwner::~IteratorOwner()
{
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator::Delete(*ppIterator);
	}
}

bool IteratorOwner::Next(Signal sig, ValueList &valList)
{
	valList.clear();
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator *pIterator = *ppIterator;
		Value value;
		if (!pIterator->Next(sig, value)) return false;
		valList.push_back(value);
	}
	return true;
}

bool IteratorOwner::IsInfinite() const
{
	foreach_const (IteratorOwner, ppIterator, *this) {
		if (!(*ppIterator)->IsInfinite()) return false;
	}
	return true;
}

}
