//
// Object_Function
//

#include "Object_Function.h"
#include "Expr.h"
#include "Module.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Function
//-----------------------------------------------------------------------------
bool Object_Function::IsFunction() const { return true; }

Object_Function::~Object_Function()
{
	Function::Delete(_pFunc);
}

Value Object_Function::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_Function::Eval(Environment &env, Signal sig, Context &context) const
{
	return _pFunc->Eval(env, sig, context);
}

Object *Object_Function::Clone() const
{
	return new Object_Function(*this);
}

String Object_Function::ToString(Signal sig, bool exprFlag)
{
	return _pFunc->ToString();
}

//-----------------------------------------------------------------------------
// Object_WrappedMethod
//-----------------------------------------------------------------------------
bool Object_WrappedMethod::IsWrappedMethod() const { return true; }

Object_WrappedMethod::~Object_WrappedMethod()
{
}

Value Object_WrappedMethod::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	contextExpr.SetSelf(_valueSelf);
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_WrappedMethod::Eval(Environment &env, Signal sig, Context &context) const
{
	context.SetSelf(_valueSelf);
	return _pFunc->Eval(env, sig, context);
}

Object *Object_WrappedMethod::Clone() const
{
	return new Object_WrappedMethod(*this);
}

String Object_WrappedMethod::ToString(Signal sig, bool exprFlag)
{
	String str;
	ObjectBase *pObj = _valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) return str;
	if (pObj->IsModule()) {
		str += dynamic_cast<Module *>(pObj)->GetName();
		str += ".";
	} else if (pObj->IsClass()) {
		str += dynamic_cast<Class *>(pObj)->GetName();
		str += ".";
	} else if (pObj->IsObject()) {
		str += dynamic_cast<Object *>(pObj)->GetClass()->GetName();
		str += "#";
	}
	str += _pFunc->ToString();
	return str;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Function
//-----------------------------------------------------------------------------
// function#name()
AScript_DeclareMethod(Function, name)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp("Get a name of the function object as a string.");
}

AScript_ImplementMethod(Function, name)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	return Value(env, pSelf->GetFunction()->GetName());
}

// function#help(help?:string)
AScript_DeclareMethod(Function, help)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "help", VTYPE_String, OCCUR_ZeroOrOnce);
	SetHelp(
		"Set a help string of the function object if argument help is specified.\n"
		"Otherwise, it returns a help string already defined.");
}

AScript_ImplementMethod(Function, help)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	Function *pFunc = pSelf->GetFunction();
	if (context.IsString(0)) {
		pFunc->SetHelp(context.GetString(0));
		return Value::Null;
	} else {
		return Value(env, pFunc->GetHelp());
	}
}

// function#expr()
AScript_DeclareMethod(Function, expr)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp("Get an expression object that is associated with the function object.");
}

AScript_ImplementMethod(Function, expr)
{
	Object *pObj = context.GetSelfObj();
	Value value;
	const Function *pFunc = dynamic_cast<Object_Function *>(pObj)->GetFunction();
	if (!pFunc->IsCustom()) {
		sig.SetError(ERR_ValueError, "function '%s' is a built-in function", pFunc->GetName());
		return Value::Null;
	}
	const FunctionCustom *pFuncCustom = dynamic_cast<const FunctionCustom *>(pFunc);
	value.InitAsExpr(env, pFuncCustom->GetExprBody()->IncRef());
	return value;
}

// Assignment
Class_Function::Class_Function(Environment &env) :
									Class(env.LookupClass(VTYPE_Object))
{
	// don't assign functions here, because the function objects themselves
	// shall be constructed here!!!!!!! instead, they must be assigned in
	// Prepare() funtion below.
}

void Class_Function::Prepare()
{
	AScript_AssignMethod(Function, name);
	AScript_AssignMethod(Function, help);
	AScript_AssignMethod(Function, expr);
}

Object *Class_Function::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
