#include "Parser.h"
#include "Option.h"
#include "Module_builtins.h"
#include "Module_os.h"
#include "Module_time.h"
#include "Module_string.h"
#if defined(HAVE_LIBREADLINE)
#include <readline/readline.h>
#include <readline/history.h>
#endif

namespace AScript {

void PrintVersion(FILE *fp);
void PrintHelp(FILE *fp);
void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec);
void AcceptChar(Environment &env, Signal sig,
									Codec *pCodec, Parser &parser, int ch);

//-----------------------------------------------------------------------------
// Main entry
//-----------------------------------------------------------------------------
int Main(int argc, const char *argv[])
{
	File::InitBaseDir(argv[0]);
	Signal sig;
	EnvironmentRoot env;
	static const Option::Info optInfoTbl[] = {
		{ "help",			'h', false	},
		{ "interactive",	'i', false	},
		{ "command",		'c', true	},
		{ "quiet",			'q', false	},
		{ "printcmdline",	'P', false	},
		{ "version",		'v', false	},
	};
	Option opt(optInfoTbl, NUMBEROF(optInfoTbl));
	bool rtn = opt.Parse(sig, argc, argv);
	env.SetupBuiltIn(sig, argc, argv);	// this must be after opt.Parse()
	if (!rtn) {
		::fprintf(stderr, "%s\n", sig.GetErrString().c_str());
		return 1;
	}
	if (opt.IsSet("version")) {
		PrintVersion(stderr);
		return 0;
	}
	if (opt.IsSet("help")) {
		PrintVersion(stderr);
		PrintHelp(stderr);
		return 0;
	}
	if (opt.IsSet("printcmdline") && argc > 1) {
		for (int i = 1; i < argc; i++) {
			::fprintf(stderr, (i == 1)? "%s" : " %s", argv[i]);
		}
		::fprintf(stderr, "\n");
	}
	::init_gen_rand(1234);	// initialize random generator SFMT
	AScript_Module(__builtins__)::MixIn(env, sig);
	AScript_Module(os)::Import(env, sig);
	AScript_Module(time)::Import(env, sig);
	AScript_Module(string)::Import(env, sig);
	bool interactiveFlag = true;
	if (opt.IsSet("command")) {
		Expr *pExpr = Parser().ParseString(env, sig, opt.GetString("command", ""));
		if (sig.IsSignalled()) {
			env.PutErrString(sig);
			return 1;
		}
		Value result = pExpr->Exec(env, sig);
		if (sig.IsSignalled()) {
			env.PutErrString(sig);
			return 1;
		} else if (result.IsValid()) {
			env.PutString(result.ToString(sig).c_str());
			env.PutString("\n");
		}
		interactiveFlag = false;
	}
	Expr *pExprMain = NULL;
	if (argc >= 2) {
		File file;
		file.Open(sig, argv[1], "r", NULL);
		if (sig.IsSignalled()) {
			env.PutErrString(sig);
			return 1;
		}
		pExprMain = Parser().ParseFile(env, sig, file);
		if (sig.IsSignalled()) {
			env.PutErrString(sig);
			return 1;
		}
		pExprMain->Exec(env, sig);
		if (sig.IsSignalled()) {
			env.PutErrString(sig);
			sig.ClearSignal();
		}
		file.Close();
		interactiveFlag = false;
	}
	if (interactiveFlag || opt.IsSet("interactive")) {
		PrintVersion(stdout);
		env.SetEchoFlag(true);
		std::auto_ptr<Codec> pCodecReader(new Codec_None());
		CodecFactory *pCodecFactory = CodecFactory::Lookup("cp932");
		if (pCodecFactory != NULL) {
			pCodecReader.reset(pCodecFactory->CreateDecoder(true));
		}
		ReadEvalPrintLoop(env, sig, pCodecReader.get());
	}
	Expr::Delete(pExprMain);
	return 0;
}

void PrintVersion(FILE *fp)
{
	::fprintf(fp, "AScript %s copyright (c) 2010- Dan-san\n", VERSION);
}

void PrintHelp(FILE *fp)
{
	::fprintf(fp,
"usage: ascript [option] [file] [arg] ...\n"
"available options:\n"
"-h      print this help\n"
"-i      interactive mode after running script file if specified\n"
"-c cmd  execute program from command line\n"
"-v      print version string\n"
	);
}

#if defined(HAVE_LIBREADLINE)
void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec)
{
	Parser parser;
	char *lineBuff = NULL;
	while (lineBuff = readline(env.GetPrompt(parser.IsContinued()))) {
		for (char *p = lineBuff; ; p++) {
			int ch = (*p == '\0')? '\n' : *p;
			AcceptChar(env, sig, pCodec, parser, ch);
			if (ch == '\n') break;
		}
		if (lineBuff[0] != '\0') {
			add_history(lineBuff);
		}
		free(lineBuff);
	}
}
#else
void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec)
{
	Parser parser;
	env.PutPrompt(false);
	for (;;) {
		int ch = ::fgetc(stdin);
		AcceptChar(env, sig, pCodec, parser, ch);
		if (ch < 0) break;
		if (ch == '\n') env.PutPrompt(parser.IsContinued());
	}
}
#endif

void AcceptChar(Environment &env, Signal sig,
								Codec *pCodecReader, Parser &parser, int ch)
{
	char chConv = '\0';
	Codec::Result rtn =
			pCodecReader->FeedChar(static_cast<unsigned char>(ch), chConv);
	if (rtn != Codec::RESULT_Complete) return;
	do {
		Expr *pExpr = parser.ParseChar(env, sig, chConv);
		if (sig.IsSignalled()) {
			if (sig.GetSignalType() == SIGTYPE_DetectEncoding) {
				sig.ClearSignal();
			} else {
				env.PutErrString(sig);
				sig.ClearSignal();
				parser.Reset();
			}
		} else if (pExpr != NULL) {
			Value result = pExpr->Exec(env, sig);
			if (sig.IsSignalled()) {
				env.PutErrString(sig);
				sig.ClearSignal();
			} else if (!env.GetEchoFlag()) {
				// nothing to do
			} else if (result.IsValid()) {
				env.PutString(result.ToString(sig).c_str());
				env.PutString("\n");
			}
			delete pExpr;
		}
	} while (pCodecReader->FollowChar(chConv));
}

}

int main(int argc, const char *argv[])
{
	return AScript::Main(argc, argv);
}
