#include "Function.h"
#include "Iterator.h"
#include "Expr.h"
#include "Object.h"
#include "Object_Custom.h"
#include "Object_Matrix.h"

namespace AScript {

//-----------------------------------------------------------------------------
// FunctionType
//-----------------------------------------------------------------------------
const char *GetFuncTypeName(FunctionType funcType)
{
	static const struct {
		FunctionType funcType;
		const char *name;
	} tbl[] = {
		{ FUNCTYPE_Function,	"function",		},
		{ FUNCTYPE_Class,		"class",		},
		{ FUNCTYPE_Instance,	"instance",		},
		{ FUNCTYPE_Block,		"block",		},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].funcType == funcType) return tbl[i].name;
	}
	return "unknown";
}

//-----------------------------------------------------------------------------
// ICallable
//-----------------------------------------------------------------------------
Value ICallable::Call(Environment &env, Signal sig,
			const Value &valueSelf, const Value &valueCar,
			const Expr_Caller *pExprCaller, const ExprList &exprArgs,
			const Function **ppFuncSuccRequester)
{
	if (ppFuncSuccRequester != NULL) {
		const Function *pFuncSuccRequester = *ppFuncSuccRequester;
		*ppFuncSuccRequester = NULL;
		if (pFuncSuccRequester != NULL) {
			if (!pFuncSuccRequester->IsSucceedable(this)) {
				pExprCaller->SetError(sig,
						ERR_SyntaxError, "invalid succeeding process");
				return Value::Null;
			}
		}
	}
	ContextExpr contextExpr(pExprCaller->GetAttrs(), pExprCaller->GetAttrsOpt(),
		pExprCaller->GetBlock(), ppFuncSuccRequester, valueSelf, valueCar, exprArgs);
	Value result = DoCall(env, sig, contextExpr);
	if (sig.IsSignalled()) {
		sig.AddExprCause(pExprCaller);
		return Value::Null;
	}
	return result;
}

//-----------------------------------------------------------------------------
// Function
//-----------------------------------------------------------------------------
bool Function::IsCustom() const			{ return false; }
bool Function::IsPos() const			{ return false; }
bool Function::IsNeg() const			{ return false; }
bool Function::IsPlus() const			{ return false; }
bool Function::IsMinus() const			{ return false; }
bool Function::IsMultiply() const		{ return false; }
bool Function::IsDivide() const			{ return false; }
bool Function::IsModulo() const			{ return false; }
bool Function::IsPower() const			{ return false; }
bool Function::IsContainCheck() const	{ return false; }
bool Function::IsStructPrototype() const{ return false; }

Class *Function::GetClassToConstruct() const { return NULL; }

Function::Function(const Function &func) : _cntRef(1),
	_pSymbol(func._pSymbol),
	_envScope(func._envScope), _declOwner(func._declOwner),
	_elemType(func._elemType), _mathSymbol(func._mathSymbol),
	_funcType(func._funcType),
	_rsltMode(func._rsltMode), _mapFlag(func._mapFlag),
	_flatFlag(func._flatFlag), _forkFlag(func._forkFlag), _attrsOpt(func._attrsOpt),
	_dynamicScopeFlag(func._dynamicScopeFlag),
	_allowTooManyArgsFlag(func._allowTooManyArgsFlag),
	_symbolFuncFlag(func._symbolFuncFlag)
{
	_blockInfo.occurPattern = func._blockInfo.occurPattern;
	_blockInfo.pSymbol = func._blockInfo.pSymbol;
	_blockInfo.insideScopeFlag = func._blockInfo.insideScopeFlag;
	_blockInfo.quoteFlag = func._blockInfo.quoteFlag;
}

Function::Function(Environment &envScope, const Symbol *pSymbol, FunctionType funcType) : _cntRef(1),
	_pSymbol(pSymbol), _envScope(envScope), _funcType(funcType),
	_elemType(Parser::ETYPE_Unknown),
	_rsltMode(RSLTMODE_Normal), _mapFlag(MAP_Off),
	_flatFlag(FLAT_Off), _forkFlag(FORK_Off),
	_dynamicScopeFlag(false), _allowTooManyArgsFlag(false), _symbolFuncFlag(false)
{
	_blockInfo.occurPattern = OCCUR_Zero;
	_blockInfo.pSymbol = NULL;
	_blockInfo.insideScopeFlag = false;
	_blockInfo.quoteFlag = false;
}

Function::~Function()
{
}

bool Function::CustomDeclare(Environment &env, Signal sig,
					const SymbolSet &attrsAcceptable, ContextExpr &contextExpr)
{
	// delcaration of arguments
	if (!_declOwner.Declare(env, sig, contextExpr.GetArgs())) return false;
	// declaration of attributes
	foreach_const (SymbolSet, ppSymbol, contextExpr.GetAttrs()) {
		const Symbol *pSymbol = *ppSymbol;
		if (pSymbol->IsIdentical(AScript_Symbol(map))) {
			_mapFlag = MAP_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(nomap))) {
			_mapFlag = MAP_Off;
		} else if (pSymbol->IsIdentical(AScript_Symbol(flat))) {
			_flatFlag = FLAT_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(noflat))) {
			_flatFlag = FLAT_Off;
		} else if (pSymbol->IsIdentical(AScript_Symbol(fork))) {
			_forkFlag = FORK_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(list))) {
			_rsltMode = RSLTMODE_List;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xlist))) {
			_rsltMode = RSLTMODE_XList;
		} else if (pSymbol->IsIdentical(AScript_Symbol(set))) {
			_rsltMode = RSLTMODE_Set;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xset))) {
			_rsltMode = RSLTMODE_XSet;
		} else if (pSymbol->IsIdentical(AScript_Symbol(iter))) {
			_rsltMode = RSLTMODE_Iterator;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xiter))) {
			_rsltMode = RSLTMODE_XIterator;
		} else if (pSymbol->IsIdentical(AScript_Symbol(void_))) {
			_rsltMode = RSLTMODE_Void;
		} else if (pSymbol->IsIdentical(AScript_Symbol(static_))) {
			// just ignore it
		} else if (pSymbol->IsIdentical(AScript_Symbol(dynamic_scope))) {
			_dynamicScopeFlag = true;
		} else if (pSymbol->IsIdentical(AScript_Symbol(symbol_func))) {
			_symbolFuncFlag = true;
		} else if (attrsAcceptable.IsSet(pSymbol)) {
			// nothing to do
		} else {
			sig.SetError(ERR_TypeError,
				"unsupported attribute '%s' for function declaration",
															pSymbol->GetName());
			return false;
		}
	}
	_attrsOpt = contextExpr.GetAttrsOpt();
	// declaration of a block
	if (!contextExpr.IsBlockSpecified()) return true;
	const ExprList &exprList = contextExpr.GetBlock()->GetExprList();
	if (exprList.size() != 1) {
		SetError_InvalidFunctionExpression(sig);
		return false;
	}
	const Expr *pExpr = exprList.front();
	OccurPattern occurPattern = OCCUR_Once;
	if (pExpr->IsSuffix()) {
		const Expr_Suffix *pExprSuffix = dynamic_cast<const Expr_Suffix *>(pExpr);
		pExpr = pExprSuffix->GetChild();
		occurPattern = pExprSuffix->GetOccurPattern();
		if (occurPattern == OCCUR_Invalid) {
			SetError_InvalidFunctionExpression(sig);
			return false;
		}
	}
	bool quoteFlag = false;
	if (pExpr->IsQuote()) {
		const Expr_Quote *pExprQuote = dynamic_cast<const Expr_Quote *>(pExpr);
		quoteFlag = true;
		pExpr = pExprQuote->GetChild();
	}
	if (!pExpr->IsSymbol()) {
		sig.SetError(ERR_TypeError,
				"content of block in a function declaration must be a symbol");
		return false;
	}
	const Expr_Symbol *pExprSym = dynamic_cast<const Expr_Symbol *>(pExpr);
	DeclareBlock(occurPattern, pExprSym->GetSymbol(), false, quoteFlag);
	foreach_const (SymbolSet, ppSymbol, pExprSym->GetAttrs()) {
		const Symbol *pSymbol = *ppSymbol;
		if (pSymbol->IsIdentical(AScript_Symbol(inside_scope))) {
			_blockInfo.insideScopeFlag = true;
		} else {
			sig.SetError(ERR_TypeError, 
				"unsupported attribute '%s' for block declaration",
														pSymbol->GetName());
			return false;
		}
	}
	return true;
}

void Function::CopyDeclare(const Function &func)
{
	_declOwner = func._declOwner;
}

Declaration *Function::DeclareArg(Environment &env, const Symbol *pSymbol,
						ValueType valType, OccurPattern occurPattern,
						bool listFlag, bool nomapFlag, Expr *pExprDefault)
{
	Declaration *pDecl = new Declaration(pSymbol,
					valType, occurPattern, listFlag, nomapFlag, pExprDefault);
	ASSUME(env, !_declOwner.IsVariableLength());
	ASSUME(env, !(!pDecl->IsOptional() &&
				!_declOwner.empty() && _declOwner.back()->IsOptional()));
	_declOwner.push_back(pDecl);
	return pDecl;
}

void Function::DeclareBlock(OccurPattern occurPattern,
					const Symbol *pSymbol, bool insideScopeFlag, bool quoteFlag)
{
	_blockInfo.occurPattern = occurPattern;
	_blockInfo.pSymbol = (pSymbol == NULL)? AScript_Symbol(block) : pSymbol;
	_blockInfo.insideScopeFlag = insideScopeFlag;
	_blockInfo.quoteFlag = quoteFlag;
}

void Function::SetHelp(const char *help)
{
	_help = help;
}

Value Function::EvalExpr(Environment &env, Signal sig, ContextExpr &contextExpr) const
{
	if (GetType() == FUNCTYPE_Instance && contextExpr.GetSelfObj() == NULL) {
		sig.SetError(ERR_ValueError,
			"object is expected as l-value of field");
		return Value::Null;
	} else if (GetType() == FUNCTYPE_Class && contextExpr.GetSelfClass() == NULL) {
		sig.SetError(ERR_ValueError,
			"class is expected as l-value of field");
		return Value::Null;
	} else if (_blockInfo.occurPattern == OCCUR_Once &&
											!contextExpr.IsBlockSpecified()) {
		sig.SetError(ERR_ValueError,
			"block must be specified for '%s'", ToString().c_str());
		return Value::Null;
	} else if (_blockInfo.occurPattern == OCCUR_Zero &&
											contextExpr.IsBlockSpecified()) {
		sig.SetError(ERR_ValueError,
			"block is unnecessary for '%s'", ToString().c_str());
		return Value::Null;
	}
	MapFlag mapFlag = _mapFlag;
	FlatFlag flatFlag = _flatFlag;
	ForkFlag forkFlag = _forkFlag;
	ResultMode rsltMode = _rsltMode;
	foreach_const (SymbolSet, ppSymbol, contextExpr.GetAttrs()) {
		const Symbol *pSymbol = *ppSymbol;
		if (pSymbol->IsIdentical(AScript_Symbol(map))) {
			mapFlag = MAP_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(nomap))) {
			mapFlag = MAP_Off;
		} else if (pSymbol->IsIdentical(AScript_Symbol(flat))) {
			flatFlag = FLAT_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(noflat))) {
			flatFlag = FLAT_Off;
		} else if (pSymbol->IsIdentical(AScript_Symbol(fork))) {
			forkFlag = FORK_On;
		} else if (pSymbol->IsIdentical(AScript_Symbol(list))) {
			rsltMode = RSLTMODE_List;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xlist))) {
			rsltMode = RSLTMODE_XList;
		} else if (pSymbol->IsIdentical(AScript_Symbol(set))) {
			rsltMode = RSLTMODE_Set;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xset))) {
			rsltMode = RSLTMODE_XSet;
		} else if (pSymbol->IsIdentical(AScript_Symbol(iter))) {
			rsltMode = RSLTMODE_Iterator;
		} else if (pSymbol->IsIdentical(AScript_Symbol(xiter))) {
			rsltMode = RSLTMODE_XIterator;
		} else if (pSymbol->IsIdentical(AScript_Symbol(void_))) {
			rsltMode = RSLTMODE_Void;
		} else if (_attrsOpt.IsSet(pSymbol)) {
			// nothing to do
		} else {
			sig.SetError(ERR_AttributeError,
				"unsupported attribute '%s' for '%s'",
										pSymbol->GetName(), ToString().c_str());
			return Value::Null;
		}
	}
	const ExprList &exprArgs = contextExpr.GetArgs();
	ValueList valListArg;
	Value valueWithDict;
	valueWithDict.InitAsDict(env);
	if (mapFlag == MAP_On && IsUnary() && (exprArgs.size() > 1 ||
						!exprArgs.empty() && exprArgs.front()->IsSuffix())) {
		if (!_declOwner.PrepareArgsForUnary(env, sig, this, exprArgs, valListArg, valueWithDict)) {
			return Value::Null;
		}
	} else if (!_declOwner.PrepareArgs(env, sig, this, exprArgs, valListArg, valueWithDict)) {
		return Value::Null;
	}
	Context context(contextExpr, rsltMode, flatFlag, valListArg, valueWithDict);
	if (IsModulo()) {
		const Value &valueLeft = valListArg[0];
		const Value &valueRight = valListArg[1];
		if (valueLeft.IsFunction()) {
			const Function *pFunc = valueLeft.GetFunction();
			Value result;
			if (!valueRight.IsList()) {
				ValueList valListArg(valueRight);
				Context contextSub(valListArg);
				result = pFunc->Eval(env, sig, contextSub);
			} else if (pFunc->GetMapFlag() == MAP_Off ||
								pFunc->IsApplicable(valueRight.GetList())) {
				Context contextSub(valueRight.GetList());
				result = pFunc->Eval(env, sig, contextSub);
			} else if (pFunc->IsUnary()) {
				ValueList valListArg(valueRight);
				Context contextSub(valListArg);
				result = pFunc->EvalMap(env, sig, contextSub);
			} else {
				Context contextSub(valueRight.GetList());
				result = pFunc->EvalMap(env, sig, contextSub);
			}
			return result;
		} else if (valueLeft.IsString()) {
			const char *format = valueLeft.GetString();
			if (!valueRight.IsList()) {
				String str = Formatter::Format(sig, format, ValueList(valueRight));
				if (sig.IsSignalled()) return Value::Null;
				return Value(env, str.c_str());
			} else {
				const ValueList &valList = valueRight.GetList();
				if (valList.IsFlat() && !valList.IsContainIterator()) {
					String str = Formatter::Format(sig, format, valList);
					if (sig.IsSignalled()) return Value::Null;
					return Value(env, str.c_str());
				} else {
					IteratorOwner iterOwner;
					foreach_const (ValueList, pValue, valList) {
						Iterator *pIterator = NULL;
						if (pValue->IsList() || pValue->IsIterator()) {
							pIterator = pValue->CreateIterator(sig);
							if (pIterator == NULL) return Value::Null;
						} else {
							pIterator = new Iterator_Constant(*pValue);
						}
						iterOwner.push_back(pIterator);
					}
					return Formatter::Format(env, sig, format, iterOwner);
				}
			}
		}
	} else if (IsMultiply()) {
		const Value &valueLeft = valListArg[0];
		const Value &valueRight = valListArg[1];
		if (valueLeft.IsFunction()) {
			const Function *pFunc = valueLeft.GetFunction();
			if (pFunc->IsUnary()) {
				// nothing to do
			} else if (valueRight.IsList()) {
				const ValueList &valList = valueRight.GetList();
				if (valList.IsFlat()) {
					const Function *pFuncSuccRequester = NULL;
					Context context(SymbolSet::Null, SymbolSet::Null,
								NULL, &pFuncSuccRequester, Value::Null, valList);
					return pFunc->Eval(env, sig, context);
				}
				Iterator *pIterator = valueRight.CreateIterator(sig);
				if (sig.IsSignalled()) return Value::Null;
				Iterator *pIteratorFuncBinder = new Iterator_FuncBinder(env,
									pFunc->IncRef(), Value::Null, pIterator);
				return pIteratorFuncBinder->Eval(env, sig, context);
			} else if (valueRight.IsIterator()) {
				Iterator *pIterator = valueRight.CreateIterator(sig);
				if (sig.IsSignalled()) return Value::Null;
				Iterator *pIteratorFuncBinder = new Iterator_FuncBinder(env,
									pFunc->IncRef(), Value::Null, pIterator);
				return Value(env, pIteratorFuncBinder);
			}
		} else if (valueLeft.IsMatrix() && valueRight.IsList() ||
				   valueLeft.IsList() && valueRight.IsMatrix()) {
			return Eval(env, sig, context);
		}
	}
	if (forkFlag == FORK_On) {
		return EvalFork(env, sig, context);
	} else if (mapFlag == MAP_Off || IsApplicable(valListArg)) {
		return Eval(env, sig, context);
	} else {
		return EvalMap(env, sig, context);
	}
}

Value Function::Eval(Environment &env, Signal sig, Context &context) const
{
	bool exprFlag = false;
	ValueList valListCasted;
	if (!_declOwner.ValidateAndCast(env, sig,
							this, context.GetArgs(), valListCasted, exprFlag)) {
		return Value::Null;
	}
	Context contextCasted(context, valListCasted, context.GetValueWithDict());
	Value value = DoEval(env, sig, contextCasted);
	if (context.IsRsltVoid()) return Value::Null;
	return value;
}

Value Function::EvalFork(Environment &env, Signal sig, Context &context) const
{
#if 0
	Iterator_Fork *pIterator = new Iterator_Fork(env, sig,
								IncRef(), context.GetSelf(), context.GetArgs());
	if (sig.IsSignalled()) {
		Iterator::Delete(pIterator);
		return Value::Null;
	}
	pIterator->ForkProcess();
	return Value(env, pIterator);
#endif
	return Value::Null;
}

Value Function::EvalMap(Environment &env, Signal sig, Context &context) const
{
	if (context.IsRsltIterator() || context.IsRsltXIterator()) {
		// nothing to do
	} else if (!(GetDeclList().ShouldGenerateIterator(context.GetArgs()) &&
													context.IsRsltNormal())) {
		return EvalMapRecursive(env, sig, NULL, context);
	}
	bool skipInvalidFlag = context.IsRsltXIterator();
	Iterator_ImplicitMap *pIterator = new Iterator_ImplicitMap(env, sig, IncRef(),
						context.GetSelf(), context.GetArgs(), skipInvalidFlag);
	if (sig.IsSignalled()) {
		Iterator::Delete(pIterator);
		return Value::Null;
	}
	return Value(env, pIterator);
}

Value Function::EvalMapRecursive(Environment &env, Signal sig,
				ResultListComposer *pResultListComposer, Context &context) const
{
	IteratorOwner iterOwner;
	if (!iterOwner.PrepareForMap(sig, GetDeclList(), context.GetArgs())) {
		return Value::Null;
	}
	Value result;
	std::auto_ptr<ResultListComposer> pResultListComposerOwner;
	if (pResultListComposer == NULL) {
		pResultListComposer = new ResultListComposer(env, context, result);
		pResultListComposerOwner.reset(pResultListComposer);
	}
	for (size_t n = 0; ; n++) {
		ValueList valListArg;
		if (!iterOwner.Next(sig, valListArg)) {
			if (sig.IsSignalled()) return Value::Null;
			if (n == 0 && !context.IsRsltVoid() &&
										pResultListComposerOwner.get() != NULL) {
				result.InitAsList(env);
			}
			break;
		}
		Context contextEach(context, valListArg, context.GetValueWithDict());
		if (IsApplicable(valListArg)) {
			Value valueEach = Eval(env, sig, contextEach);
			if (sig.IsSignalled()) return Value::Null;
			pResultListComposer->Store(valueEach);
		} else if (context.IsRsltFlat()) {
			EvalMapRecursive(env, sig, pResultListComposer, contextEach);
			if (sig.IsSignalled()) return Value::Null;
		} else {
			Value valueEach = EvalMapRecursive(env, sig, NULL, contextEach);
			if (sig.IsSignalled()) return Value::Null;
			pResultListComposer->Store(valueEach);
		}
	}
	return result;
}

Value Function::EvalOverride(Signal sig, Context &context, bool &evaluatedFlag) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Object *pObj = NULL;
	if (valueLeft.IsObject()) {
		Value valueObj = valueLeft;
		pObj = valueObj.GetObject();
	} else if (valueRight.IsObject()) {
		Value valueObj = valueRight;
		pObj = valueObj.GetObject();
	} else {
		evaluatedFlag = false;
		return Value::Null;
	}
	evaluatedFlag = true;
	return pObj->EvalMethod(sig, GetSymbol(), context.GetArgs(), evaluatedFlag);
}

Environment *Function::PrepareEnvironment(
						Environment &env, Signal sig, Context &context) const
{
	const ValueList &valListArg = context.GetArgs();
	const Expr_Block *pExprBlock = context.GetBlock();
	EnvType envType =
		(_funcType == FUNCTYPE_Class)?		ENVTYPE_Method :
		(_funcType == FUNCTYPE_Instance)?	ENVTYPE_Method :
		(_funcType == FUNCTYPE_Block)?		ENVTYPE_Block :
		ENVTYPE_Local;
	Environment *pEnvLocal = new Environment(&env, envType);
	_declOwner.AssignAsDeclaration(*pEnvLocal, sig, valListArg);
	if (sig.IsSignalled()) {
		delete pEnvLocal;
		return NULL;
	}
	if (_declOwner.GetSymbolDict() != NULL) {
		Value valueWithDict = context.GetValueWithDict();
		if (valueWithDict.IsInvalid()) {
			valueWithDict.InitAsDict(env);
		}
		pEnvLocal->AssignValue(_declOwner.GetSymbolDict(), valueWithDict, false);
	}
	if (_blockInfo.occurPattern == OCCUR_Zero) {
		// nothing to do
	} else if (pExprBlock == NULL) {
		// set nil value to the variable with a symbol specified by
		// _blockInfo.pSymbol
		pEnvLocal->AssignValue(_blockInfo.pSymbol, Value::Null, false);
	} else if (_blockInfo.quoteFlag) {
		Value value;
		value.InitAsExpr(env, pExprBlock->IncRef());
		pEnvLocal->AssignValue(_blockInfo.pSymbol, value, false);
	} else {
		Environment *pEnv = _blockInfo.insideScopeFlag? pEnvLocal : &env;
		FunctionCustom *pFuncBlock = FunctionCustom::CreateBlockFunc(*pEnv, sig,
							_blockInfo.pSymbol, pExprBlock, FUNCTYPE_Block);
		if (pFuncBlock == NULL) {
			delete pEnvLocal;
			return NULL;
		}
		pEnvLocal->AssignFunction(pFuncBlock);
	}
	return pEnvLocal;
}

bool Function::IsSucceedable(const ICallable *pCallable) const
{
	return false;
}

Value Function::ReturnIterator(Environment &env, Signal sig,
								Context &context, Iterator *pIterator) const
{
	if (sig.IsSignalled()) {
		Iterator::Delete(pIterator);
		return Value::Null;
	}
	Value result;
	if (context.IsBlockSpecified()) {
		Environment envBlock(&env, ENVTYPE_Block);
		const Function *pFuncBlock =
						context.GetBlockFunc(envBlock, sig, GetSymbolForBlock());
		if (pFuncBlock == NULL) return Value::Null;
		// :iter and :xiter must be effective here
		result = pIterator->Eval(envBlock, sig, context, pFuncBlock);
		Iterator::Delete(pIterator);
		if (sig.IsSignalled()) return Value::Null;
	} else if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		result = pIterator->Eval(env, sig, context);
		Iterator::Delete(pIterator);
		if (sig.IsSignalled()) return Value::Null;
	} else {
		pIterator->SetSkipInvalidFlag(context.IsRsltXIterator());
		result = Value(env, pIterator);
	}
	return result;
}

Value Function::DoRepeater(Environment &env, Signal sig,
								Context &context, Iterator *pIterator) const
{
	if (sig.IsSignalled()) {
		Iterator::Delete(pIterator);
		return Value::Null;
	}
	Value result;
	if (context.IsRsltIterator() || context.IsRsltXIterator()) {
		result = Value(env, pIterator);
	} else if (context.IsRsltList() || context.IsRsltXList() ||
									context.IsRsltSet() || context.IsRsltXSet()) {
		result = pIterator->Eval(env, sig, context);
		Iterator::Delete(pIterator);
		if (sig.IsSignalled()) return Value::Null;
	} else {
		while (pIterator->Next(sig, result)) ;
		Iterator::Delete(pIterator);
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

Expr *Function::DiffUnary(Environment &env, Signal sig,
							const Expr *pExprArg, const Symbol *pSymbol) const
{
	SetError_MathDiffError(sig);
	return NULL;
}

Expr *Function::DiffBinary(Environment &env, Signal sig,
		const Expr *pExprArg1, const Expr *pExprArg2, const Symbol *pSymbol) const
{
	SetError_MathDiffError(sig);
	return NULL;
}

Expr *Function::OptimizeUnary(Environment &env, Signal sig, Expr *pExprOpt) const
{
	SetError_MathOptimizeError(sig);
	return NULL;
}

Expr *Function::OptimizeBinary(Environment &env, Signal sig,
										Expr *pExprOpt1, Expr *pExprOpt2) const
{
	SetError_MathOptimizeError(sig);
	return NULL;
}

String Function::ToString() const
{
	String str(GetName());
	if (GetSymbol()->IsFlowControlSymbol()) {
		str += " ";
	}
	str += "(";
	str += _declOwner.ToString();
	str += ")";
	if (_funcType == FUNCTYPE_Class) {
		str += ":";
		str += AScript_Symbol(static_)->GetName();
	} else if (_funcType == FUNCTYPE_Block) {
		str += ":";
		str += AScript_Symbol(block)->GetName();
	}
	if (_mapFlag == MAP_On) {
		str += ":";
		str += AScript_Symbol(map)->GetName();
	}
	if (_flatFlag == FLAT_On) {
		str += ":";
		str += AScript_Symbol(flat)->GetName();
	}
	if (_dynamicScopeFlag) {
		str += ":";
		str += AScript_Symbol(dynamic_scope)->GetName();
	}
	if (_symbolFuncFlag) {
		str += ":";
		str += AScript_Symbol(symbol_func)->GetName();
	}
	if (_rsltMode == RSLTMODE_List) {
		str += ":";
		str += AScript_Symbol(list)->GetName();
	} else if (_rsltMode == RSLTMODE_XList) {
		str += ":";
		str += AScript_Symbol(xlist)->GetName();
	} else if (_rsltMode == RSLTMODE_Set) {
		str += ":";
		str += AScript_Symbol(set)->GetName();
	} else if (_rsltMode == RSLTMODE_XSet) {
		str += ":";
		str += AScript_Symbol(xset)->GetName();
	} else if (_rsltMode == RSLTMODE_Void) {
		str += ":";
		str += AScript_Symbol(void_)->GetName();
	} else if (_rsltMode == RSLTMODE_Iterator) {
		str += ":";
		str += AScript_Symbol(iter)->GetName();
	} else if (_rsltMode == RSLTMODE_XIterator) {
		str += ":";
		str += AScript_Symbol(xiter)->GetName();
	} else if (_forkFlag == FORK_On) {
		str += ":";
		str += AScript_Symbol(fork)->GetName();
	}
	if (!_attrsOpt.empty()) {
		str += ":[";
		foreach_const (SymbolSet, ppSymbol, _attrsOpt) {
			const Symbol *pSymbol = *ppSymbol;
			if (ppSymbol != _attrsOpt.begin()) str += ",";
			str += pSymbol->GetName();
		}
		str += "]";
	}
	if (_blockInfo.occurPattern != OCCUR_Zero) {
		str += " {";
		if (_blockInfo.quoteFlag) str += "`";
		str += _blockInfo.pSymbol->GetName();
		str += GetOccurPatternSymbol(_blockInfo.occurPattern)->GetName();
		if (_blockInfo.insideScopeFlag) str += ":inside_scope";
		str += "}";
	}
	return str;
}

void Function::SetError_UnsupportedAttr(Signal sig, const SymbolSet &attrs) const
{
	String str;
	str += "function '";
	str += GetName();
	str += "' doesn't support attribute ";
	foreach_const (SymbolSet, ppSymbol, attrs) {
		if (ppSymbol != attrs.begin()) str += ", ";
		str += "'";
		str += (*ppSymbol)->GetName();
		str += "'";
	}
	sig.SetError(ERR_AttributeError, "%s", str.c_str());
}

void Function::SetError_DivideByZero(Signal sig) const
{
	sig.SetError(ERR_ZeroDivisionError, "divide by zero");
}

void Function::SetError_NotConstructor(Signal sig) const
{
	sig.SetError(ERR_ValueError, "'%s' is not a constructor", GetName());
}

void Function::SetError_ArgumentTypeByIndex(Signal sig,
						size_t idxArg, const Value &value) const
{
	if (idxArg < _declOwner.size()) {
		const Declaration *pDecl = _declOwner[idxArg];
		pDecl->SetError_ArgumentType(sig, value);
	} else {
		sig.SetError(ERR_TypeError, "argument error");
	}
}

void Function::SetError_InvalidValue(Signal sig, const Value &value) const
{
	sig.SetError(ERR_TypeError, "can't evaluate %s(%s)",
				GetName(), value.ToString(sig).c_str());
}

void Function::SetError_InvalidValue(Signal sig, const Value &value1, const Value &value2) const
{
	sig.SetError(ERR_TypeError, "can't evaluate %s(%s, %s)",
				GetName(), value1.ToString(sig).c_str(), value2.GetTypeName());
}

void Function::SetError_InvalidValType(Signal sig, const Value &value) const
{
	sig.SetError(ERR_TypeError, "can't evaluate %s(%s)",
				GetName(), value.GetTypeName());
}

void Function::SetError_InvalidValType(Signal sig, const Value &value1, const Value &value2) const
{
	sig.SetError(ERR_TypeError, "can't evaluate %s(%s, %s)",
				GetName(), value1.GetTypeName(), value2.GetTypeName());
}

void Function::SetError_InvalidValTypeM(Signal sig, const Value &value1, const Value &value2) const
{
	sig.SetError(ERR_TypeError, "can't evaluate (%s %s %s)",
				value1.GetTypeName(), GetMathSymbol(), value2.GetTypeName());
}

void Function::SetError_InvalidFunctionExpression(Signal sig) const
{
	sig.SetError(ERR_SyntaxError, "invalid function expression");
}

void Function::SetError_MathDiffError(Signal sig) const
{
	sig.SetError(ERR_ValueError, "failed to generate a differential function");
}

void Function::SetError_MathOptimizeError(Signal sig) const
{
	sig.SetError(ERR_ValueError, "mathematical optimization is not supported");
}

//-----------------------------------------------------------------------------
// Function::ResultListComposer
// this function's behaviour is affected by the following attributes.
//   :void, :list, :xlist, :set, :xet, :flat
//-----------------------------------------------------------------------------
Function::ResultListComposer::ResultListComposer(Environment &env,
										ContextBase &contextBase, Value &result) :
	_env(env), _contextBase(contextBase), _result(result), _pValList(NULL), _cnt(0),
	_excludeNilFlag(contextBase.IsRsltXList() || contextBase.IsRsltXSet()),
	_setFlag(contextBase.IsRsltSet() || contextBase.IsRsltXSet())
{
	if (_contextBase.IsRsltList() || _contextBase.IsRsltXList() ||
							_contextBase.IsRsltSet() || _contextBase.IsRsltXSet()) {
		_pValList = &_result.InitAsList(_env);
	}
}

void Function::ResultListComposer::Store(const Value &value)
{
	if (_contextBase.IsRsltVoid()) {
		// nothing to do
	} else if (_contextBase.IsRsltFlat() && value.IsList()) {
		const ValueList &valList = value.GetList();
		foreach_const (ValueList, pValue, value.GetList()) {
			Store(*pValue);
		}
	} else {
		if (_contextBase.IsRsltList()) {
			_pValList->push_back(value);
		} else if (value.IsValid()) {
			if (_pValList == NULL) {
				_pValList = &_result.InitAsList(_env, _cnt, Value::Null);
			}
			if (!_setFlag || !_pValList->IsContain(value)) {
				_pValList->push_back(value);
			}
		} else if (_excludeNilFlag) {
			// nothing to do
		} else if (_pValList != NULL) {
			if (!_setFlag || !_pValList->IsContain(value)) {
				_pValList->push_back(value);
			}
		}
		_cnt++;
	}
}

//-----------------------------------------------------------------------------
// FunctionCustom
//-----------------------------------------------------------------------------
bool FunctionCustom::IsCustom() const { return true; }

FunctionCustom::FunctionCustom(Environment &envScope,
				const Symbol *pSymbol, Expr *pExprBody, FunctionType funcType) :
		Function(envScope, pSymbol, funcType), _pExprBody(pExprBody)
{
}

FunctionCustom::~FunctionCustom()
{
	Expr::Delete(_pExprBody);
}

Value FunctionCustom::DoEval(Environment &env, Signal sig, Context &context) const
{
	Environment *pEnvOuter =
				_dynamicScopeFlag? &env : const_cast<Environment *>(&_envScope);
	std::auto_ptr<Environment> pEnvLocal(PrepareEnvironment(*pEnvOuter, sig, context));
	if (pEnvLocal.get() == NULL) return Value::Null;
	//if (_funcType == FUNCTYPE_Class || _funcType == FUNCTYPE_Instance) {
	if (_funcType != FUNCTYPE_Block) {
		pEnvLocal->AssignValue(AScript_Symbol(self), context.GetSelf(), false);
	}
	EnvType envType = pEnvLocal->GetType();
	Value result = GetExprBody()->Exec(*pEnvLocal, sig);
	if (envType == ENVTYPE_Block) {
		// nothing to do. simply pass the signal to the outside.
	} else if (sig.IsError()) {
		// nothing to do
	} else if (sig.IsBreak()) {
		sig.ClearSignal();
	} else if (sig.IsContinue()) {
		sig.ClearSignal();
	} else if (sig.IsReturn()) {
		result = sig.GetValue();
		sig.ClearSignal();
	}
	return result;
}

Expr *FunctionCustom::DiffUnary(Environment &env, Signal sig,
							const Expr *pExprArg, const Symbol *pSymbol) const
{
	SetError_MathDiffError(sig);
	return NULL;
}

FunctionCustom *FunctionCustom::CreateBlockFunc(Environment &env, Signal sig,
	const Symbol *pSymbol, const Expr_Block *pExprBlock, FunctionType funcType)
{
	const Expr_BlockParam *pExprBlockParam = pExprBlock->GetParam();
	if (pExprBlock->GetExprList().empty() && pExprBlockParam != NULL) {
		// check if the block parameter specifies a delegated block information
		// like "g() {|block|}"
		const ExprList &exprList = pExprBlockParam->GetExprList();
		if (exprList.size() == 1 && exprList.front()->IsSymbol()) {
			const Expr_Symbol *pExprSymbol =
						dynamic_cast<const Expr_Symbol *>(exprList.front());
			const Value *pValue = env.LookupValue(pExprSymbol->GetSymbol(), true);
			if (pValue == NULL) {
				// nothing to do
			} else if (pValue->IsExpr()) {
				const Expr *pExpr = pValue->GetExpr();
				if (!pExpr->IsBlock()) {
					sig.SetError(ERR_ValueError,
									"invalid value for block delegation");
					return NULL;
				}
				pExprBlock = dynamic_cast<const Expr_Block *>(pExpr);
				pExprBlockParam = pExprBlock->GetParam();
			} else {
				sig.SetError(ERR_ValueError,
								"invalid value for block delegation");
				return NULL;
			}
		}
	}
	FunctionCustom *pFunc = new FunctionCustom(env,
								pSymbol, pExprBlock->IncRef(), funcType);
	pFunc->AllowTooManyArgs(true);
	ContextExpr contextExpr(pExprBlockParam->GetExprList());
	if (pExprBlockParam != NULL &&
			!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
		Function::Delete(pFunc);
		return NULL;
	}
	return pFunc;
}

//-----------------------------------------------------------------------------
// ClassPrototype
//-----------------------------------------------------------------------------
ClassPrototype::ClassPrototype(Environment &envScope, const Symbol *pSymbol,
			Expr *pExprBody, Class_Custom *pClassCustom, FunctionType funcType) :
		Function(envScope, pSymbol, funcType), _envScope(envScope),
		_pExprBody(pExprBody), _pClassCustom(pClassCustom)
{
}

ClassPrototype::~ClassPrototype()
{
	Expr::Delete(_pExprBody);
	Class::Delete(_pClassCustom);
}

Value ClassPrototype::DoEval(Environment &env, Signal sig, Context &context) const
{
	Environment *pEnvOuter =
				_dynamicScopeFlag? &env : const_cast<Environment *>(&_envScope);
	std::auto_ptr<Environment> pEnvLocal(PrepareEnvironment(*pEnvOuter, sig, context));
	if (pEnvLocal.get() == NULL) return Value::Null;
	EnvType envType = pEnvLocal->GetType();
	Value valueSelf(context.GetSelf());
	if (!valueSelf.IsObject()) {
		Object *pObj = _pClassCustom->CreateDescendant(*pEnvLocal, sig, _pClassCustom);
		valueSelf.InitAsObject(pObj, _pClassCustom->GetValueType());
	}
	Class *pClassSuper = _pClassCustom->GetClassSuper();
	Function *pConstructorSuper =
				(pClassSuper == NULL)? NULL : pClassSuper->GetConstructor();
	if (pConstructorSuper != NULL) {
		const Expr *pExpr = GetExprBody();
		const ExprList *pExprList = &ExprList::Null;
		if (pExpr->IsBlock()) {
			const Expr_Block *pExprBlock = dynamic_cast<const Expr_Block *>(pExpr);
			const Expr_BlockParam *pExprParam = pExprBlock->GetParam();
			if (pExprParam != NULL) {
				pExprList = &pExprParam->GetExprList();
			}
		}
		Environment envSuper(pEnvLocal.get(), ENVTYPE_Method);
		ContextExpr contextExpr(SymbolSet::Null, SymbolSet::Null,
							NULL, NULL, valueSelf, Value::Null, *pExprList);
		pConstructorSuper->EvalExpr(envSuper, sig, contextExpr);
		if (sig.IsSignalled()) return Value::Null;
	}
	pEnvLocal->AssignValue(AScript_Symbol(self), valueSelf, false);
	GetExprBody()->Exec(*pEnvLocal, sig);
	if (sig.IsSignalled()) return Value::Null;
	return valueSelf;
}

Class *ClassPrototype::GetClassToConstruct() const
{
	return _pClassCustom;
}

//-----------------------------------------------------------------------------
// StructPrototype
//-----------------------------------------------------------------------------
bool StructPrototype::IsStructPrototype() const { return true; }

StructPrototype::StructPrototype(Environment &env, Class_Custom *pClassCustom) :
			Function(env, AScript_Symbol(_anonymous_), FUNCTYPE_Function),
			_pClassCustom(pClassCustom)
{
}

StructPrototype::~StructPrototype()
{
	Class::Delete(_pClassCustom);
}

Value StructPrototype::DoEval(Environment &env, Signal sig, Context &context) const
{
	Object *pObjSelf = NULL;
	Value valueSelf(context.GetSelf());
	if (valueSelf.IsObject()) {
		pObjSelf = valueSelf.GetObject();
	} else {
		pObjSelf = _pClassCustom->CreateDescendant(env, sig, _pClassCustom);
		valueSelf.InitAsObject(pObjSelf, _pClassCustom->GetValueType());
	}
	ValueList::const_iterator pValue = context.GetArgs().begin();
	DeclarationList::const_iterator ppDecl = GetDeclList().begin();
	for ( ; pValue != context.GetArgs().end() && ppDecl != GetDeclList().end();
														pValue++, ppDecl++) {
		const Declaration *pDecl = *ppDecl;
		pObjSelf->AssignValue(pDecl->GetSymbol(), *pValue, false);
	}
	return valueSelf;
}

Class *StructPrototype::GetClassToConstruct() const
{
	return _pClassCustom;
}

//-----------------------------------------------------------------------------
// ContextBase
//-----------------------------------------------------------------------------
ContextBase::ContextBase(const SymbolSet &attrs, const SymbolSet &attrsOpt,
		const Expr_Block *pExprBlock, const Function **ppFuncSuccRequester,
		const Value &valueSelf, ResultMode rsltMode, FlatFlag flatFlag) :
		_attrs(attrs), _attrsOpt(attrsOpt), _pExprBlock(pExprBlock), _pFuncBlock(NULL),
		_ppFuncSuccRequester(ppFuncSuccRequester), _valueSelf(valueSelf),
		_rsltMode(rsltMode), _flatFlag(flatFlag)
{
}

ContextBase::ContextBase(ContextBase &context) :
		_attrs(context._attrs), _attrsOpt(context._attrsOpt),
		_pExprBlock(context._pExprBlock),
		_pFuncBlock((context._pFuncBlock == NULL)?
									NULL : context._pFuncBlock->IncRef()),
		_ppFuncSuccRequester(context._ppFuncSuccRequester),
		_valueSelf(context._valueSelf),
		_rsltMode(context._rsltMode), _flatFlag(context._flatFlag)
{
}

ContextBase::ContextBase(ContextBase &context, ResultMode rsltMode, FlatFlag flatFlag) :
		_attrs(context._attrs), _attrsOpt(context._attrsOpt),
		_pExprBlock(context._pExprBlock),
		_pFuncBlock((context._pFuncBlock == NULL)?
									NULL : context._pFuncBlock->IncRef()),
		_ppFuncSuccRequester(context._ppFuncSuccRequester),
		_valueSelf(context._valueSelf),
		_rsltMode(rsltMode), _flatFlag(flatFlag)
{
}

ContextBase::~ContextBase()
{
	Function::Delete(_pFuncBlock);
}

// return NULL without error if block is not specified
const Function *ContextBase::GetBlockFunc(Environment &env, Signal sig, const Symbol *pSymbol)
{
	if (_pExprBlock == NULL || pSymbol == NULL) return NULL;
	if (_pFuncBlock == NULL) {
		_pFuncBlock = FunctionCustom::CreateBlockFunc(env, sig,
										pSymbol, _pExprBlock, FUNCTYPE_Block);
	}
	return _pFuncBlock;
}

}
