/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.Nullable;
import plus.util.Escape;

/**
 * Lexical Analyzer - Advance.
 *
 * @author kunio himei.
 */
abstract class Advance {

    //* 特殊な行終端文字.
    static final String YYEOL = "\n";
    //* Annotation バッファ.
    final LexArray<String> yyAnnotation = new LexArray<>();
    //* 行注釈を表わす正規表現.
    private static final LexRegx
            rxComment = new LexRegx("^(\\s*)(#.*|//.*|/\\*.*?\\*/)(.*)");
    //* 改行可能なトークンを表わす正規表現 ? , : { } ( && || do else.
    private static final LexRegx
            rxEat = new LexRegx("^([?,:{}(]|(&&)|(\\|\\|)|do|else)$");
    //* 先頭空白を表わす正規表現.
    private static final LexRegx rxLTrim = new LexRegx("^(\\s+).*");
    //* 空文を検出する正規表現.
    private static final LexRegx rxNL = new LexRegx("^\n");
    //* 改行をたべる正規表現.
    private static final LexRegx rxNlSemi = new LexRegx("^[;\n]");
    //* 文字列 """の作業用バッファ.
    private static final StringBuilder buf3str = new StringBuilder();
    //* Node が参照する現在のスクリプト行番号.
    private static int yyLineNumber;
    //* Log が参照する現在のスクリプト名.
    String yyScriptName;
    //* RegExp 解析を許可.
    int yyEnableANAREG; // U162prep3.awk: (100*$3/poptot, $2, 100*$2/ ..)
    //* token が左空白で区切られている.
    boolean yyHasLEFTSPACE;
    //* 現在のカラム位置.
    int yyLexColumn;
    //* 現在の Lex スクリプト行番号(ユーザでの変更は不可).
    int yyLexNumber;
    //* 旧テキストを退避.
    String yyOline = "";
    //* () の数.
    int yyPARENc;
    //* tokが([])となる前の状態を示す.
    int yyPARENca;
    //* 前回の差分を吸収.
    int yyPARENg;
    //* 入力テキスト.
    String yyText = "";
    //* トークン.
    Object tok;
    //* token が右空白で区切られている.
    private boolean yyHasRIGHTSPACE;
    //* ()[] の数.
    private int yyPARENa;

    /**
     * Node が参照する現在のスクリプト行番号を返す.
     */
    public static int yyLineNumber() {
        return yyLineNumber;
    }

    /**
     * @param x Node が参照する現在のスクリプト行番号を設定する.
     */
    static void yyLineNumber(int x) {
        yyLineNumber = x;
    }

    /**
     * 文字列 """（3 quotation mark）.
     */
    private static void append3qt(String x) {
        // 先頭から'|'までの左空白と改行を削除する.
        String xx = x.replaceAll("(^\\s*[|])|(\n$)", "");
        if (xx.endsWith("\\")) { // 最後が '\'なら改行を挿入しない.
            buf3str.append(xx, 0, xx.length() - 1);
        } else {
            buf3str.append(xx).append('\n');
        }
    }

    //* """ 解析、完了. 最後の """の手前では、改行しない.
    private static String finish3qt() {
        String x = buf3str.toString().replaceFirst("\n$", "");
        x = Escape.encodeString(x, "\""); // ダブルクォート文字列化.
        return "\"" + x + '"';
    }

    /**
     * advance.
     */
    @SuppressWarnings("null")
    final void advance() {
        if (null == yyText) {
            tok = null;
            return; // EOF
        }
        yyHasLEFTSPACE = yyHasRIGHTSPACE;
        while (yyText.isEmpty() || ("\n".equals(tok))) {
            buf3str.setLength(0); // 文字列 """の作業用バッファ.
            tok = null;
            yyText = null;
            boolean hasNext = true;
            boolean has3str = false;
            String w1 = "";
            while (hasNext) {
                String w = readline();
                if (null == w) {
                    break;
                }
                // 文字列 """
                if (w.contains("\"\"\"")) {
                    // Potential null pointer access: The variable w may be null at this location.
                    // Eclipse Juno BUG
                    while (has3str || w.contains("\"\"\"")) {
                        final int ix = w.indexOf("\"\"\"");
                        if (0 <= ix) { // 開始または、終了行
                            final String s1 = w.substring(0, ix); // 左側
                            final String s2 = w.substring(ix + 3); // 右側
                            if (has3str) {
                                append3qt(s1); // 左側、終了データ
                                w = s2; // 右側
                                has3str = false; // 終了
                            } else {
                                w1 = s1; // 左側
                                final int iy = s2.indexOf("\"\"\"");
                                buf3str.setLength(0); // 開始
                                if (0 > iy) { // この行では、終了していない.
                                    append3qt(s2); // 右側
                                    w = "";
                                    has3str = true;
                                } else {
                                    append3qt(s2.substring(0, iy)); // ☆
                                    w = s2.substring(iy + 3);
                                    has3str = false;
                                }
                            }
                        } else {
                            append3qt(w);
                            w = "";
                        }
                        if (has3str) {
                            w = readline();
                            if (null == w) {
                                hasNext = false;
                                break;
                            }
                        }
                    }
                    // """ 解析、完了.
                    w = w1 + finish3qt() + ((null == w) ? "" : w);
                    has3str = false;
                }
                if (hasNext) { // 継続行でコメントでない(先頭コメントのみ有効!)
                    if (w.endsWith("\\") && !rxComment.find(w)) {
                        w = w.substring(0, w.length() - 1);
                    } else {
                        hasNext = false;
                    }
                    yyText = ((null == yyText) ? w
                            : (yyText + w));
                }
            }
            yyOline = yyText; // 旧テキストに退避
            if (null == yyText) {
                return; // EOF
            }
            yyLexNumber++;
            yyLineNumber(yyLexNumber); // スクリプト行番号
            yyLexColumn = 0;
            if (0 < lTrim()) {
                yyText = yyText.trim();
            }
            doComment();
            if (!yyText.isEmpty()) {
                yyText += YYEOL;
            }
        }
        if (YYEOL.equals(yyText)) {
            yyText = "";
            tok = YYEOL;
            yyHasRIGHTSPACE = true;
        } else {
            yyText = yyText.trim();
            final boolean hasDblsp = yyText.contains("　"); // 全角空白ワーニング
            final String oldText = yyText;
            tok = nextToken();
            if (hasDblsp && (!yyText.contains("　"))) {
                yyINFOMATION("全角空白 >>" + oldText + "<<");
            }
            yyHasRIGHTSPACE = (0 < lTrim()); // tok の前に空白が存在するか?
            yyText += YYEOL; // 空にしない        // ↑(DOTALL無効なら)正規表現は`\n`が無い状態でのみ有効
        } //  ;TRACE("<advance----", yyHasRIGHTSPACE)
        yyPARENg += yyPARENa - yyPARENca; // 前回の差分を吸収
        yyPARENca = yyPARENa; // tokが([])となる前の状態を示す
        if (1 == tok.toString().length()) {
            if ("[".equals(tok)) {
                yyPARENa++;
            } else if ("]".equals(tok)) {
                yyPARENa--;
            } else if ("(".equals(tok)) {
                yyPARENa++;
                yyPARENc++;
            } else if (")".equals(tok)) {
                yyPARENa--;
                yyPARENc--;
            }
        }
    }

    /**
     * たべる.
     *
     * @param x 期待するトークン
     */
    final void eat(Object x) {
        if (!x.equals(tok))
            throw new IllegalStateException(
                    yyOline + '\n' +
                            "expect <" + x + "> actual <" + tok + '>');
        advance();
        if (rxNL.find(tok) && rxEat.find(x))
            nl();
    }

    /**
     * トークンが"ブレース '{'であることを強制する.
     */
    void requireLeftBraces() {
        if (!"{".equals(tok))
            throw new IllegalStateException("Lex '{' : '" + tok + "'");
    }

    /**
     * 次のトークン - Token.nextToken()
     */
    abstract Object nextToken();

    /**
     * 注釈.
     */
    abstract void doComment();

    /**
     * 左空白を削除して削除した文字数を返す.
     */
    final int lTrim() {
        if (rxLTrim.find(yyText)) {
            final String a = rxLTrim.group(1);
            dropMatchToken(a);
            return a.length();
        }
        return 0;
    }

    /**
     * yyText からマッチした Token文字列を削除する.
     */
    void dropMatchToken(String x) {
        int i = x.length();
        yyLexColumn += i;
        yyText = ((yyText.length() <= i) ? "" : yyText.substring(i));
    }

    /**
     * 改行をたべる; absorb newlines and semicolons.
     */
    final void nl() {
        while (rxNlSemi.find(tok)) {
            advance();
        }
    }

    /**
     * 空文をスキップする.
     */
    final void optSEMI() {
        while (("\\n".equals(tok)) || rxNL.find(tok)) {
            advance();
        }
    }

    /**
     * スクリプト入力.
     * 入力テキスト、EOF なら null
     */
    @Nullable
    abstract String readline();

    /**
     * INFORMATION メッセージを表示する.
     *
     * @param msg メッセージ
     */
    abstract void yyINFOMATION(Object msg);
}