/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.reflect;

import org.jetbrains.annotations.Nullable;
import plus.runtime.RegExp;
import plus.util.NumHelper;

import java.lang.reflect.*;
import java.util.Collection;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * Reflection.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public final class Reflection {

    /**
     * [%enclosure%] 空のオブジェクト配列.
     */
    static final Object[] EMPTY_OBJECT_ARRAY = {};

    /**
     * get Class Object.
     */
    public static Class<?> classForName(Object obj) {
        String name = obj.toString();
        Class<?> x = (obj instanceof Class<?>) ?
                (Class<?>) obj : Cache.CLASS.get(name);
        if (null == x) {
            try {
                x = Class.forName(name.isEmpty() ? "''" : name);
                Cache.CLASS.put(name, x);
            } catch (ClassNotFoundException e) {
                throw new IllegalArgumentException(e.toString());
            }
        }
        return x;
    }

    /**
     * クラスオブジェクトを返す.
     * object.
     */
    private static Class<?> getClassOf(Object obj) {
        return (null == obj) ? Object.class : ((obj instanceof Class<?>)
                ? (Class<?>) obj : obj.getClass());
    }

    /**
     * メソドまたはフィールドの存在確認 (Duck Typing).
     *
     * <code>('.'callExpression 'in' expression)</code>
     */
    public static boolean has(final Object obj, String name, Object[] args) {
        if (null != obj) {
            int arglen = args.length;
            if ((0 == arglen) && //
                    (("length".equals(name) //
                            && ((obj instanceof Object[])
                            || (obj instanceof CharSequence) //
                            || ((obj instanceof Class<?>) && (((Class<?>) obj)
                            .isArray() || (String.class == obj))))) || //
                            ("size".equals(name) //
                                    && ((obj instanceof Map<?, ?>) || (obj instanceof Collection<?>))))) {
                return true;
            }
            Class<?> clazz = getClassOf(obj);
            return (0 != RefHelper.getMethods(clazz, name, arglen).length)
                    || (null != RefHelper.getField(clazz, name));
        }
        return false;
    }

    /**
     * クラスの存在確認.
     */
    public static boolean hasClass(String name) {
        if ((null == name) || name.isEmpty())
            return false;
        Class<?> x = Cache.CLASS.get(name);
        if (null == x) {
            try {
                x = Class.forName(name);
                Cache.CLASS.put(name, x);

            } catch (ClassNotFoundException e) {
                return false; // 引き逃げ
            }
        }
        return true;
    }

    /**
     * フィールドの存在確認.
     */
    public static boolean hasField(String obj, String name) { //
        return 1 == hasMethodImpl(obj, name, 0);
    }

    /**
     * メソドの存在確認.
     */
    public static boolean hasMethod(String obj, String name, int argslen) {
        return 2 == hasMethodImpl(obj, name, argslen);
    }

    /**
     * メソドまたはフィールドの存在確認.
     */
    private static int hasMethodImpl(String obj, String name, int argslen) {
        if (hasClass(obj)) {
            Class<?> clazz = classForName(obj);
            if (0 != RefHelper.getMethods(clazz, name, argslen).length) {
                return 2; // メソド.
            }
            return ((0 == argslen) && (null != RefHelper.getField(clazz, name)))
                    ? 1 : // フィールド.
                    0;
        }
        return -1;
    }

    /**
     * ☆ 指定されたインターフェイスを実装する (オブジェクト実装 ).
     */
    public static Object imple(Class<?>[] interfaces,
                               Listener[] listeners) {
        return mixin(null, interfaces, listeners);
    }

    /**
     * ☆ 指定されたインターフェイスを実装する (オブジェクト実装 ).
     */
    @SuppressWarnings({"unchecked"})
    public static <E> E implement(Object interfac, Listener listener) {
        Class<?>[] clazz = new Class<?>[]{classForName(interfac)};
        Listener[] objs = {listener};
        return (E) mixin(null, clazz, objs);
    }

    /**
     * Invoke Method or Field.
     */
    private static Object invoke(Class<?> clazz, Object obj,
                                 String name, Object[] args) {
        Class<?>[] argTypes = toClass(args);
        Method method = null;
        Field x = Cache.FIELD.get(RefHelper.getFieldKey(clazz, name));
        if (null == x) {
            method = RefHelper.getMethod(clazz, name, args, argTypes);
            if (null == method) {
                x = RefHelper.getField(clazz, name);
            }
        }
        try {
            if (null != method) {
                return method.invoke(obj, RefHelper.cast(
                        method.getParameterTypes(), argTypes, args)); // メソド呼び出し
            }
            if (null != x) {
                // if (!x.isAccessible()) {
                x.setAccessible(true); // このオブジェクトの accessible フラグを設定
                // }
                if (0 != args.length) {
                    Object o = RefHelper.cast(x.getType(), argTypes[0],
                            args[0]);
                    x.set(obj, o); // フィールド値を設定
                    return o;
                }
                return x.get(obj); // フィールド値を取得
            }
            throw new IllegalArgumentException(RefHelper.mkString(
                    "NoSuchMethodException: " + clazz.getName() + '.' + name,
                    args));
        } catch (Exception e) {
            RefHelper.info("invoke: "
                    + ((null == method) ? clazz.getName() + '.' + name
                    : method));
            RefHelper.info(RefHelper.mkString("args: ", args));
            RefHelper.info(RefHelper.mkString("type: ", argTypes));
            throw new IllegalArgumentException(unwrap(e));
        }
    }

    /**
     * Invoke Method or Field.
     */
    public static Object invoke(Object obj, String name, Object[] args) {
        int arglen = args.length;
        if (0 == arglen) {
            if ("length".equals(name)) {
                if (obj instanceof Object[]) { // 配列オブジェクト
                    return ((Object[]) obj).length;
                } else if (obj instanceof CharSequence) {
                    return ((CharSequence) obj).length();
                }
            } else if ("size".equals(name)) { // short cut
                if (obj instanceof Map) {
                    return ((Map<?, ?>) obj).size();
                } else if (obj instanceof Collection) {
                    return ((Collection<?>) obj).size();
                }
            } else if ("toString".equals(name)) {
                return obj.toString();
            }
        }
        Class<?> clazz = getClassOf(obj);
        if ((Pattern.class == clazz) && "compile".equals(name)) { // 正規表現をコンパイル
            assert 0 != arglen;
            int flg = (1 < arglen) ? NumHelper.intValue(args[1]) : 0;
            String r = args[0].toString();
            return RegExp.compile(r, flg);

        }
        return invoke(clazz, obj, name, args);
    }

    /**
     * 指定されたオブジェクトに指定されたインターフェイスを実装する (ミックスイン).
     */
    public static Object mixin(final Object obj,
                               Class<?>[] interfaces, Listener[] listeners) {
        // 動的プロキシ
        return Proxy.newProxyInstance(Reflection.class.getClassLoader(),
                interfaces, new InvocationHandler() {
                    /**
                     * プロキシインスタンスのメソッドを呼び出し、その結果を返す.
                     * プロキシインスタンスでのメソッド呼び出しからの戻り値
                     */
                    @Nullable
                    @Override
                    public Object invoke(Object proxy,
                                         Method method, Object[] args) {
                        Object[] argv = (null == args)
                                ? EMPTY_OBJECT_ARRAY : args; // 引数の補正
                        /*
                         * invoke the method on the delegate. I only catch the
                         * InvocationTargetException here so that I can unwrap
                         * it and throw the contained target exception. If a
                         * checked exception is thrown by this method that is
                         * not assignable to any of the exception types declared
                         * in the throws clause of the interface method, then an
                         * UndeclaredThrowableException containing the exception
                         * that was thrown by this method will be thrown by the
                         * method invocation on the proxy instance.
                         */
                        String name = method.getName();
                        for (Listener x : listeners) {
                            // パラメータ長で絞り込む場合はここに追加する
                            if (name.equals(x.getName())) {
                                return x.apply(argv);
                            }
                        }
                        if (null != obj) {
                            return Reflection.invoke(obj, name, argv);
                        }
                        return null;
                    }
                });
    }

    /**
     * Create new Instance.
     */
    public static Object newInstance(String obj, Object[] args) {
        Class<?> clazz = classForName(obj);
        Class<?>[] argTypes = toClass(args);
        Constructor<?> x = RefHelper
                .getConstructor(clazz, args, argTypes);
        if (null == x) {
            throw new IllegalArgumentException(RefHelper.mkString(
                    "NoSuchConstructorException: " + clazz.getName(), args));
        }
        try {
            return (0 == args.length) ? x.newInstance() : //
                    x.newInstance(RefHelper.cast(x.getParameterTypes(),
                            argTypes, args));
        } catch (Exception e) {
            RefHelper.info("new: " + clazz.getName());
            RefHelper.info(RefHelper.mkString("args: ", args));
            RefHelper.info(RefHelper.mkString("type: ", argTypes));
            throw new RuntimeException(unwrap(e)); // アンラップした例外を投げる
        }
    }

    /**
     * クラスオブジェクト配列を返す.
     */
    private static Class<?>[] toClass(Object[] args) {
        int i = args.length;
        Class<?>[] arr = new Class[i];
        while (0 <= --i) {
            arr[i] = getClassOf(args[i]);
        }
        return arr;
    }

    /**
     * ラップされた例外チェーンを解除する.
     */
    private static Throwable unwrap(Throwable e) {
        Throwable ex = e;
        while (null != ex.getCause()) {
            ex = ex.getCause();
        }
        if (ex instanceof RuntimeException) {
            throw (RuntimeException) ex;
        }
        return ex;
    }
}