/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.eval;

import plus.concurrent.AtomicNumber;
import plus.exception.*;
import plus.lex.Flags;
import plus.lex.Keyword;
import plus.lex.Node;
import plus.util.NumHelper;
import plus.variable.Frame;

/**
 * Evaluate - Implement.
 *
 * @author kunio himei.
 */
abstract class EvalAssign extends EvalInvoke {

    /**
     * 割り込みクラスが代入互換の関係にあるかどうかを返す.
     */
    private boolean isCatchException(Throwable e, String[] catchs) {
        for (String x : catchs) {
            String path = applyImport(x).toString(); // ☆
            Class<?> xx = classForName(path);
            if (xx.isInstance(e)) return true;
        }
        return false;
    }

    /**
     * try　{} { catch (exception |... e) {... finally {}.
     */
    Object tryCatch(Node.Try e) throws Throwable {
        Throwable throwable = null;
        Object result = null;
        try {
            result = eval(e.e1);
        } catch (Exception ex) {
            boolean hasNext = true;
            for (int i = 0, len = e.e2.length; hasNext && (len > i); i++) {
                Node.Catch cat = e.e2[i];
                Frame._putLocalVar(cat.name, ex);
                boolean bo = isCatchException(ex, cat.claz);
                if (bo) {
                    result = eval(cat.stmt);
                    hasNext = false;
                }
            }
            if (hasNext)
                throwable = ex;
        } finally {
            eval(e.e3);
        }
        if (exists(throwable))
            throw throwable;
        return result;
    }

    /**
     * 代入文.
     */
    Object assignStmt(Node.Ass e) {
        String name = e.left.name;
        String index = mkIndex(e.left.index); // 左辺
        Object rval = eval(e.right); // 右辺
        Object dd;
        if ("=".equals(e.name)) {
            if (Keyword.SymVAR == e.id) { // VAR定義
                if (isNil(rval)) {
                    dd = null;
                } else if (Flags.isNumber(e.nType)) {
                    dd = new AtomicNumber(NumHelper.numberValue(rval));
                } else if (Flags.isString(e.nType)) {
                    dd = rval.toString();
                } else {
                    dd = _cloneAtom(rval); // REMIND Atomic を複製 ☆
                }
            } else { // Var以外
                dd = _strip(rval); // REMIND Atomic を剝がす ☆
            }
        } else {
            dd = rval;
        }
        if (!isGlobalDef // ローカル Var or Val定義
                && ((Keyword.SymVAL == e.id) || (Keyword.SymVAR == e.id))) {
            if (isNil(index)) {
                if (dd == null) dd = ""; //☆ PATCH Var/Val
                return Frame._putLocalVar(name, dd);
            }
            _VARIABLES._allocLocalArray(name, null);
        }
        return _putValue(name, index, e.name, dd); // グローバル定義
    }

    /**
     * 単純な文.
     */
    void simpleStmt(Node.Stmt e) {
        Object arg = isNil(e.e) ? 0 : eval(e.e);
        if (Keyword.SymBREAK == e.id) {
            throw BreakException.THIS;
        } else if (Keyword.SymCONTINUE == e.id) {
            throw ContinueException.THIS;
        } else if (Keyword.SymEXIT == e.id) {
            throw new ExitException(arg);
        } else if (Keyword.SymNEXT == e.id) {
            throw NextException.THIS;
        } else if (Keyword.SymNEXTFILE == e.id) {
            nextfile();
            throw NextException.THIS;
        } else if (Keyword.SymRETURN == e.id) {
            throw new ReturnException(arg);
        } else {
            throw new IllegalStateException("unmatch: " + e);
        }
    }
}