/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Function NumHelper.
 *
 * @author kunio himei.
 */
abstract class Function extends PerseHelper {

    //* 空の配列.
    static final Object[] EMPTY_ARRAY = {};
    //* 空の Integer 配列.
    private static final Integer[] EMPTY_INT_ARRAY = {};
    //* 関数名を '$' で分解する正規表現.
    private static final Pattern RX_DOLLAR_SPLIT = Pattern.compile("[$]");
    //* 関数名を '.' で分解する正規表現.
    private static final Pattern RX_DOT_SPLIT = Pattern.compile("[.]");
    //* has to be at least one.
    private static final LexRegx rxENDLIST = new LexRegx("^[<|>;)}\n]");
    //* 関数名からメソド名を削除するための正規表現.
    private static final Pattern RX_METHOD_NAME = Pattern.compile("[.][^.]+$");
    //* 関数呼出しパラメータ属性のマップ（定義した関数の利用判定にも使用）.
    final Map<String, Integer[]> callMAP = new HashMap<>();
    //* コードバッファ.
    final List<Object> codeBuf = new LexArray<>();
    //* 関数のマップ.
    final Map<String, Node.Func> functionMap = new HashMap<>();
    //* built-in 関数のマップ.
    private final Map<String, Node.Func> builtinMAP = new HashMap<>();
    //* CLOSURE カウンタ.
    int cCLOSURE;
    //* return カウンタ.
    int cRETURN;
    //* 現在処理中の変数スコープを含む関数名.
    String functionId = "";
    //* スクリプト行番号を退避.
    int lexlinenumber;
    //* 一時変数の連番.
    private int varSequence;

    /**
     * アクション - Action.action().
     */
    abstract void action();

    /**
     * 関数の呼び出し属性または空の属性を返す.
     */
    private Integer[] applyCallMAP(String k) {
        Integer[] x = callMAP.get(k);
        return (null == x) ? EMPTY_INT_ARRAY : x;
    }

    /**
     * 式または項を返す.
     */
    private Object callArgExpr(Object op) {
        return "(".equals(op) ? expression() : term();
    }

    /**
     * 関数呼出の引数を返す: arguments = [ (*empty*) '(' { expression } ')' ].
     */
    Object[] callArgments() {
        Collection<Object> buf = new ArrayList<>();
        while (("(".equals(super.tok) && !super.yyHasLEFTSPACE)
                || "{".equals(super.tok)) {
            Object op = super.tok;
            eat(super.tok);
            if (!rxENDLIST.find(super.tok)) { // has to be at least one
                buf.add(callArgExpr(op));
                while (",".equals(super.tok)) {
                    eat(super.tok);
                    buf.add(callArgExpr(op));
                }
            }
            if (("(".equals(op) && ")".equals(super.tok))
                    || ("{".equals(op) && "}".equals(super.tok))) {
                advance();
            }
        }
        // getline x <f >0
        // print >f
        return buf.toArray();
    }

    /**
     * call.
     */
    Node.YyCall callStmt(String name) { //
        return callStmtImpl(getFunctionId(name), callArgments());
    }

    /**
     * call(実装).
     */
    Node.YyCall callStmtImpl(String id, Object[] args) {
        boolean hasVarArgs = false;
        int argslen = args.length;
        Node.Func x = functionMap.get(id);
        int parmlen;
        int rtyp;
        if (null != x) {
            parmlen = x.argsLength;
            rtyp = x.nType;
            if ((0 < x.argsLength) && Flags.isVarArgs(x.parm[x.argsLength - 1].nType)) {
                hasVarArgs = true;
            }
        } else {
            x = builtinMAP.get(id);
            if (null != x) {
                parmlen = x.argsLength;
                rtyp = x.nType;
                if ((0 < x.argsLength) && Flags.isVarArgs(x.parm[x.argsLength - 1].nType)) {
                    hasVarArgs = true;
                }
            } else {
                parmlen = argslen;
                rtyp = Flags.T26NIL; // 現時点で判明している関数の戻り値を取得(事前定義されていない関数は未定義)
            } // 現在定義中のローカル関数※の属性は不完全なためこのMAPに登録されていない(@see #functionDecImpl)
        }
        Integer[] cald = applyCallMAP(id);
        int caldlen = cald.length;
        Integer[] arr = new Integer[(hasVarArgs) ? parmlen : Math.max(
                argslen, caldlen)];
        for (int i = 0, len = arr.length; len > i; i++) {
            int t1 = (caldlen > i) ? cald[i] : Flags.T26NIL;
            int m1 = t1 & Flags.T12X15M; // 回収済みの属性
            int t2 = (argslen > i) ? Type.getNodeType(args[i]) : t1;
            int m2 = t2 & Flags.T12X15M; // このスコープ属性
            int t3 = (m1 == m2) ? t2 : checkType(t2, t1);
            if ((Flags.T26NIL == t1) && (Flags.T26NIL == t2)) {
                t3 = Flags.T11ANY;
            } else if (Flags.isArray(t3) && (argslen > i)) {
                if (args[i] instanceof Node.NAME) {
                    int typ = Symbols.getType(((Node.NAME) args[i]).name);
                    if (Flags.isArray(typ)) {
                        t3 = typ; // 配列タイプを設定
                    }
                } else { // 引数が配列の要素指定の場合は単純変数のため、配列タイプをリセット
                    t3 &= ~Flags.T16ARRAY;
                }
            } // 変数への代入をリセット
            arr[i] = t3 & ~Flags.T20ASSIGN;
        }
        // 関数呼出しハラメータを強制的に設定する
        if ("split".equals(id)) { // split(s,A,fs)
            int typ = Flags.T20ASSIGN | Flags.T16ARRAY | Flags.T11ANY;
            updateType((Node.YyVariable) args[1], typ, typ);
        } else if ((2 < args.length)
                && (("gsub".equals(id)) || ("sub".equals(id)))) { // sub(r,s,T)
            int typ = Flags.T20ASSIGN | Flags.T18REF;
            updateType((Node.YyVariable) args[2], typ, typ);
        }
        callMAP.put(id, arr); // 関数呼出しパラメータ属性を記憶する
        return new Node.Call(id, args, rtyp); // 関数呼出し
    }

    /**
     * 関数定義(Function Declaration).
     */
    Node.Root functionDecl() {
        String name;
        boolean isAnonymous;
        if (super.tok instanceof Node.NAME x) { // 関数名有り.
            advance();
            name = x.name;
            isAnonymous = false;
        } else if (super.tok instanceof Keyword x) { // 予約語: keywords: getlin, print, ...
            advance();
            name = Keyword.toName(x);
            isAnonymous = false;
        } else { // 無名関数: anonymous function.
            name = makeAnonymousName();
            isAnonymous = true;
        }
        String fullname = Symbols.startLocal(makeFunctionId(name)); // 関数名設定.
        functionId = fullname;
        Object[] backup = unloadBuffer();
        functionDeclImpl(fullname); // 関数定義実装.
        restoreBuffer(backup);
        functionId = Symbols.endLocal(fullname); // clear local Symbols.
        if (!fullname.equals(name)) { // POSTIT インライン関数で出力.
            gen(new Node.FnI(fullname)); // POSTIT インライン定義位置.
        }
        return (isAnonymous) ? // 無名関数.
                new Node.Call(fullname, EMPTY_ARRAY, Symbols.getType(fullname))
                : new Node.FnI(fullname); // POSTIT インライン定義位置.
    }

    /**
     * 関数定義(Function Declaration).
     */
    private void functionDeclImpl(String name) {
        String[] anno = yyAnnotation.moveArray(); // POSTIT アノテーション
        LexArray<Node.FnP> parm = new LexArray<>();
        int linenumber = super.yyLexNumber; // ▼スクリプト行番号を退避
        Symbols.setLocalDefMode(true); // 関数パラメータ定義開始(local)
        boolean hasLength = false; // 呼び出しパラメータ長の確定の有無
        int callPrmlen = 0; // 呼び出しパラメータ長
        while ("(".equals(super.tok)) {
            eat(super.tok); // , パラメータリスト [(]term, ...[) [(...)]]
            if (!rxENDLIST.find(super.tok)) { // has to be at least one
                boolean hasNext = true;
                while (hasNext) { // パラメータリスト作成
                    Object o = term();
                    if (o instanceof Node.YyVariable x) {
                        T4Types rr = optType(x.name, false);
                        int typ = rr.nType;
                        String sType = rr.sType;
                        parm.add(new Node.FnP(x.name, typ, sType)); // 仮パラメータ属性
                    }
                    if (",".equals(super.tok)) {
                        eat(super.tok);
                    } else {
                        hasNext = false;
                    }
                }
            }
            eat(")");
            if (!hasLength) { // 呼び出しパラメータ長を固定
                hasLength = true;
                callPrmlen = parm.size();
            }
        }
        T4Types rr = optType(name, false); // 戻り値属性
        int typ = rr.nType;
        String sType = rr.sType;
        Symbols.setLocalDefMode(false); // 関数パラメータ定義終了
        int parmlen = parm.size();
        int plen = (hasLength) ? callPrmlen : parmlen; // 呼び出しパラメータ長を仮設定
        String[] comment = getCommentArray();
        if (";".equals(super.tok)) {
            eat(super.tok);
            nl(); // プロトタイプ宣言 name(parmlist, ...)
            Node.FnP[] fnp = parm.toArray(new Node.FnP[parmlen]);
            builtinMAP.put(name, new Node.Func(name, fnp, plen, typ, sType,
                    comment, anno, EMPTY_ARRAY));
        } else {
            Object[] backup = unloadBuffer();
            int closureCount = cCLOSURE; // CLOSURE の出現数
            int returnCount = cRETURN; // return の出現数
            action(); // POSTIT 関数本体定義
            for (int i = 0; parmlen > i; i++) { // ローカル変数の属性を回収
                Node.FnP x = parm.get(i);
                int t1 = Symbols.getType(x.name);
                int w = (Flags.T26NIL == x.nType) ? t1 : x.nType;
                w = checkType(w, t1);
                parm.set(i, new Node.FnP(x.name, w, x.sType)); // この時点での未定義変数は、読込専用または未使用変数
            }
            Object[] stmt = unloadBuffer();
            restoreBuffer(backup);
            int x = Symbols.getType(name);
            int ntype = (Flags.T26NIL == x) ? Flags.T00VOID : x;
            closureCount = cCLOSURE - closureCount; // CLOSURE カウンタ
            cCLOSURE -= closureCount;
            if (0 < closureCount) {
                ntype |= Flags.T22CLOSURE; // CLOSURE 属性を設定. ☆
            } // 複数の RETURN が出現
            returnCount = cRETURN - returnCount;
            cRETURN -= returnCount;
            if (1 < returnCount) {
                ntype |= Flags.T23RETURNS;
            } // 複数の return が出現
            Advance.yyLineNumber(linenumber); // ▼▲スクリプト行番号を復元
            Node.FnP[] fnp = parm.toArray(new Node.FnP[parmlen]);
            functionMap.put(name, new Node.Func(name, fnp, plen, ntype, sType,
                    comment, anno, stmt)); // POSTIT 関数登録
        }
    }

    /**
     * コード生成.
     */
    void gen(Object x) {
        codeBuf.add(x);
    }

    /**
     * 現在処理中の変数スコープを含む関数名を返す.
     */
    String getFunctionId(String name) {
        String k = makeFunctionId(name); // 現在処理中の変数スコープを含む関数名
        if (name.equals(functionId) || k.equals(functionId)
                || // 現在処理中
                functionMap.containsKey(k)
                || builtinMAP.containsKey(k)) { // 関数定義済み
            return k;
        } else if (0 <= name.indexOf('.')) {
            return name; // object呼出し（関数定義なし）
        } else { // 現在定義中の関数のローカル関数
            String[] parts = RX_DOLLAR_SPLIT.split(functionId);
            StringBuilder sb = new StringBuilder();
            for (String part : parts) {
                sb.append(part).append('$');
                String key = sb.append(name).toString();
                if (functionMap.containsKey(key)) {
                    return key;
                }
                sb.setLength(sb.length() - name.length()); // name を消す
            }
            return name; // ルート関数または後方で定義(または未定義)
        }
    }

    /**
     * 関数は定義済みかどうかを返す.
     */
    boolean hasFunction(String name) {
        String id = getFunctionId(name);
        return builtinMAP.containsKey(id)
                || functionMap.containsKey(id);
    }

    /**
     * 関数呼出し(invoke).
     */
    Node.Invoke invokeStmt(Keyword id, String name) {
        String[] parts = RX_DOT_SPLIT.split(name);
        parts[parts.length - 1] = getFunctionId(parts[parts.length - 1]); // 現在処理中の変数スコープを含む関数名
        Object obj;
        String method;
        if ((2 == parts.length) && Symbols.findType(parts[0])) { // 変数が定義済みであることを確認
            if (Symbols.isClosure(parts[0])) {
                cCLOSURE++; // POSTIT Symbols.CLOSURE に登録されている場合 ☆
            }
            obj = new Node.NAME(Keyword.SyyNAME, parts[0]);
            method = parts[1]; // instance object.method
        } else if (1 < parts.length) { // static package.Class.method
            obj = new Term.BOXING(RX_METHOD_NAME.matcher(name).replaceFirst(
                    ""));
            method = parts[parts.length - 1];
        } else { // `.NAME` function value
            obj = new Term.BOXING("");
            method = parts[parts.length - 1];
        }
        return invokeStmtImpl(id, obj, method);
    }

    /**
     * invoke(インスタンス).
     */
    Node.Invoke invokeStmtImpl(Keyword id, Object obj, String method) {
        Object[] args = EMPTY_ARRAY;
        boolean isMethod = false; // メソドかどうか
        if ("(".equals(super.tok)) { // ()
            args = parenlist();
            isMethod = true; // メソド呼出し
        }
        int type;
        String sType;
        if (":".equals(super.tok) && super.yyText.startsWith(":")) {
            // ':' が一つの場合、 ()? a.b : c で誤動作するため
            T4Types rr = optType(obj + "." + method, false); // 関数呼び出し属性
            type = rr.nType;
            sType = rr.sType;
        } else {
            type = Flags.T11ANY;
            sType = "";
        }
        if (obj.toString().isEmpty()) {
            callStmtImpl(method, args); // ユーザ関数の呼び出しを記憶する
        }
        return new Node.Invoke(id, obj, method, args, type, sType, isMethod);
    }

    /**
     *
     */
    private String makeAnonymousName() {
        return "_anon" + (++varSequence) + "_";
    }

    /**
     * 関数の呼び出し名を返す.
     */
    private String makeFunctionId(String name) {
        if (functionId.isEmpty() || name.equals(functionId)) {
            return name; // ルート関数
        }
        String x = '$' + name; // 現在処理中の変数スコープ
        return (functionId.endsWith(x)) ? functionId
                : (functionId + x);
    }

    /**
     * バッファを復元する.
     */
    void restoreBuffer(Object[] a) {
        codeBuf.clear();
        codeBuf.addAll(Arrays.asList(a));
        Advance.yyLineNumber(lexlinenumber); // ▼▲スクリプト行番号を復元
    }

    /**
     *
     */
    abstract Object term();

    /**
     * バッファをアンロードして返す.
     */
    Object[] unloadBuffer() {
        Object[] x = codeBuf.toArray();
        codeBuf.clear();
        return x;
    }

    /**
     * 関数パラメータ属性と関数呼び出し引数の属性をマージする.
     */
    void updateFunctionDef() {
        for (Map.Entry<String, Node.Func> e : functionMap.entrySet()) {
            String k = e.getKey();
            Node.Func x = e.getValue();
            Advance.yyLineNumber(x.linenumber); // スクリプト行番号を設定
            super.yyLexColumn = 0;
            boolean hasCall = callMAP.containsKey(k); // callで回収した属性
            Integer[] cald = applyCallMAP(k);
            int caldlen = cald.length;
            int parmlen = x.parm.length;
            Node.FnP[] arr = new Node.FnP[parmlen];
            for (int i = 0; parmlen > i; i++) {
                int t1 = (caldlen > i) ? cald[i] : Flags.T26NIL; // callで回収した属性
                int t2 = x.parm[i].nType; // 関数宣言の属性
                int typ = (Flags.T26NIL == t2) ? t1 : t2;
                if ((Flags.T26NIL == typ) && (x.argsLength <= i)) { // 未代入のローカル変数
                    yyINFOMATION("uninitialized Function Parameter: " + k
                            + "( " + x.parm[i].name + " )"); // 未解決な属性
                }
                arr[i] = new Node.FnP(x.parm[i].name, typ, x.parm[i].sType);
            }
            int siz = (parmlen == x.argsLength)
                    ? ((hasCall && (parmlen != caldlen)) ? caldlen : x.argsLength)
                    : x.argsLength; // 仮引数の数
            functionMap.put(k, new Node.Func(k, arr, siz, x.nType, x.sType,
                    x.comment, x.annotation, x.stmt));
            assert parmlen >= caldlen;
        }
    }
}