/*
 * Copyright (C) 2011 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.regex.Pattern;

/**
 * [%inet%] ソケット(Socket) ヘルパ.
 *
 * @author kunio himei.
 */
final class InetHelper {

    /**
     * プロトコルとポート番号の対応表.
     */
    private static final Object[][] INET_PORTNUMBER = {
            {"echo", 7}, // 入力をエコー表示
            {"discard", 9}, // 入力を廃棄
            {"daytime", 13}, // システム日付と時刻を返す
            {"chargen", 19}, // ASCII データのストリームを生成
            // { "ftp", 21}, { "telnet", 23},
            // { "smtp", 25}, { "finger", 79},
            {"http", 80},
            // { "pop2", 109}, { "pop3", 110},
            // { "nntp", 119}, { "ntp", 123},
            // { "irc", 194},
    };

    /**
     * 秒をミリ秒に変換する下駄 (milliseconds).
     */
    private static final int MILLISECOND = 1000;

    /**
     * inet アドレスを分解する正規表現.
     */
    private static final Pattern RX_INET_SPLIT = Pattern.compile("[/]");

    /**
     * 数字: [0-9.] を識別する正規表現.
     */
    private static final Pattern RX_NUMBER = Pattern.compile("^[0-9.]+$");

    /**
     * ポート番号を識別する正規表現.
     */
    private static final Pattern RX_PORT_NUMBER = Pattern.compile("^[0-9]+$");

    /**
     * private constructor (singleton).
     */
    private InetHelper() {
        super();
    }

    /**
     * IP ソケットアドレス (IP アドレス + ポート番号) を返す.
     *
     * @param host       ホスト名
     * @param localPort  ポート番号
     * @param remotePort ポート番号
     * @throws UnknownHostException host(IP アドレス) が見つからなかった
     */
    private static InetSocketAddress getAddress(final String host,
                                                final int localPort, final int remotePort)
            throws UnknownHostException {
        final boolean hasHost = !host.isEmpty() && !"0".equals(host);
        if (0 == localPort) {
            if (hasHost) { // 0/host/remotePort
                return new InetSocketAddress(host, remotePort);
            }
            return new InetSocketAddress(InetAddress.getLocalHost(), remotePort);
        }
        if (hasHost) { // localPort/host/?
            return new InetSocketAddress(host, localPort);
        }
        return new InetSocketAddress(localPort);
    }

    /**
     * ポート番号を取得する.
     *
     * @param port ポート
     */
    private static int getPort(final String port) {
        if (RX_PORT_NUMBER.matcher(port).matches()) {
            return Integer.parseInt(port);
        }
        for (final Object[] o : INET_PORTNUMBER) {
            if (port.equalsIgnoreCase((String) o[0])) {
                return (Integer) o[1];
            }
        }
        return 0;
    }

    /**
     * ソケットの構築.
     *
     * @param inet inet://tcp/localPort/hostname/remoteport
     */
    static Connectable open(final String inet) throws IOException {
        // if ((name of remote host given) && (other side accepts CONNECTIONS)) {
        // . rendez-vous successful; transmit with getline or print
        // } else {
        // . if ((other side did not accept) && (localPort == 0))
        // .. exit unsuccessful
        // . if (TCP) {
        // .. set up a server accepting connections
        // . this means waiting for the client on the other side to connect
        // . } else
        // .. ready
        // }
        if ((null != inet)) {
            final String[] parts = RX_INET_SPLIT.split(inet);
            // 0 file:///3 inet /4 protocol /5 localPort /6 hostname /7 remoteport
            // 0 /1 inet        /2
            int p = (parts[1].startsWith("inet")) ? 2 : 4;
            if ((p + 3) < parts.length) {
                final String protocol = parts[p++];
                final int localPort = getPort(parts[p++]);
                final String host = parts[p++];
                final int remotePort = getPort(parts[p++]);
                final int timeout = (int) (((p < parts.length) && RX_NUMBER
                        .matcher(parts[p]).matches()) ? (Double
                        .parseDouble(parts[p]) * MILLISECOND) : 0); // 接続待ちタイマ (省略時は∞)
                final InetSocketAddress addr = getAddress(host, localPort,
                        remotePort);
                // System.err.println("InetHelper: " + inet + ": " + addr);
                if ("tcp".equals(protocol)) {
                    if (0 == localPort) {
                        // java.nio.channels.SocketChannel soc =
                        //  java.nio.channels.SocketChannel.open(addr);

                        // サーバ側が起動していないとき、
                        //  ConnectException: Connection refused: connect
                        final Socket soc = new Socket(addr.getAddress(),
                                remotePort);
                        return new InetSocket(soc, timeout); // クライアント
                    }
                    return new InetServerSocket(addr, timeout);
                } else if ("udp".equals(protocol)) {
                    if (0 == localPort) {
                        return new InetDatagramSocket(addr, timeout); // クライアント
                    }
                    return new InetDatagramSocket(addr, localPort, timeout);
                }
            }
        }
        throw new FileNotFoundException(inet);
    }
}