/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import org.jetbrains.annotations.NotNull;
import plus.io.Device;
import plus.io.Io;
import plus.io.IoHelper;
import plus.spawn.system.UtilInterface;

import java.io.FilterWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;

/**
 * [%command%] implementation of the `cat` command.
 * <p>
 * Usage: cat [OPTION]... [FILE]...<br>
 * ファイルまたは標準入力を結合し、標準出力へ書き出す <br>
 * <br>
 * -n 全ての行の出力に行番号を付加<br>
 * -b 空でない行の出力に行番号を付加<br>
 * -s 連続した空行を圧縮する<br>
 * -T TAB文字を`^I'で表示<br>
 * -u 出力をバッファリングしない<br>
 * </p>
 *
 * @author kunio himei.
 */
public final class Cat extends FilterWriter implements UtilInterface {

    /**
     * TAB を識別する正規表現.
     */
    private static final Pattern RX_TAB = Pattern.compile("\t");

    /**
     * USAGE.
     */
    private static final String USAGE = """
            Usage: cat [OPTION] [FILE]...
            Concatenate FILE(s), or standard input, to standard output.

              -b    number nonblank output lines
              -n    number all output lines
              -s    never more than one single blank line
              -T    display TAB characters as ^I
              -u    The output is not buffered.
                  --help      display this help and exit (should be alone)

            With no FILE, or when FILE is -, read standard input.""";
    /**
     * -u 出力をバッファリングしない .
     */
    private final boolean hasAutoFlush;
    /**
     * -s 連続した空行を圧縮する(前回は空白行か?).
     */
    private final AtomicBoolean hasBlankLine;
    /**
     * -n 行番号カウンタ.
     */
    private final AtomicInteger hasNumberLines;
    /**
     * -b 非空白行に 1 から開始する番号を付ける.
     */
    private final boolean hasNumberNonblank;
    /**
     * -T タブ を `^I' で表示する.
     */
    private final boolean hasShowTab;
    /**
     * 入力ファイル名配列.
     */
    private final ConcurrentLinkedQueue<String> inputFiles = new ConcurrentLinkedQueue<>();

    /**
     * cat.
     */
    public Cat(String[] args, Writer output) {
        super(output);
        StringBuilder flags = new StringBuilder();
        for (String arg : args) {
            char c = arg.charAt(0);
            if (('-' == c)) {
                if (1 == arg.length()) {
                    this.inputFiles.add("-"); // 標準入力
                } else {
                    flags.append(arg.substring(1));
                }
            } else {
                this.inputFiles.add(Helper.filename(arg)); // 入力ファイル名
            }
        }
        // -b -n が指定された場合は -b が優先される
        this.hasNumberNonblank = (0 <= flags.indexOf("b")); // -b 非空白行に番号を付ける

        this.hasNumberLines = ((this.hasNumberNonblank || (0 <= flags
                .indexOf("n"))) ? new AtomicInteger() : null);

        this.hasBlankLine = ((0 <= flags.indexOf("s")) ? new AtomicBoolean(
                false) : null);

        this.hasShowTab = (0 <= flags.indexOf("T")); // -T タブ を `^I' で表示する

        this.hasAutoFlush = (0 > flags.indexOf("u")); // -u 出力をバッファリングしない

        if ((0 <= flags.indexOf("v"))) {
            StringBuilder sb = new StringBuilder("`cat");
            if (0 < flags.length()) {
                sb.append(" -").append(flags);
            }
            if (!this.inputFiles.isEmpty()) {
                sb.append(' ').append(this.inputFiles);
            }
            System.err.println(sb.append('`'));
        }
    }

    /**
     *
     */
    public static void main(String[] args) throws IOException {
        if ((0 != args.length) && args[0].startsWith("--help")) {
            System.out.println(USAGE);
        } else {
            UtilInterface me = new Cat(args, Device.openOutput("",
                    Io.STDOUT));
            if (me.hasInput()) {
                IoHelper.copyline(Io.STDIN, (Writer) me);
            }
            me.close();
        }
    }

    /**
     * このストリームを閉じる.
     */
    @Override
    public void close() throws IOException {
        try {
            String input = this.inputFiles.poll();
            while (null != input) {
                IoHelper.copyline(input, this);
                input = this.inputFiles.poll();
            }
            // super.flush();
        } finally {
            Io.close(super.out);
        }
    }

    /**
     * サブプロセスの終了値を返す.
     */
    @Override
    public int exitValue() {
        return 0;
    }

    /**
     *
     */
    @Override
    public boolean hasInput() {
        return this.inputFiles.isEmpty();
    }

    /**
     * このストリームに文字列を書き込む.
     * <p>
     * -n 行番号オプションを使用する場合は、行データは改行で終了すること
     */
    @Override
    public void write(@NotNull String x) throws IOException {
        String xx = (this.hasShowTab) ? RX_TAB.matcher(x)
                .replaceAll("^I") : x;
        if ((null == this.hasBlankLine) && (null == this.hasNumberLines)) {
            super.write(xx);
            if (this.hasAutoFlush) {
                super.flush();
            }
        } else {
            String[] arr = Helper.split(xx); // 入力データを配列に分割
            for (String line : arr) {
                writeln(line);
            }
        }
    }

    /**
     * 文字列に行区切り文字列を付加してストリームに書き込む.
     */
    private void writeln(String x) throws IOException {
        boolean isSpace = x.isEmpty(); // これは空行か?
        boolean hasSpace = (null != this.hasBlankLine)
                && this.hasBlankLine.getAndSet(isSpace);
        if (!isSpace || !hasSpace) { // 空行圧縮でない
            StringBuilder sb = new StringBuilder(x);
            if ((null != this.hasNumberLines) // 行番号を付け加える
                    && (!isSpace || !this.hasNumberNonblank)) { // 非空白行に番号を付ける
                Integer lno = this.hasNumberLines
                        .incrementAndGet();
                sb.insert(0, String.format("%6d\t", lno));
            }
            IoHelper.writeln(super.out, sb, this.hasAutoFlush);
        }
    }
}