/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import org.jetbrains.annotations.Nullable;
import plus.runtime.RegExp;

import java.io.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * [%helper%] ストリーム リーダ の実装 (排他制御は呼び出し元でおこなう).
 * <p>
 * The class to which this annotation is applied is not thread-safe.
 *
 * @author kunio himei.
 */
public final class TextReader extends BufferedReader {

    /**
     * CRLF 行入力区切り.
     */
    private static final String CRLF_SEPARATOR = "\r\n";

    /**
     * STRING BUILDER のデフォルト サイズ.
     */
    private static final int DEFAULT_STRING_BUILDERSIZE = 509;

    /**
     * Reader の EOF 定数.
     */
    private static final int EOF = -1;

    /**
     * プラットフォームに依存しない行セパレータ.
     */
    private static final char EOL = '\n';

    /**
     * 入力バッファで正規表現マッチをおこなうバッファサイズの初期値.
     */
    private static final int REGX_DEFALUT_CAPACITY = 1024 * 4; // 4K

    /**
     * REMIND: REGEXP: 入力バッファで正規表現マッチをおこなうバッファサイズの最大値.
     */
    private static final int REGX_MAX_CAPACITY = REGX_DEFALUT_CAPACITY * 256;
    /**
     * close は有効か?.
     */
    private final boolean isCloseable;
    /**
     * 入力バッファ.
     */
    private final StringBuilder line = new StringBuilder(
            DEFAULT_STRING_BUILDERSIZE);
    /**
     * 入力バッファで正規表現マッチをおこなうバッファサイズ (自動拡張).
     */
    private int capacity = REGX_DEFALUT_CAPACITY;
    /**
     * このｽﾄﾘｰﾑは閉じているか?.
     */
    private boolean isClosed;
    /**
     * RS にマッチし入力レコードセパレータとなった入力テキスト.
     */
    private String recTerm = "";

    /**
     * System Input Stream (disable close).
     *
     * @param input 入力ストリーム
     */
    public TextReader(final InputStream input) {
        super(new InputStreamReader(input));
        this.isCloseable = false;
    }

    /**
     * User's Input Reader (enable close).
     *
     * @param input 入力リーダ
     */
    public TextReader(final Reader input) {
        super(input, IoConstants.DEFAULT_BYTE_BUFFER_SIZE);
        this.isCloseable = true;
    }

    /**
     * 指定されたインデックス位置にある文字を返す.
     *
     * @param index 入力バッファのインデックス
     * @return 指定位置の文字 | EOF
     */
    private int charAt(final int index) throws IOException {
        while ((this.line.length() <= index) && !this.isClosed) {
            getBuffer(); // バッファを補充
        }
        return ((0 <= index) && (this.line.length() > index)) ? this.line
                .charAt(index) : EOF;
    }

    /**
     * 入力ストリームを閉じる.
     */
    @Override
    public void close() throws IOException {
        if (this.isCloseable) { // close は有効か?
            this.isClosed = true;
            super.close(); // EOF でストリームを自動的に閉じる
        }
    }

    /**
     * リーダが閉じていなければ行末までのデータをバッファへ読み込む.
     */
    private void getBuffer() throws IOException {
        if (!this.isClosed) {
            String lin = null;
            try {
                lin = super.readLine();
                if (null != lin) {
                    this.line.append(lin);
                    this.line.append(EOL); // end of line
                }
            } finally {
                if ((null == lin) || Thread.currentThread().isInterrupted()) {
                    close();
                }
            }
        }
    }

    /**
     * 入力レコード区切子 (RS) にマッチし入力レコードセパレータとなった入力テキストを返す.
     *
     * @return 入力レコード区切りとなった入力テキスト (REMIND: CRLF の場合は NL を返す)
     */
    public String getRT() {
        return this.recTerm;
    }

    /**
     * 次の要素の有無を返す.
     */
    private boolean hasNext() throws IOException {
        if (!this.isClosed && (0 == this.line.length())) {
            getBuffer();
        }
        return 0 < this.line.length();
    }

    /**
     * 指定された部分文字列が出現する位置のインデックスを返す.
     *
     * @param str   検索対象の部分文字列
     * @param index 入力バッファのインデックス
     */
    private int indexOf(final String str, final int index) throws IOException {
        int ix;
        while ((0 > (ix = this.line.indexOf(str, index))) && !this.isClosed) {
            getBuffer(); // バッファを補充
        }
        return ix;
    }

    /**
     * 入力レコード区切子 (RS) で区切った、1 行のテキストを読み込む.
     *
     * @param rs 入力レコードセパレータ (既定値は改行)
     * @return 読み込んだ文字列 (行の区切り文字は含まない、ストリームの終端では null)
     */
    @Nullable
    public String readLine(final String rs) throws IOException {
        final boolean isCRLFRS = CRLF_SEPARATOR.equals(rs); // CRLF区切り
        final String rsep = (isCRLFRS) ? Character.toString(EOL) : rs;
        final boolean isNewlineRS = "\n".equals(rsep); // 改行区切り
        if (hasNext()) { // バッファを補充
            final boolean isEmptyRS = rsep.isEmpty(); // 空の区切り
            int start = 0, end, eol, last;
            if (isCRLFRS || isNewlineRS || isEmptyRS) {
                if (isEmptyRS) { // '' 連続改行区切り
                    start = skipNL(start); // 先行する改行をスキップ
                    end = indexOf("\n\n", start);
                    if (0 <= end) {
                        eol = last = skipNL(end + 2); // 残りの改行をスキップ
                    } else { // EOF の場合
                        eol = last = this.line.length();
                        while ((start < eol) && ('\n' == charAt(eol - 1))) {
                            eol--;
                        }
                        end = eol;
                    }
                } else { // '\n' '\r\n'
                    end = indexOf(rsep, start);
                    if (0 <= end) {
                        eol = last = end + rsep.length();
                    } else {
                        eol = last = end = this.line.length();
                    }
                }
            } else { // 正規表現区切り ('.'は行末記号を含む任意の文字にマッチする)
                final Pattern pattern = RegExp.compile(rsep);
                do {
                    while ((this.line.length() < this.capacity)
                            && !this.isClosed) {
                        getBuffer(); // バッファを補充
                    }
                    end = eol = last = this.line.length();
                    final Matcher re = pattern.matcher(this.line);
                    if (re.find()) {
                        end = re.start();
                        eol = last = re.end();
                        // REMIND: BUGパッチ: see:'gawk/test/rsstart1.awk'
                        if ('^' == rsep.charAt(0)) { // 正規表現が先頭マッチ/^/なら
                            while (!this.isClosed) { // 現在のファイルを全て読み込む
                                getBuffer();
                            }
                        }
                        break;
                    } else if (REGX_MAX_CAPACITY > this.capacity) {
                        break;
                    }
                    this.capacity <<= 1;
                    // System.err.println("capacity: " + this.capacity);
                } while (!this.isClosed);
            }
            this.recTerm = this.line.substring(end, eol);
            // this.builtIns.RT_(this.rt); // レコード区切子 RT の設定
            if ((0 < this.line.length()) && (start < last)) {
                final String ws = this.line.substring(start, end);
                this.line.delete(0, last);
                return ws;
            }
            this.line.setLength(0); // '' 連続改行区切りで有効データなしの場合
        }
        return null;
    }

    /**
     * 入力バッファの先行する改行をスキップ.
     *
     * @param index スキャン開始位置
     * @return スキャン終了位置 ('\n'の位置 +1)
     */
    private int skipNL(final int index) throws IOException {
        int i = index;
        while ('\n' == charAt(i)) { // 必要ならバッファを補充
            i++; // 改行をスキップ
        }
        return i;
    }
}