/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus;

import org.jetbrains.annotations.Nullable;
import plus.io.Io;
import plus.io.NanoTools;
import plus.runtime.BuiltInVar;
import plus.runtime.RunHelper;
import plus.util.Escape;
import plus.util.NumHelper;

import java.io.Closeable;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

// ------------------------------------------------------------------------
// Built-in - I/O.
// ------------------------------------------------------------------------
public class BiIO extends BiBase {
    //* ------------------------------------------------------------------------
    //* Environ.
    //* ------------------------------------------------------------------------
    private static final Integer C_INTEGER1 = 1;

    private static final Io ioPool = new Io();

    //* ------------------------------------------------------------------------
    //* I/O Functions.
    //* ------------------------------------------------------------------------
    public static void close(Object... io) throws IOException {
        if (0 < io.length) {
            for (Object x : io)
                if (x instanceof Closeable)
                    Io.close((Closeable) x);
                else
                    ioPool.close(x.toString());
        } else {
            ioPool.closeAll();
        }
    }

    @SuppressWarnings("unused")
    public static void fflush(Object... io) throws IOException {
        ioPool.fflush(io);
    }

    private static final NanoTools nanoTools = new NanoTools();

    @SuppressWarnings("unused")
    public static int set(Object... x) {
        return nanoTools.set(x);
    }

    @SuppressWarnings("unused")
    public static int copy(String in, String out, Object... x) {
        return nanoTools.copy(in, out, x);
    }

    @SuppressWarnings("unused")
    public static int move(String in, String out, Object... x) {
        return nanoTools.move(in, out, x);
    }

    @SuppressWarnings("unused")
    public static int remove(String input, Object... x) {
        return nanoTools.remove(input, x);
    }

    @SuppressWarnings("unused")
    public static int tree(String path, Object... x) {
        return nanoTools.tree(path, x);
    }

    @SuppressWarnings("unused")
    public static int ls(String path, Object... x) {
        return nanoTools.ls(path, x);
    }

    /**
     * ARGVからファイル名を取得する.(EOF の場合は、""を返す)
     */
    public static Object nextfile() {
        return ARGV.next();
    }

    /**
     * テキスト行入力 (コマンド実装).
     * NR、FNR を設定する (ARGV作成時、FILENAMEは空)
     *
     * @param rid   リダイレクトID.
     * @param file  ファイル名.
     * @param index $ のインデックス.
     */
    @Nullable
    public static String getline(String rid, Object file, Object... index) throws IOException {
        Object filename;
        if (exists(file)) {
            filename = file; // 呼び出しパラメータ
        } else if (BuiltInVar.ARGIND.intValue() < BuiltInVar.ARGC.intValue()) {
            filename = BuiltInVar.FILENAME.toString(); // 現在のファイル
            if (isNil(filename)) filename = nextfile(); // 次のファイル
        } else {
            return null;
        }
        if (isNil(filename)) filename = "-"; // パラメータが省略されたとき
        String name = filename.toString();
        boolean noStream = ioPool.noStream(name); // FNRリセット要求

        String line = RunHelper.getlineImpl(ioPool.getReader(rid, name));
        if (null != line) { // POSTIT len=0 は有効なため exists()は、不可.
            if (exists(index)) {
                int ix = NumHelper.intValue(index[0]); // $[index]
                BuiltInVar.$.putAt(ix, line);
            }
            if (!rid.contains("|") && isNil(file)) {
                // NOTE 'The AWK 2-5 getline関数'参照.
                // '|', file指定なしの場合.(getline or getlone var)
                // mainループから呼ばれたときは、上記条件により、ここには来ない.
                if (noStream) BuiltInVar.FNR.put(0); // FNRをリセット
                BuiltInVar.NR.calculate("+", C_INTEGER1); // 入力したレコード数の合計
                BuiltInVar.FNR.calculate("+", C_INTEGER1); // 現在のファイルから入力したレコード数
            }
            return line;
        }
        close(name); // 自動クローズ
        return null;
    }

    public static int print(String rid, String file, Object... args) {
        String ors = BuiltInVar.ORS.toString();
        String x = (0 == args.length) ? BuiltInVar.$.getAt(0).toString() :
                RunHelper.concat(BuiltInVar.OFS.toString(), BuiltInVar.OFMT.toString(), args);
//        x = RunHelper.replaceNL2ORS(x, ors); // 改行の置き換え
        try {
            return ioPool.print(rid, file,
                    Escape.outputFilter(x) + ors);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Groovyとバッティングするため、エイリアス'_'を付ける.
     */
    public static int _printf(String rid, String name, Object... args) throws IOException {
        String x = _sprintf(args);
//        x = RunHelper.replaceNL2ORS(x, BuiltInVar.ORS.toString()); // 改行の置き換え
        return ioPool.print(rid, name, x);
    }

    public static String _sprintf(Object... args) {
        return Escape.outputFilter(RunHelper.sprintf(args)); // フィルタ #0
    }

    /**
     * エスケープシーケンスを無視(無効化)する - for T9Escape.awk.
     */
    @SuppressWarnings("unused")
    public static Object disableEscape(Object x) {
        if (x instanceof CharSequence)
            return x.toString().replace("\\", "\\\\");
        return x;
    }

    /*
     * URLエンコード (日本語ドメインは、未対応) - for Android.
     *
     * reserved　= gen-delims / sub-delims
     * gen-delims = ":" / "/" / "?" / "#" / "[" / "]" / "@"
     * sub-delims = "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / ";" / "="
     * unreserved = ALPHA / DIGIT / "-" / "." / "_" / "~"
     */
    @SuppressWarnings("unused")
    public static String encodeURL(String url) {
        try {
            if (url.matches(".*%[0-9A-Fa-f]{2}.*"))
                return url;
            String domain = url.replaceFirst("/[^/]*$", "");
            String file = url.replaceFirst("^.+/", "");
            return domain + '/' + (URLEncoder.encode(file, StandardCharsets.UTF_8)
                    .replace("+", "%20")); // " " REMIND

        } catch (final Throwable e) { // UnsupportedEncodingException
            throw new IllegalStateException(e);
        }
    }

    /*
     * NOTE 'ORS'の初期値にプラットフォーム依存の改行コードをセットしているが、
     * - Windows10の、メモ帳、Type、Fcで、UTF-8、'LF(\n)'をサポートしているため.
     * - デファクトスタンダードに合わせる.
     */
}