/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import java.io.*;

/**
 * [%helper%] I/O Helper.
 *
 * @author kunio himei.
 */
public final class IoHelper {

    /**
     * Don't let anyone instantiate this class.
     */
    private IoHelper() {
        super();
    }

    /**
     * 入力ファイルを行またはバッファ単位に複写する.
     *
     * @param input  入力ファイル名
     * @param output 出力ライタ
     */
    public static void copy(final String input, final Writer output)
            throws IOException {
        if ((output instanceof PrintWriter) || Io.STDIN.equals(input)
                || "-".equals(input)) {
            copyLiLne(input, output);
        } else {
            final char[] buf = new char[IoConstants.DEFAULT_CHAR_BUFFER_SIZE];
            final Reader in = Device.openInput(input);
            for (int len; 0 <= (len = in.read(buf)); ) {
                if (0 < len) {
                    output.write(buf, 0, len);
                }
            }
            output.flush();
            Io.close(in);
        }
    }

    /**
     * 入力ファイルを行単位に出力する.
     *
     * @param input  入力ファイル名
     * @param output 出力ライタ
     */
    public static void copyLiLne(final String input, final Writer output)
            throws IOException {
        final BufferedReader in = new BufferedReader(Device.openInput(input),
                IoConstants.DEFAULT_BYTE_BUFFER_SIZE);
        String x = in.readLine(); // System.in -> System.out 時の対応
        while (null != x) {
            writeln(output, x, true);
            x = in.readLine();
        }
        Io.close(in);
    }

    /**
     * 文字列に行区切り文字列を付加してストリームに書き込む.
     *
     * @param out 出力ストリーム
     * @param x   出力文字列
     */
    public static void writeln(final Writer out, final CharSequence x)
            throws IOException {
        writeln(out, x, out instanceof PrintWriter); // PrintWriter は flush
    }

    /**
     * 文字列に行区切り文字列を付加してストリームに書き込む.
     *
     * @param out       出力ストリーム
     * @param x         出力文字列
     * @param autoFlush 出力バッファをフラッシュする
     */
    public static void writeln(final Writer out, final CharSequence x,
                               final boolean autoFlush) throws IOException {
        final StringBuilder sb = (x instanceof StringBuilder)
                ? (StringBuilder) x : new StringBuilder(x);
        out.write(sb.append(IoConstants.PLATFORM_LINE_SEPARATOR).toString());
        if (autoFlush) {
            out.flush();
        }
    }
}