/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT TANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.HashMap;
import java.util.Map;

/**
 * Parser - タイプ名(sType) - フラグ(Flags)対応テーブル.
 *
 * @author kunio himei.
 */
public final class TypeForGroovy {

    private static final String T1NUMBER = "Number"; // REMIND sTypeテーブル ☆
    private static final String T2DOUBLE = "Double";
    private static final String T2FLOAT = "Float";
    private static final String T3INTEGER = "Integer";
    private static final String T3LONG = "Long";
    private static final String TAMAP = "Map";
    private static final String TALIST = "List";
    private static final String TARRAY = "AtomicArray";
    private static final String TATOMAP = "AtomicMap";
    private static final String TBOOLEAN = "Boolean";
    private static final String TOBJECT = "Object";
    private static final String TSTRING = "String";
    private static final String TVOID = "Void";
    private static final String TXFUNCTION = "Function";
    private static final String TXREFERENCE = "Reference";
    private static final String TZCLASS = "Class"; // 未使用
    private static final String TZSTRNUM = "StrNum"; // 未使用

    /**
     * Type Name to Type ID.
     */
    private static final Map<String, Integer> VAR_TYPES = new HashMap<>();

    static {
        VAR_TYPES.put(T1NUMBER, Flags.T01NUM);
        VAR_TYPES.put(T2DOUBLE, Flags.T06DBL);
        VAR_TYPES.put(T2FLOAT, Flags.T05FLOAT);
        VAR_TYPES.put(T3INTEGER, Flags.T03INT);
        VAR_TYPES.put(T3LONG, Flags.T04LONG);
        VAR_TYPES.put(TAMAP, Flags.T16ARRAY | Flags.T11ANY);
        VAR_TYPES.put(TALIST, Flags.T16ARRAY | Flags.T11ANY);
        VAR_TYPES.put(TARRAY, Flags.T16ARRAY | Flags.T11ANY);
        VAR_TYPES.put(TATOMAP, Flags.T16ARRAY | Flags.T11ANY);
        VAR_TYPES.put(TBOOLEAN, Flags.T02BOOL);
        VAR_TYPES.put(TOBJECT, Flags.T13OBJ);
        VAR_TYPES.put(TSTRING, Flags.T07STR);
        VAR_TYPES.put(TVOID, Flags.T00VOID);
        VAR_TYPES.put(TXFUNCTION, Flags.T17FVAL);
        VAR_TYPES.put(TXREFERENCE, Flags.T18REF);
        VAR_TYPES.put(TZCLASS, Flags.T10CLAZ); // 未使用
        VAR_TYPES.put(TZSTRNUM, Flags.T09STNUM); // 未使用

        VAR_TYPES.put("boolean", Flags.T02BOOL);
        VAR_TYPES.put("double", Flags.T06DBL);
        VAR_TYPES.put("float", Flags.T05FLOAT);
        VAR_TYPES.put("int", Flags.T03INT);
        VAR_TYPES.put("long", Flags.T04LONG);
    }

    /**
     * Don't let anyone instantiate this class.
     */
    private TypeForGroovy() {
        super();
    }

    /**
     * タイプ名に対応する属性を返す.
     */
    public static int getVarType(String k) {
        return VAR_TYPES.get(k);
    }

    /**
     * タイプ名が存在するかどうかを返す.
     */
    public static boolean hasVarType(String k) {
        return VAR_TYPES.containsKey(k);
    }
}