/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.eval;

import plus.concurrent.AtomicMap;
import plus.concurrent.AtomicNumber;
import plus.exception.ReturnException;
import plus.lex.Flags;
import plus.lex.Node;
import plus.reflect.Listener;
import plus.runtime.BuiltInVar;
import plus.util.Debug;
import plus.variable.Frame;

import java.util.Arrays;
import java.util.TreeSet;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Evaluate - Function.
 *
 * @author kunio himei.
 */
abstract class EvalFunc extends EvalVar {

    //* used in Concurrent Function Value.
    private static final ReentrantLock SYNC = // SYNC static.
            new ReentrantLock();

    /**
     * ユーザ定義関数,呼出しパラメータの設定.
     */
    private int setUserCallParam(Node.Func fn, Object[] args, Object[] vals) {
        assert (null != args) && (null != vals);
        int argslen = args.length;
        if ((1 <= argslen) && (0 == fn.argsLength)) {
            throw new IllegalArgumentException(fn.name + Debug.objectOf(args));
        }
        int idx = 0, k = 0;
        while (argslen > k) {
            String id = fn.parm[idx].name;
            int typ = fn.parm[idx].nType;
            if (Flags.isVarArgs(typ)) { // 可変長引数
                Object[] arr = new Object[argslen - k];
                for (int i = 0; argslen > k; i++) { // 実引数を全て変換
                    Object w;
                    if (Flags.isReference(typ)) { // Reference Variable
                        assert args[k] instanceof Node.YyVariable;
                        Node.YyVariable x = (Node.YyVariable) args[k];
                        w = new Reference(x.name, mkIndex(x.index));
                    } else if (!Flags.isArray(typ)) {
                        w = vals[k]; // 呼出元の単純変数
                    } else {
                        assert args[k] instanceof Node.NAME;
                        Node.NAME x = (Node.NAME) args[k];
                        w = _VARIABLES._allocArray(x.name); // 呼出元の配列
                    }
                    arr[i] = w;
                    k++;
                }
                Frame._putLocalVar(id, arr);
            } else if (!Flags.isArray(typ)) { // 単純変数
                Frame._putLocalVar(id, vals[k]);
            } else { // 呼出元の配列を参照
                Object x = args[k];
                if (x instanceof Node.NAME) {
                    _VARIABLES._allocLocalArray(id, ((Node.NAME) x).name);
                } else if (!(x instanceof AtomicMap) && //
                        !(x instanceof Object[])) {
                    throw new IllegalStateException("unmatch: "
                            + Debug.objectOf(x));
                }
            }
            idx++;
            k++;
        }
        return idx;
    }

    /**
     * ユーザ定義関数,ローカル変数の初期化.
     */
    private void setUserLocalVars(Node.Func fn, int ix) {
        int len = fn.parm.length; // 関数定義のパラメータ長
        for (int i = ix; len > i; i++) {
            String id = fn.parm[i].name;
            int typ = fn.parm[i].nType;
            if (Flags.isVarArgs(typ)) { // 可変長引数
                Frame._putLocalVar(id, EMPTY_ARRAY);
            } else if (Flags.isArray(typ)) { // ローカル配列を作成
                _VARIABLES._allocLocalArray(id, null);
            } else { // 単純変数
                Object w;
                if (Flags.isInteger(typ) || Flags.isNumber(typ)) {
                    w = new AtomicNumber();
                } else if (Flags.isString(typ)) {
                    w = ""; // 文字列
                } else {
                    w = NIL; // POSTIT 0
                }
                Frame._putLocalVar(id, w);
            }
        }
    }

    /**
     * 関数呼出し.
     */
    Object callFunction(String name, Object[] args) {
        return sTree.func.containsKey(name) ?
                userFunction(name, args, evalArgs(args)) :
                builtinFunction(name, args, evalArgs(args));
    }

    /**
     * ユーザ定義関数呼出し - User-defined function call.
     */
    Object userFunction(String name, Object[] args, Object[] vals) {
        Frame._startLocal();
        Node.Func fn = sTree.func.get(name);
        setUserLocalVars(fn, setUserCallParam(fn, args, vals));
        Object rs;
        try {
            rs = eval(fn.stmt);
        } catch (ReturnException ex) {
            rs = ex.value();
        }
        Frame._endLocal();
        return rs;
    }

    /**
     * 組み込み関数呼出し - Built-in function call.
     */
    private Object builtinFunction(String x, Object[] args, Object[] vals) {
        if (Node.INDEX_OP.equals(x)) { // インデックス
            return (0 == vals.length) ? null : _index(vals);
        } else if (Node.CONCAT_OP.equals(x)) { // 文字列結合
            return _cat(vals);

        } else if ("double".equals(x)) {
            return _double(vals[0]);
        } else if ("float".equals(x)) {
            return _float(vals[0]);
        } else if ("long".equals(x)) {
            return _long(vals[0]);
        } else if ("int".equals(x)) {
            return _int(vals[0]);

        } else if ("gsub".equals(x)) { // gsub(r, s [, t])
            vals = preGsub(args, vals);
            int cc = gsub(vals);
            if (2 < vals.length) postGsub((Node.YyVariable) args[2]);
            return cc;
        } else if ("sub".equals(x)) {
            vals = preGsub(args, vals);
            int cc = sub(vals);
            if (2 < vals.length) postGsub((Node.YyVariable) args[2]);
            return cc;

        } else if ("assertEquals".equals(x)) {
            return assertEquals(scriptLineNumber(), vals);

        } else if ("assertTrue".equals(x)) {
            return assertTrue(scriptLineNumber(), vals);

        } else {
            System.err.println(new TreeSet<>(sTree.func.keySet()));
            throw new IllegalArgumentException("'" + x + "'");
        }
    }

    /**
     * gsub,sub(r,s(,t)) の前処理.
     */
    private Object[] preGsub(Object[] args, Object[] vals) {
        if (2 < args.length) {
            if (args[2] instanceof Node.YyVariable) {
                Node.YyVariable x = (Node.YyVariable) args[2];
                // Targetが、'$0'の時は、呼び出しパラメータから削除(2重代入防止).
                if ("$[0]".equals(x.toString())) // POSTIT この手があった！
                    return Arrays.copyOf(vals, 2);
            } else
                throw new IllegalArgumentException("Target must be a variable. " +
                        args[2] + " " + args[2].getClass());
        }
        return vals;
    }

    //* gsub,sub(r,s(,t)) の後処理、ターゲット変数に値を返す.
    private void postGsub(Node.YyVariable e) {
        String x = BuiltInVar.RESULT.toString(); // 置換結果を取得
        _putValue(e.name, mkIndex(e.index), "=", x);
    }

    /**
     * Reference Variable - メソドが、引数に、値を返すために使用する.
     */
    protected static class Reference extends Listener {

        private final String name; //  このオブジェクトの識別名
        private final String index; // のオブジェクトのインデックス
        private Object value; //  このオブジェクトの値

        Reference(String name, String index) {
            this.name = name;
            this.index = index;
            this.value = _getRefValue(name, index);
        }

        //* このオブジェクトの値を返す.
        @Override
        public Object apply(Object... x) {
            return this.value;
        }

        //* このオブジェクトの値を更新する.
        @Override
        public Object update(Object x) {
            this.value = _putRefValue(this.name, this.index, x);
            return this.value;
        }

        //* このオブジェクトの文字列表現を返す.
        @Override
        public String toString() {
            return name + "[" + index + "] " + value;
        }
    }

    /**
     * Function Value - userFunction を呼ぶため、 sttatic 指定は、不可.
     */
    protected class Function extends Listener {

        private final Object obj; // このオブジェクト
        private final String name; // このオブジェクトの識別名
        private final Object[] args; // このオブジェクトの引数
        private final AtomicMap closure; // 親のスコープ

        Function(Object obj, String name, Object[] args) {
            this.obj = obj;
            this.name = name;
            this.args = args.clone();
            this.closure = Frame._cloneLocalContext();
        }

        //* このオブジェクトに引数を適用する.
        // パラメータが省略された場合は、初期化時に指定された値を使用.
        @Override
        public Object apply(Object... a) {
            Object[] arr = (0 == a.length) ? this.args : a;
            Frame._startNewContext(this.closure);
            return (null == this.obj) ?
                    userFunction(this.name, arr, arr) :
                    _invoke(this.obj, this.name, arr);
        }

        //* インタフェース Callable<の実装.
        @Override
        public Object call() {
            return apply(this.args);
        }

        //* このオブジェクトの simple name を返す.
        @Override
        public String getName() {
            return this.name.replaceFirst("^.*[$]", "");
        }


        //* このオブジェクトの full name を返す.
        @Override
        public String toString() {
            return this.name;
        }
    }

    /**
     * Concurrent Function Value.
     */
    protected class ConcurrentFunction extends Function {

        ConcurrentFunction(Object obj, String name, Object[] args) { //
            super(obj, name, args);
        }

        //* 同期呼び出し.
        @Override
        public Object apply(Object... a) {
            SYNC.lock(); // SYNC. ConcurrentFunction.
            try {
                return super.apply(a);
            } finally {
                SYNC.unlock();
            }
        }
    }
}