/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.util;

/**
 * Escape sequence Helper.
 *
 * @author kunio himei.
 */
public final class Escape {

    //* 8 進数を表現する bit 幅 (bit).
    private static final int C_OCT_BITWIDTH = 3;

    //* 16 進数を表現する bit 幅 (bit).
    private static final int C_HEX_BITWIDTH = 4;

    //* 文字列を 8 進数 に変換する場合の最大文字列長 (3bitx3=9bit).
    private static final int C_OCT_STRING_MAXLENGTH = 3;

    //* 文字列を 16 進数 に変換する場合の最大文字列長 (4bitx2=8bit).
    private static final int C_HEX_STRING_MAXLENGTH = 2;

    //* 文字列を UNICODE に変換する場合の最大文字列長 (4bitx4=16bit).
    private static final int C_UNICODE_STRING_MAXLENGTH = 4;

    //* Octal decimal 8 進数.
    private static final int TO_OCT = 8;

    //* Hex decimal 16 進数.
    private static final int TO_HEX = 16;

    /**
     * スクリプト"文字列"のエスケープシーケンスを内部コードに変換して返す.
     *
     * @param escape エスケープ対象文字(複数指定可).
     */
    public static String encodeString(String x, String escape) {
        int len = x.length();
        StringBuilder sb = new StringBuilder(len + 19);
        String ignore = "/'\""; // <- ここから、エスケープ対象文字を引く ☆
        if (!escape.isEmpty())
            ignore = ignore.replaceAll("[" + escape + "]+", "");
        for (int p = 0; len > p; ) {
            char c = x.charAt(p++);
            switch (c) {
                case '\\' -> {
                    char next = x.charAt(p++);
                    if (0 > ignore.indexOf(next))
                        sb.append(c); // テーブルになければ、エスケープ ☆
                    sb.append(next); // あれば、エスケープを解除.
                    if (('\\' == next) && (0 <= escape.indexOf('\\')))
                        sb.append("\\\\"); // 指示があれば、追加.
                }
                case '/', '"', '\'' -> {
                    if (0 <= escape.indexOf(c)) sb.append('\\'); // エスケープ.
                    sb.append(c);
                }
                case '\n' -> sb.append("\\n");
                case '\r' -> {
                    if (x.length() > p && '\n' == x.charAt(p)) // CRLF
                        break; // for Windows.
                    sb.append("\\r"); // for Old Mac.
                }
                case '\t' -> sb.append("\\t");
                default -> sb.append(c);
            }
        }
//        System.err.println("encode: " + escape + '\t' + x + "\n\t\t" + sb);
        return sb.toString();
    }

    /**
     * 出力フィルタ #0 - エスケープシーケンスを外部コードに変換して返す.
     * <p>
     * \a alert (bell) <br>
     * \v vertical tab <br>
     * \b backspace <br>
     * \f form feed <br>
     * \n new line <br>
     * \r carriage return <br>
     * \t horizontal tab <br>
     * \\ backslash <br>
     * \\uFEFF UNICODE <br>
     * \xFF Hexa decimal <br>
     * \0777 Octal decimal <br>
     */
    public static String outputFilter(String x) {
        int len = x.length();
        StringBuilder sb = new StringBuilder(len + 19);
        for (int p = 0; len > p; ) {
            char c = x.charAt(p++);
            if (('\\' == c) && (len > p)) {
                //* Fallthrough //
                switch (c = x.charAt(p++)) {
                    case '\\': // このエスケープは、削除.
                    case '\'':
                    case '"':
                        break;
                    case 'a':
                        c = '\u0007'; // Bell (何も起きない)
                        break;
                    case 'v':
                        c = '\u000B'; // VT (♂)　
                        break;
                    case 'b':
                        c = '\b';
                        break;
                    case 'f':
                        c = '\f'; // form feed (♀)
                        break;
                    case 'n':
                        c = '\n';
                        break;
                    case 'r':
                        c = '\r';
                        break;
                    case 't':
                        c = '\t';
                        break;
                    case 'U':
                    case 'u':
                        if (len > p) { // unicode
                            c = 0;
                            int k;
                            for (int i = C_UNICODE_STRING_MAXLENGTH - 1; (0 <= i)
                                    && (len > p)
                                    && (0 <= (k = Character.digit(x.charAt(p),
                                    TO_HEX))); i--, p++) {
                                c = (char) ((c << C_HEX_BITWIDTH) | k);
                            }
                        }
                        break;
                    case 'X':
                    case 'x':
                        if (len > p) { // hexadecimal
                            c = 0;
                            int k;
                            for (int i = C_HEX_STRING_MAXLENGTH - 1; (0 <= i)
                                    && (len > p)
                                    && (0 <= (k = Character.digit(x.charAt(p),
                                    TO_HEX))); i--, p++) {
                                c = (char) ((c << C_HEX_BITWIDTH) | k);
                            }
                        }
                        break;
                    default:
                        int k; // Octal
                        if (0 <= (k = Character.digit(c, TO_OCT))) {
                            c = (char) k;
                            for (int i = C_OCT_STRING_MAXLENGTH - 2; (0 <= i)
                                    && (len > p)
                                    && (0 <= (k = Character.digit(x.charAt(p),
                                    TO_OCT))); i--, p++) {
                                c = (char) ((c << C_OCT_BITWIDTH) | k);
                            }
                        } else {
                            sb.append('\\'); // REMIND 未定義のエスケープ.
                        }
                }
            }
            sb.append(c);
        }
        return sb.toString();
    }

    /*
     * 入力フィルタ - 未定義エスケープを'\'エスケープして返す.
     */
/*
    public static String getlineFilter(String x) {
        int len = x.length();
        StringBuilder sb = new StringBuilder(len + 19);
        for (int p = 0; len > p; ) {
            int c = x.charAt(p++);
            if (('\\' == c) && (len > p)) {
                switch (c = x.charAt(p++)) {
                    //* Fallthrough //
                    case '\\': // このエスケープは、そのまま.
                    case '\'':
                    case '"':
//                        break;
                    case 'a':
                    case 'v':
                    case 'b':
                    case 'f':
                    case 'n':
                    case 'r':
                    case 't':
                    case 'U':
                    case 'u':
                    case 'X':
                    case 'x':
                        sb.append('\\');
                        break;
                    default:
                        if (0 <= Character.digit(c, TO_OCT)) { // Octal
                            sb.append('\\');
                        } else {
                            sb.append("\\\\"); // REMIND 未定義のエスケープ.
                        }
                }
            }
            sb.append((char) c);
        }
        return sb.toString();
    }
*/

    /**
     * NOPフィルタ - 何もしない.
     */
    public static String nopFilter(String x) {
        return x;
    }
}