/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import plus.BiAccessor;
import plus.reflect.Reflection;
import plus.runtime.BuiltInVar;
import plus.util.NumHelper;

import java.util.regex.Pattern;

/**
 * Atomic コマンド行引数配列 (ARGV).
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author Kunio Himei.
 */
public final class Argument extends FutureMap<Integer, Object> {

    public static final String USAGE =
            "AWK~Plus [Script] [Options] [Args]\n" +
                    "Options:\n" +
                    "  -F val\tフィールドセパレータ、初期値は空白とタブ\n" +
                    "  -v var = val\t実行前に変数値を代入\n" +
                    "  var = val\tコマンドラインで指定したファイルオープン時に代入\n";
    private static final long serialVersionUID = 1L;
    /**
     * コマンド行引数オプションの解析テーブル.
     * <p>
     * 指定されたオプションと先行文字一致でマッチする
     */
    private static final String[][] ARGV_OPTS = {
            // システムプロパティ
            {"-D", "D"},
            // 欄区切子変数設定
            {"-F", "F"},
            {"--field-separator=", "F"}, //
            // 変数値設定
            {"-v", "v"},
            {"--assign=", "v"}};

    //* 長さ 1 の空オブジェクト配列.
    private static final Object[] ONE_OBJECT_ARRAY = new Object[1];
    //* 変数値設定(var=val)を判定する正規表現.
    private static final Pattern
            rxASSIGN_PAT = Pattern.compile("^[A-Za-z_$][\\w$]+[\\s]*=");
    //* 呼び出し元の変数スコープ(スクリプトスコープ).
    private final Object owner;

    /**
     * コマンド行引数配列の構築.
     *
     * @param args  ARGV
     * @param owner スクリプトのスコープ.(インタプリタはnull)
     */
    public Argument(String[] args, Object owner) {
        super(args.length);
        this.owner = owner;
        parse(args);
        BuiltInVar.ARGC.put(super.size()); // コマンド行引数の数
        BuiltInVar.ARGIND.put(0); // コマンド行引数の開始位置
        BuiltInVar.FILENAME.put(""); // ファイル名は空白
    }

    //* var='val'を分解する.
    private static String[] splitVar(String var) {
        int i = var.indexOf('='); // 最初の'='で分割
        String name = var.substring(0, i).trim(); // var
        String value = var.substring(i + 1); // 'val'
        return new String[]{name, value};
    }

    //* システムプロパティーを設定.
    private static void setProperty(String var) {
        String[] a = splitVar(var);
        System.setProperty(a[0], a[1]);
    }

    /**
     * コマンド行引数配列(ARGV)で、指定された変数に値を設定する.
     */
    private void invoke(String var) {
        String[] a = splitVar(var);
        String name = a[0];
        Object value = NumHelper.toStringNumber(a[1]);
        if ("FS".equals(name) && a[1].isEmpty())
            value = " "; // POSTIT 'FS=' 値が省略された場合のパッチ.
        System.err.println("`invoke: " + name + "=" +
                ((value instanceof String) ? "'" + value + "'" : value));

        BuiltInVar x = BuiltInVar.forName(name); // 組込変数
        if (null != x) {
//            System.err.println("`builtin " + var);
            x.put(value);
        } else if (Reflection.has(this.owner, name, ONE_OBJECT_ARRAY)) {
            // スクリプトスコープに、セッタで設定 - for Groovy.
            name = "set" + name.substring(0, 1).toUpperCase() + name.substring(1);
//            System.err.println("`invoke " + name + "(" + value + ")");
            Reflection.invoke(this.owner, name, new Object[]{value});
        } else {
//            System.err.println("`global " + var);
            BiAccessor._GLOBALS(name, value); // グローバル変数
        }
    }

    /**
     * コマンド行引数配列(ARGV)に追加(StringNumber)する.
     */
    private void append(String x) {
        super.putAt(super.size(), NumHelper.toStringNumber(x));
    }

    /**
     * コマンド行引数配列の構築.
     * NOTE 囲み文字 " ' は、Javaが事前に削除する.
     * - 空白を含む文字列は、" ' で囲む.
     * - 囲まれていない文字列に、" ' を使用する場合は、\" \' でエスケープする.
     * - '='の前後空白は、不可.
     */
    private void parse(String[] args) {
//        System.err.println("ARGS: " + Arrays.toString(args));
        append("-"); // ARGV[0] はダミー
        boolean hasOptions = true;
        int len = args.length;
        for (int idx = 0; len > idx; idx++) {
            String arg = args[idx];
            if (arg.isEmpty() || "''".equals(arg) || "\"\"".equals(arg)) {
                append(""); // 空のパラメータも設定する
            } else if ('-' == arg.charAt(0)) { // '-'で始まる
                StringBuilder sb = new StringBuilder(arg);
                if (hasOptions) {
                    if ("--".equals(arg)) {
                        hasOptions = false;
                        continue; // オプション解析終了
                    }
                    for (String[] argopt : ARGV_OPTS) {
                        String opt = argopt[0];
                        int p = 1;
                        while ((p < opt.length()) && (p < sb.length())
                                && (opt.charAt(p) == sb.charAt(p))) {
                            p++;
                        }
                        if (('-' != opt.charAt(p - 1))) { // 1文字以上一致
                            char optchr = argopt[1].charAt(0);
                            while ((p < sb.length())
                                    && Character.isWhitespace(sb.charAt(p))) {
                                p++; // 先頭空白を削除
                            }
                            if ((sb.length() <= p) && ((idx + 1) < args.length)
                                    && ('-' != args[idx + 1].charAt(0))) {
                                sb.append(args[idx + 1]); // 泣別れオプションを結合
                                idx++;
                            }
//                            String val = Escape.decodeTerminal( // POSTIT
//                                    sb.substring(p));
                            String val = sb.substring(p);
                            if ('F' == optchr) {
                                invoke("FS=" + val);
                            } else if ('v' == optchr) { // 変数値設定 VAR=value
                                invoke(val);
                            } else if ('D' == optchr) { // システムプロパティ
                                setProperty(val);
                            }
                            sb.setLength(0);
                            break;
                        }
                    }
                } // end of Option analysis
                if (0 < sb.length()) {
                    append(arg); // file name
                }
            } else if (rxASSIGN_PAT.matcher(arg).find()) { // VAR=value
                append(arg); // ここでは、そのまま出力する
            } else {
                append(arg); // file name
            }
        }
    }

    /**
     * ARGVからファイル名を取得する.(EOF の場合は、""を返す)
     */
    public Object next() {
        int argc = BuiltInVar.ARGC.intValue(); // 要素数は ARGCを使用する
        int i = BuiltInVar.ARGIND.intValue();
        Object x = "";
        String file;
        while (argc > ++i) {
            x = super.getAt(i);
            file = x.toString();
            if (!file.isEmpty()) { // 空は読み飛ばす
                if (rxASSIGN_PAT.matcher(file).find()) { // VAR='value'
                    invoke(file); // 変数の代入
                    x = "";
                } else {
                    break; // found.
                }
            }
        }
        BuiltInVar.ARGIND.put(i);
        BuiltInVar.FILENAME.put(x);
        return x;
    }

    //* ------------------------------------------------------------------------
    //* Array Getter / Setter - for Groovy.
    //* ------------------------------------------------------------------------
    //* この配列にアクセスするキーを返す.
    @Override
    public Integer arrayKey(Object index) {
        if (index instanceof Number)
            return ((Number) index).intValue();
        return NumHelper.intValue(index);
    }
}