/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;
import java.util.TreeMap;

/**
 * Lexical Analyzer - Symbols Table.
 *
 * @author kunio himei.
 */
public final class Symbols {

    /**
     * CLOSURE variables.
     */
    private static final Map<String, Integer> CLOSURE = new HashMap<>();

    /**
     * default value.
     */
    private static final int DEFAULT = Flags.T26NIL;

    /**
     * default value.
     */
    private static final Integer DEFAULT_INT = Flags.T26NIL;

    /**
     * Function name.
     */
    private static final Map<String, Integer> FUNCTIONS = new HashMap<>();

    /**
     * Global variables.
     */
    private static final Map<String, Integer> GLOBAL = new HashMap<>();
    /**
     * Local variables.
     */
    private static final Map<String, Integer> LOCAL = new HashMap<>();
    /**
     * Predef value.
     */
    private static final Map<String, Integer> PREDEF = new HashMap<>();
    /**
     * CLOSURE Stack.
     */
    private static final Stack<Map<String, Integer>> STACK_CLOSURE = new Stack<>();
    /**
     * Function Stack.
     */
    private static final Stack<String> STACK_FUNCTIONID = new Stack<>();
    /**
     * Local Stack.
     */
    private static final Stack<Map<String, Integer>> STACK_LOCAL = new Stack<>();
    /**
     * Local define mode (関数パラメータ定義中).
     */
    private static boolean isLocalDefMode;

    static {
        PREDEF.put("$", Flags.T09STNUM | Flags.T16ARRAY); // レコード変数
        PREDEF.put("ARGC", Flags.T03INT); // 配列 ARGV の要素数
        PREDEF.put("ARGIND", Flags.T03INT); // 配列 ARGV のインデックス (現在処理中のファイル名が格納されている要素を指す)
        PREDEF.put("ARGV", Flags.T09STNUM | Flags.T16ARRAY); // コマンド行引数配列
        PREDEF.put("CONVFMT", Flags.T07STR); // 数値から文字列への変換書式 (既定値は %.6g)
        PREDEF.put("ENVIRON", Flags.T07STR | Flags.T16ARRAY); // 現在の環境変数のコピー配列
        PREDEF.put("ERRNO", Flags.T07STR); // 例外発生理由を表わす文字列 (close,ファイル名指定のgetline)
        PREDEF.put("FIELDWIDTHS", Flags.T07STR); // 空白で区切られたフィールド長のリスト
        PREDEF.put("FILENAME", Flags.T09STNUM); // 現在の入力ファイル名
        PREDEF.put("FNR", Flags.T03INT); // 現在の入力ファイルから入力したレコード数
        PREDEF.put("FS", Flags.T07STR); // 入力フィールドセパレータ (既定値は空白)
        PREDEF.put("IGNORECASE", Flags.T03INT); // 文字列比較と正規表現マッチングで英大小文字を区別
        PREDEF.put("NF", Flags.T03INT); // 現在の入力レコードのフィールド数
        PREDEF.put("NR", Flags.T03INT); // 現在までに入力したレコード数の合計
        PREDEF.put("OFMT", Flags.T07STR); // 数値の出力書式 (既定値は %.6g)
        PREDEF.put("OFS", Flags.T07STR); // 出力フィールドセパレータ (既定値は空白)
        PREDEF.put("ORS", Flags.T07STR); // 出力レコードセパレータ (既定値はプラットフォーム依存の改行)
        PREDEF.put("RLENGTH", Flags.T03INT); // match で正規表現にマッチした文字列の長さ
        PREDEF.put("RS", Flags.T07STR); // 入力レコードセパレータ (既定値は改行)
        PREDEF.put("RSTART", Flags.T03INT); // match で正規表現にマッチした文字列の開始位置
        PREDEF.put("RT", Flags.T07STR); // RS にマッチし入力レコードセパレータとなった入力テキスト
        PREDEF.put("SUBSEP", Flags.T07STR); // 多次元配列の添字区切り文字
        PREDEF.put("PROCINFO", Flags.T11ANY | Flags.T16ARRAY); // 固有の拡張機能を制御するためのコンテナ

        PREDEF.put("RESULT", Flags.T11ANY); // ☆gsub,sub 関数での置換結果の文字列
        PREDEF.put("true", Flags.T02BOOL);
        PREDEF.put("false", Flags.T02BOOL);
        PREDEF.put("null", Flags.T11ANY);
    }

    /**
     * Don't let anyone instantiate this class.
     */
    private Symbols() {
        super();
    }

    /**
     * パース完了時に、未使用のグローバル変数を削除して返す.
     */
    public static Map<String, Integer> cleanGlobal() {
        Map<String, Integer> map = new TreeMap<>();
        for (Map.Entry<String, Integer> e : GLOBAL.entrySet()) {
            Integer x = e.getValue();
            if (DEFAULT != x) {
                map.put(e.getKey(), x);
            }
        }
        return map;
    }

    /**
     * GLOBAL の初期化.
     */
    public static void clearGlobal() {
        GLOBAL.clear();
    }

    /**
     * 変数(NAME)出現時に属性を初期登録.
     */
    public static void createType(String k) {
        if (isLocalDefMode) {
            // LOCALモード(関数パラメータ解析中)なら LOCAL変数を作成、それ以外は GLOBAL
            if (!LOCAL.containsKey(k) && !PREDEF.containsKey(k)) {
                LOCAL.put(k, DEFAULT_INT);
            }
        } else if (!GLOBAL.containsKey(k) && !PREDEF.containsKey(k)
                && !FUNCTIONS.containsKey(k)) {
            GLOBAL.put(k, DEFAULT_INT);
        }
    }

    /**
     * end Block.
     */
    public static void endBlock() {
        endLocal();
    }

    /**
     * end Local mode.
     */
    private static void endLocal() {
        CLOSURE.clear();
        CLOSURE.putAll(STACK_CLOSURE.pop());
        LOCAL.clear();
        LOCAL.putAll(STACK_LOCAL.pop());
    }

    /**
     * end Local mode.
     */
    public static String endLocal(String name) {
        endLocal();
        String x = STACK_FUNCTIONID.pop();
        assert name.equals(x);
        return STACK_FUNCTIONID.isEmpty() ? "" : STACK_FUNCTIONID.peek();
    }

    /**
     * 指定されたスコープ(コンテナ)を検索する.
     * ☆ 未使用パラメータの削除
     */
    @SafeVarargs
    @Nullable
    private static Map<String, Integer> find(String k,
                                             Map<String, Integer>... con) {
        for (Map<String, Integer> x : con) {
            if (x.containsKey(k)) {
                return x;
            }
        }
        return null;
    }

    /**
     * 変数が定義済みであることを確認(未登録エラーとしない).
     */
    public static boolean findType(String k) {
        return Flags.T27DISABLE != serch(k);
    }

    /**
     * 変数属性を取得する(未登録はエラー).
     */
    public static int getType(String k) {
        int x = serch(k);
        if (Flags.T27DISABLE == x) {
            dump();
            throw new IllegalStateException("UNDEFINED: '" + k + "'");
        }
        return x;
    }

    /**
     * Atomic かどうかを返す.
     */
    public static boolean isAtomic(String k) {
        Map<String, Integer> x = find(k, LOCAL, CLOSURE, GLOBAL);
        return (null != x) && Flags.isAtomic(x.get(k));
    }

    /**
     * Symbols.CLOSURE に存在するかどうかを返す.
     */
    public static boolean isClosure(String k) {
        return CLOSURE.containsKey(k) && !LOCAL.containsKey(k);
    }

    /**
     * Symbols.GLOBAL に存在するかどうかを返す.
     */
    public static boolean isGlobal(String k) {
        return GLOBAL.containsKey(k);
    }

    /**
     * Symbols.PREDEF の整数タイプ組込変数かどうかを返す.
     */
    public static boolean isIntPredefVar(String k) {
        return PREDEF.containsKey(k)
                && Flags.isInteger(PREDEF.get(k));
    }

    /**
     * Symbols.LOCAL に存在するかどうかを返す.
     */
    public static boolean isLocal(String k) {
        return LOCAL.containsKey(k);
    }

    /**
     * Symbols.PREDEF に存在するかどうかを返す.
     */
    public static boolean isPredef(String k) {
        return PREDEF.containsKey(k);
    }

    /**
     * Symbols.PREDEF に存在する文字列タイプの組込変数かどうかを返す.
     */
    public static boolean isStringPredef(String k) {
        return PREDEF.containsKey(k) && Flags.isString(PREDEF.get(k));
    }

    /**
     * GLOBAL 変数をマーキングする.
     */
    public static void markGlobal(String k, int x) {
        GLOBAL.put(k, x);
    }

    /**
     * LOCAL 変数をマーキングする.
     */
    public static void markLocal(String k, int x) {
        LOCAL.put(k, x);
    }

    /**
     * 関数の戻り値をリセットする.
     */
    public static void resetFunctionType(String k) {
        FUNCTIONS.put(k, DEFAULT_INT);
    }

    /**
     * for(var) ブロックのローカル変数をリセットする.
     */
    public static void resetLocalType(String k) {
        LOCAL.put(k, DEFAULT_INT);
    }

    /**
     * 現在のスコープを検索する.
     */
    private static int serch(String k) {
        Map<String, Integer> x = find(k, FUNCTIONS, LOCAL, CLOSURE,
                GLOBAL, PREDEF);
        return (null == x) ? Flags.T27DISABLE : x.get(k);
    }

    /**
     * 関数パラメータ定義中かどうかを設定する.
     */
    public static void setLocalDefMode(boolean x) {
        isLocalDefMode = x;
    }

    /**
     * 変数属性を設定する.<br>
     * 有効な値が既に登録されている場合は更新しない <br>
     * 有効な値が既に登録されており orType 指定の場合は更新する.
     */
    public static int setType(String k, int typ, int orType) {
        if (FUNCTIONS.containsKey(k)) {
            return setTypeImpl(k, typ, orType, FUNCTIONS);
        } else if (LOCAL.containsKey(k)) {
            return setTypeImpl(k, typ, orType, LOCAL);
        } else if (GLOBAL.containsKey(k)) {
            return setTypeImpl(k, typ, orType, GLOBAL);
        } else {
            return getType(k); // find CLOSURE, PREDEF
        }
    }

    /**
     * 変数属性を設定する.
     */
    private static int setTypeImpl(String k, int typ,
                                   int orType, Map<String, Integer> map) {
        int x = map.get(k);
        int w = (DEFAULT == x) ? typ : x;
        if (DEFAULT != w) {
            w |= orType;
            if (x != w) {
                map.put(k, w);
            }
        }
        return w;
    }

    /**
     * start Block mode.
     */
    public static void startBlock() {
        STACK_CLOSURE.push(new HashMap<>(CLOSURE));
        CLOSURE.putAll(LOCAL);
        STACK_LOCAL.push(new HashMap<>(LOCAL));
    }

    /**
     * start Local mode.
     */
    private static void startLocal() {
        STACK_CLOSURE.push(new HashMap<>(CLOSURE));
        CLOSURE.putAll(LOCAL);
        STACK_LOCAL.push(new HashMap<>(LOCAL));
        LOCAL.clear();
    }

    /**
     * start Local mode (関数登録).
     */
    public static String startLocal(String name) {
        startLocal();
        STACK_FUNCTIONID.push(name);
        if (!FUNCTIONS.containsKey(name)) {
            FUNCTIONS.put(name, DEFAULT_INT);
        }
        return name;
    }

    /**
     * Debug Dump.
     */
    public static void dump() {
        System.err.println("## Symbols ##" + "\n# FUNCTIONS." + FUNCTIONS
                + "\n# PREDEF." + PREDEF + "\n# GLOBAL." + GLOBAL
                + "\n# CLOSURE." + CLOSURE + "\n# LOCAL." + LOCAL);
        System.err.flush();
    }
}