/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.variable;

import plus.concurrent.AtomicMap;
import plus.concurrent.AtomicNumber;
import plus.eval.EvalVar;
import plus.util.NumHelper;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

/**
 * コンテナの実装 (Implementation of the variable context).
 *
 * @author kunio himei.
 */
public final class Frame {

    /**
     * global variables.
     */
    public static final AtomicMap GLOBALS = new AtomicMap();
    //* 空の要素(null)を表す値.
    private static final String EMPTY_VALUE = null;
    /**
     * closure variables.
     */
    private static final ThreadLocal<AtomicMap> CLOSURE =
            ThreadLocal.withInitial(AtomicMap::new);
    /**
     * local variables.
     */
    private static final ThreadLocal<AtomicMap> LOCAL =
            ThreadLocal.withInitial(AtomicMap::new);
    /**
     * closure Stack.
     * Android で Deque が実装されていなかったため、
     * スタックを ArrayList で実装している.
     */
    private static final ThreadLocal<ArrayList<AtomicMap>> CLOSURESTACK =
            ThreadLocal.withInitial(ArrayList::new);
    /**
     * local Stack.
     */
    private static final ThreadLocal<ArrayList<AtomicMap>> LOCALSTACK =
            ThreadLocal.withInitial(ArrayList::new);
    /**
     * 同期オブジェクト.
     */
    private final ReentrantLock Var = new ReentrantLock(); // SYNC.
    private final ReentrantLock RefVar = new ReentrantLock(); // SYNC.

    /**
     * Singleton パターン - インスタンスは、一つだけ.
     */
    private static final Frame singleton = new Frame();

    private Frame() {
    }

    public static Frame getInstance() {
        return singleton;
    }

    /**
     * find Container.
     */
    private static AtomicMap find(String name, AtomicMap undefined,
                                  AtomicMap... container) {
        for (AtomicMap map : container)
            if (map.containsKey(name))
                return map;
        return undefined;
    }

    /**
     * 連想配列であることを確認し配列が存在しない場合は新規配列を返す.
     */
    private static Map<?, ?> getArray(String name, AtomicMap frame) {
        Object x = frame.get(name);
        if (x instanceof Map<?, ?>)
            return (Map<?, ?>) x;
        AtomicMap arr = new AtomicMap(); // 未定義の場合は新規に作成
        frame.put(name, arr);
        return arr;
    }

    /**
     * スタック変数を更新する.
     * <nl>
     * <li>変数は既に存在するはず、存在しなければ NoSuchVariableException.
     * <li>スタックは ThreadLocal かつ 不活性領域のため排他の考慮は不要.
     */
    private static void updateStack(String name, Object x) {
        if (null == x) throw new NullPointerException(name);
        for (AtomicMap map : LOCALSTACK.get()) { // 必ず存在する
            if (null != map.replace(name, x)) {
                break; // local スタックを更新
            }
        }
        for (AtomicMap map : CLOSURESTACK.get()) { // 存在は不明
            if (null != map.replace(name, x)) {
                break; // closure スタックを更新
            }
        }
    }

    /**
     * 四則演算.
     */
    private static Object calculate(AtomicMap frame, Object index,
                                    String op, Object value) {
        String k = (String) frame.arrayKey(index);
        Object x = frame.get(k);
        if ((x instanceof Number) && (value instanceof Number)) {
            if (!(x instanceof AtomicNumber)) {
                x = EvalVar._cloneAtom(x);
                frame.put(k, x); // Atomicに昇格.
            }
            return ((AtomicNumber) x).calculate(op, value);
        }
        if ("=".equals(op)) {
            frame.put(k, value); // Var/Val対応のため、生データを格納.
            return value;
        }
        throw new IllegalArgumentException(x + " " + op + " " + value);
    }

    /**
     * ローカル変数を新規に作成し値をセットする.
     */
    public static Object _putLocalVar(String name, Object value) {
        LOCAL.get().put(name, value);
        return value;
    }

    /**
     * 現在のスコープの複製を返す (新規スレッドの初期値として使用する).
     */
    public static AtomicMap _cloneLocalContext() { //
        return new AtomicMap(LOCAL.get());
    }

    /**
     * ローカル スコープを終了する.
     */
    public static void _endLocal() { // closure, localを復元する
        AtomicMap closure = CLOSURE.get();
        closure.clear();
        closure.putAll(CLOSURESTACK.get().remove(0)); // pop, override
        AtomicMap local = LOCAL.get();
        local.clear();
        local.putAll(LOCALSTACK.get().remove(0)); // pop, override
    }

    /**
     * 現在のスコープを開始する (ブロックの開始時に呼び出される).
     */
    public static void _startBlockContext() {
        CLOSURE.get().clear();
        CLOSURESTACK.get().clear();
        LOCAL.get().clear();
        LOCALSTACK.get().clear();
    }

    /**
     * 現在のスコープ を開始する (新規スレッドの開始時に呼び出される).
     */
    public static void _startNewContext(AtomicMap context) { //
        CLOSURE.get().putAll(context);
    }

    /**
     * 新規にローカルスコープを開始する.
     */
    public static void _startLocal() {
        // closureの複製を退避して closureに localをマージ
        AtomicMap closure = CLOSURE.get(); // clone, push
        CLOSURESTACK.get().add(0, new AtomicMap(closure));
        AtomicMap local = LOCAL.get();
        closure.putAll(local); // override
        LOCALSTACK.get().add(0, new AtomicMap(local));
        local.clear();
    }

    /**
     * 連想配列を返す(存在しない場合は新規に作成).
     */
    public Map<?, ?> _allocArray(String name) {
        AtomicMap frame = find(name, GLOBALS, //
                LOCAL.get(), CLOSURE.get());
        assert (null != frame);
        return getArray(name, frame);
    }

    /**
     * ローカル配列を作成(存在しない場合は新規に作成).
     */
    public void _allocLocalArray(String name, String oldkey) {
        Map<?, ?> arr = (null != oldkey) //
                ? _allocArray(oldkey) // 呼出し元の配列を参照
                : new AtomicMap(); // 新規に配列を作成
        LOCAL.get().put(name, arr);
    }

    /**
     * 要素の値を返す.
     */
    public Object _getAt(String name, Object index) {
        this.Var.lock(); // SYNC. _getAt.
        try {
            Object x = null;
            AtomicMap frame = find(name, null, //
                    LOCAL.get(), CLOSURE.get(), GLOBALS);
            if (null != frame) {
                x = frame.getAt(name);
                if (null != index) {
                    if (x instanceof Object[]) { // 可変長引数
                        Object[] arr = (Object[]) x;
                        int ix = NumHelper.intValue(index);
                        x = ((arr.length > ix) ? arr[ix] : EMPTY_VALUE);
                    } else {
                        if (!(x instanceof AtomicMap)) {
                            x = _allocArray(name); // 初回の代入
                        }
                        Map<?, ?> arr = (Map<?, ?>) x;
                        if (arr.containsKey(index)) {
                            x = arr.get(index);
                        } else { // 参照副作用エミュレーション
                            x = (("$".equals(name)) ? EMPTY_VALUE : arr.get(index));
                        }
                    }
                }
            }
            return x;
        } finally {
            this.Var.unlock();
        }
    }

    /**
     * 要素に値を設定する.
     */
    public Object _putAt(String op, String name, Object index, Object value) {
        this.Var.lock(); // SYNC. _putAt.
        try {
            AtomicMap closure = CLOSURE.get();
            AtomicMap frame = find(name, GLOBALS, LOCAL.get(), closure);
            if (null == frame)
                throw new IllegalStateException("NoSuchVariableException: " + name);
            Object x;
            if (null != index) {
                Object arr = frame.getAt(name);
                if (!(arr instanceof AtomicMap)) {
                    arr = _allocArray(name); // 初回の代入
                }
                x = calculate((AtomicMap) arr, index, op, value);
            } else if ("=".equals(op) && !frame.containsKey(name)) {
                frame.put(name, value); // 初回の代入
                x = value;
            } else {
                x = calculate(frame, name, op, value);
            }
            if (closure == frame) {
                assert (null != x);
                updateStack(name, x); // closureスタックを更新.
            }
            return x;
        } finally {
            this.Var.unlock();
        }
    }

    /**
     * 参照変数から値を取得する.
     * <p>
     * {@literal IllegalStateException} - 変数が存在しない場合
     */
    public Object _getRefVar(String name, Object index) {
        this.RefVar.lock(); // SYNC. _getRefVar.
        try {
            Object x;
            AtomicMap closure = CLOSURE.get();
            AtomicMap frame = find(name, null, //
                    closure, GLOBALS); // カレントローカルは自分自身のため更新しない
            if (null == frame)
                throw new IllegalStateException("NoSuchVariableException: " + name);
            if (null != index) {
                Map<?, ?> arr = getArray(name, frame);
                x = arr.get(index);
            } else {
                x = frame.getAt(name);
            }
            return x;
        } finally {
            this.RefVar.unlock();
        }
    }

    /**
     * 参照変数に値を設定する.
     */
    public Object _putRefVar(String name, Object index, Object value) {
        this.RefVar.lock(); // SYNC. _putRefVar.
        try {
            AtomicMap closure = CLOSURE.get();
            AtomicMap frame = find(name, null, //
                    closure, GLOBALS); // カレントローカルは自分自身のため更新しない
            Object x;
            if (null == frame)
                throw new IllegalStateException("NoSuchVariableException: " + name);
            if (null != index) {
                @SuppressWarnings("unchecked")
                Map<Object, Object> arr = (Map<Object, Object>) getArray(name, frame);
                x = arr.put(index, value);
            } else {
                x = frame.put(name, value);
                //noinspection ObjectEquality
                if (closure == frame) { // == this surely will never happen
                    assert (null != x);
                    updateStack(name, x); // closureスタックを更新
                }
            }
            return x;
        } finally {
            this.RefVar.unlock();
        }
    }

    /**
     * 配列または配列要素の削除.
     */
    public void _remove(String name, Object index) {
        Map<?, ?> arr = _allocArray(name);
        if (null != index) {
            arr.remove(index);
        } else {
            arr.clear();
        }
    }
}