/**
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;

/**
 * cutSpace - Sub Modules.
 *
 * @author kunio himei.
 */
class Modules {

    //////////////////////////////////////////////////////////////////////
    static final StringBuilder joinSource = new StringBuilder(1024 * 8);
    static final StringBuilder joinTemplate = new StringBuilder(1024 * 8);

    /**
     * Join input line.
     * - Buffering of combined data.
     *
     * @return rTrim (null during join)
     */
    @Nullable
    static T2TPL joinInputLine(@NotNull String s) {
        T2TPL t2;
        if (joinSource.isEmpty()) { // Idling process
            t2 = Parser.TEMPLATE.parse(s); // Remove comments and mark strings
            if (!t2.hasJoin())
                return t2;
            joinSource.append(t2.source()).append(t2.sep1()); // 1st process
            joinTemplate.append(t2.template()).append(t2.sep2());
            return null;
        }
        t2 = Parser.TEMPLATE.parse(s); // 2nd process
        joinSource.append(t2.source()).append(t2.sep1());
        joinTemplate.append(t2.template()).append(t2.sep2());
        if (t2.hasJoin()) {
            return null;
        }
        String src = joinSource.toString(); // Final process
        String tmpl = joinTemplate.toString();
        joinSource.setLength(0);
        joinTemplate.setLength(0);
        return new T2TPL(src, tmpl, "", "", t2.request());
    }

    //////////////////////////////////////////////////////////////////////
    // 後方に空白を追加しない - Do not add trailing blank.
    private static final String COMPLETED_LINE = ";{}>";

    /**
     * Append the line.
     *
     * @param s  append text. (!isEmpty)
     * @param rs Insert at line break position. (Space or ${RS})
     */
    @NotNull
    static StringBuilder appendLine(@NotNull StringBuilder sb, String s, char rs) {
        sb.append(s);
        char c = Tools.charAtLast(s); // Not IndexOutOfBounds
        if (0 > COMPLETED_LINE.indexOf(c)) {
            sb.append(rs); // line break (\n equivalent)
        }
        return sb;
    }

    /**
     * Append a line break
     */
    static void appendNewLine(@NotNull StringBuilder sb) {
        Tools.rTrim(sb).append('\n');
    }

    //////////////////////////////////////////////////////////////////////
    /*
     * Output and flush out | err (UTF_8, \n)
     */
    static void flush(@NotNull StringBuilder output) throws IOException {
        PrintStream out = output == Parser.out ?
                System.out : System.err;
        // Binary source to set the line feed code to '\n'.
        out.write(output.toString().getBytes(StandardCharsets.UTF_8));
        out.flush();
        output.setLength(0);
    }
}