/**
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * cutSpace - Optimize the Space and Semicolon.
 *
 * @author kunio himei.
 */
class Space {

    // Language start (attribute) tag.
    private static final Pattern LANGUAGE_START_TAG = Pattern.compile(
            "<(?:script|style)[^>]+?>", Pattern.CASE_INSENSITIVE);

    private static final Pattern CUT_LAST_SEMICOLON = Pattern.compile( // ;} -> }
            "[\\s]*;[\\s]*}[\\s]*");

    private String optCSS = CSS_SELECTOR; // Inside of selector
    private String optHTML = HTML_OUTSIDE_TAG; // Outside of <>
    private boolean isStateTransitionProhibited; // 状態遷移禁止 (HTML埋め込みモード)

    /**
     * Space compression.
     * -  Optimize language start tag.
     */
    String space(String input, boolean deprecated) {
        Matcher m = LANGUAGE_START_TAG.matcher(input);
        if (m.find()) { // If the attribute is specified
            // REMIND HTML以外の言語を開始するということは、HTMLをリセットして良い.
            //  Starting a language other than HTML may reset HTML.
            optHTML = HTML_OUTSIDE_TAG;
            String tag = html(m.group(), deprecated);
            input = m.replaceFirst(tag);
            optHTML = HTML_OUTSIDE_TAG;
        }
        if (Lang.is(Lang.SCRIPT)) { // SCRIPT
            return script(input);

        } else if (Lang.is(Lang.CSS)) { // CSS
            return css(input);
        }
        return html(input, deprecated); // HTML
    }

    /**
     * script. ///////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Cut semicolon, ;} -> }
     *
     * @param input text. (trim)
     * @return source text.
     */
    @NotNull
    private String script(@NotNull String input) {
        final int capacity = input.length() + 7; // これは、お守り (Amulet)
        StringBuilder sb = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input.trim());
        String src = t2.source();
        String tpl = t2.template();
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                optimizeSpace(sb, SCRIPT_OPTIMIZE, c);
            } else
                sb.append(c);
        }
        Matcher m = CUT_LAST_SEMICOLON.matcher(sb); // ;} -> }
        if (m.find())
            return m.replaceAll("}");
        return sb.toString();
    }

    /**
     * css. //////////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Cut semicolon, ;} -> }
     *
     * @param input text. (trim)
     * @return source text.
     */
    @NotNull
    private String css(@NotNull String input) {
        final int capacity = input.length() + 7; // ';' Is added in Template
        StringBuilder sb = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input.trim());
        String src = t2.source();
        String tpl = t2.template();
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                if (!isStateTransitionProhibited) {
                    if ('[' == c) optCSS = CSS_ATTRIBUTE_SELECTOR; // [Attribute]
                    else if (']' == c) optCSS = CSS_SELECTOR;
                    else if ('{' == c) optCSS = CSS_ATTRIBUTE; // {Attribute:value;}
                    else if ('}' == c) optCSS = CSS_SELECTOR;
                    else if ('@' == c) optCSS = CSS_MEDIA_QUERIES; // @media(){}
                }
                optimizeSpace(sb, optCSS, c);
            } else
                sb.append(c);
        }
        Matcher m = CUT_LAST_SEMICOLON.matcher(sb); // ;} -> }
        if (m.find())
            return m.replaceAll("}");
        return sb.toString();
    }

    /**
     * html. /////////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Remove whitespace around html block tags.
     * - Optimize Type="CSS", media="screen and (max-width:480px)"
     * - Cut semicolon, Type="foo;", onContextMenu="return false;"
     *
     * @param input      text. (trim)
     * @param deprecated <tag foo=""  bar=""> → <tag foo=""bar="">
     * @return source text. (rTrim)
     */
    @NotNull
    private String html(@NotNull String input, boolean deprecated) {
        final int capacity = input.length() + 7; // '\r' Is added
        StringBuilder sb = new StringBuilder(capacity);
        StringBuilder type = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input.trim());
        String src = t2.source() + '\r'; // Guard, Required if ending with a string
        String tpl = t2.template() + '\r';
        String save = optCSS;
        isStateTransitionProhibited = true; // Inline mode
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                if (!type.isEmpty()) { // Optimize CSS in tag <tag Type = "CSS">
                    optCSS = CSS_ATTRIBUTE;
                    CharSequence seq = css(type.substring(1, type.length() - 1));
                    if (Tools.endsWith(seq, ';')) { // Cut last semicolon
                        seq = seq.subSequence(0, seq.length() - 1);
                    }
                    char qt = type.charAt(0);
                    sb.append(qt).append(seq).append(qt);
                    type.setLength(0);
                }
                if ('<' == c)
                    optHTML = deprecated ? HTML_INSIDE_DEPRECATED : HTML_INSIDE_TAG;
                optimizeSpace(sb, optHTML, c);
                if ('>' == c) optHTML = HTML_OUTSIDE_TAG;
            } else
                type.append(c);
        }
        optCSS = save;
        isStateTransitionProhibited = false;
        return cutSpace(Tools.rTrim(sb)); // Remove '\r'
    }

    /**
     * Optimize the space. ////////////////////////////////////////////////
     *
     * @param sb       I/O.
     * @param optimize OPTIMIZE control characters.
     * @param ch       Additional character.
     */
    private static void optimizeSpace(@NotNull StringBuilder sb, String optimize, char ch) {
        char c = Tools.toWhiteSpace(ch);
        if (!sb.isEmpty() &&
                0 <= optimize.indexOf(c)) { // if Optimize target
            char front = Tools.charAtLast(sb); // The input is not empty
            // The front is the character to be optimized
            if (0 <= optimize.indexOf(front)) {
                if (' ' == c) return; // Do not add whitespace
            }
            if (' ' == front) {
                if (' ' == c) return; // Do not add double space
                sb.setLength(sb.length() - 1); // Remove front whitespace
            }
        }
        sb.append(c);
    }

    //////////////////////////////////////////////////////////////////////
    private static final Pattern HTML_EXTRACT_TAGS = Pattern.compile(
            "[\\s]*(</?" + "([a-zA-Z]+[1-6]?)" + "[^>]*?" + ">)[\\s]*");
    //             group(1) group(2):Tag name

    /**
     * html: ブロックレベル要素の周辺空白を削除する.
     * - Remove whitespace around block-level elements.
     *
     * - パフォーマンスチューニングのため要素検索をString->Setに変更.
     * - Changed element search to String -> Set for performance tuning.
     */
    @NotNull
    static String cutSpace(@NotNull CharSequence s) {
        StringBuilder sb = new StringBuilder(s.length());
        Matcher m = HTML_EXTRACT_TAGS.matcher(s);
        while (m.find()) {
            String replacement;
            String tag = m.group(2).toLowerCase();
            if (HTML_BLOCK_ELEMENT_SET.contains(tag)) {
                replacement = m.group(1); // block-level element.
                Tools.rTrim(sb); // Eat the blank on the right.
            } else {
                replacement = m.group(); // inline-level element.
            }
            m.appendReplacement(sb, replacement);
        }
        m.appendTail(sb);
        return sb.toString();
    }

    // https://developer.mozilla.org/ja/docs/Web/HTML/Block-level_elements
    // NOTE ※ 空白を無視するかどうかで追加したタグで、ブロック要素でないものを含む.
    //  Tags added depending on whether to ignore whitespace,
    //  including those that are not block elements.
    @SuppressWarnings("SpellCheckingInspection")
    private static final List<String> HTML_BLOCK_ELEMENT_LIST = Arrays.asList(
            "address",      // 連絡先情報
            "article",      // 記事コンテンツ
            "aside",        // 本論から外れたコンテンツ
            "blockquote",   // 長い(「ブロック」の)引用
            "br",           // ※ (改行)
            "dd",           // 定義リストで用語を説明
            "details",      // 折りたたみウィジェット
            "dialog",       // ダイアログボックス
            "div",          // 文書の一部
            "dl",           // 定義リスト
            "dt",           // 定義語リストの用語
            "fieldset",     // フィールドセットのラベル
            "figcaption",   // 図表のキャプション
            "figure",       // キャプション(figcaption)を伴うメディアコンテンツをグループ化
            "footer",       // セクションまたはページのフッター
            "form",         // 入力フォーム
            "h1",           // 見出しレベル 1-6
            "h2",           //
            "h3",           //
            "h4",           //
            "h5",           //
            "h6",           //
            "header",       // セクションまたはページのヘッダー
            "hgroup",       // 見出し情報をグループ化
            "hr",           // 水平線(区切り線)
            "li",           // リストの項目
            "main",         // この文書で固有の中心的なコンテンツを含む
            "nav",          // ナビゲーションのリンクを含む
            "ol",           // 番号付きリスト
            "pre",          // 整形済みテキスト
            "p",            // 段落
            "section",      // ウェブページのセクション
            "table",        // 表
            "td",           // ※ (表のデータ)
            "th",           // ※ (表のヘッダ)
            "tr",           // ※ (表の行部分)
            "title",        // ※ (文書の題名)
            "ul");          // 番号なしリスト
    private static final Set<String> HTML_BLOCK_ELEMENT_SET =
            new HashSet<>(HTML_BLOCK_ELEMENT_LIST);

    //////////////////////////////////////////////////////////////////////
    // https://developer.mozilla.org/ja/docs/Web/CSS/CSS_Selectors
    // CSS : selector{attribute:value;}
    // -  子結合 >、一般兄弟 ~、隣接兄弟 +、E:not(s)、
    // -  属性セレクタ E[foo]、E[foo="bar"]、E[foo~="bar"]、E[foo|="en"]、
    // -  E[foo^="bar"]、E[foo$="bar"]、E[foo*="bar"]
    // h1, h2 : 1つのCSSで複数のセレクタを対象にする(カンマ区切り、セレクターリスト)
    // .p1 .c2 : 複数のセレクタで階層構造の対象を絞り込む(空白区切り、子孫セレクタ) ※
    // .ｃ1.ｃ2 : 複数のセレクタで対象を絞り込む(区切り文字なし) <p class="ｃ1 ｃ2"> ※
    // ul > li : 特定の要素の直下にある要素を>区切りで指定する（子セレクタ）
    // p ~ span : <p> 要素の後にある <span> 要素をすべて選択（一般兄弟結合）
    // p + span : <p> 要素の後にすぐに続く <span> 要素をすべて選択（隣接兄弟結合）
    // col || td : <col> 要素のスコープに所属するすべての <td> 要素を選択（列結合）
    // a[href][title] : 複数の属性セレクタで対象を絞り込む([ ])
    // a:link[class="p1"] : 擬似クラスと属性セレクタで絞り込む
    // p[class="p2"]::before : 属性セレクタと擬似要素を指定する
    // a:link::before : 擬似クラスと擬似要素を指定する
    //
    // Remove whitespace at both ends (両端の空白を削除)
    // NOTE ※ "." Is not defined in the selector
    //  because the meaning is different depending on the presence or absence of blanks.
    //  ※ 空白の有無で意味が異なるため、”.”は、セレクタに定義しない.
    //                          SCRIPT_OPERATOR = "=.,:?!|&^~+-*/%";
    private static final String SCRIPT_OPTIMIZE = " ;()[]{}<>" + Template.SCRIPT_OPERATOR;
    private static final String CSS_SELECTOR = " ,:>~+|()[]{}"; // ※
    private static final String CSS_ATTRIBUTE_SELECTOR = " ()[=~|^$*]{}"; // [Attr Selector]
    private static final String CSS_ATTRIBUTE = " .,:;!#/(=<>)[]{}";
    private static final String CSS_MEDIA_QUERIES = CSS_ATTRIBUTE; // @media{}
    private static final String HTML_OUTSIDE_TAG = " ";
    private static final String HTML_INSIDE_TAG = " =>"; // <tag foo="" bar="">
    private static final String HTML_INSIDE_DEPRECATED = " =>\"'"; // <tag foo=""bar="">
}