/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * cutSpace - Tools.
 *
 * @author kunio himei.
 */
class Tools {

    static int RSTART; // Match start
    static int RLENGTH; // Match length

    /**
     * match(r,s) - AWK specifications.
     *
     * @param group number 0, 1.. m.groupCount()
     */
    static boolean match(@NotNull Pattern r, CharSequence s, int group) {
        Matcher m = r.matcher(s);
        if (m.find()) {
            RSTART = m.start(group);
            RLENGTH = m.end(group) - RSTART;
            return true;
        }
        return false;
    }

    /**
     * Replace All.
     */
    static String replaceAll(@NotNull Pattern r, String s, String replacement) {
        Matcher m = r.matcher(s);
        if (m.find())
            return m.replaceAll(replacement);
        return s;
    }

    /*
     * Benchmark result of Trm.
     * - 4.1 Classic regex: /[\s]+$/
     * - 2.9 New regex: /[\0- ]+$/
     * - 1.2 String         ※ Algorithm of String#trim.
     * - 1.0 StringBuilder  ※
     */

    /**
     * Left trim (String) - String#trim() specifications.
     */
    @SuppressWarnings({"StatementWithEmptyBody", "unused"})
    @NotNull
    static String lTrim(@NotNull String s) {
        int i = -1, len = s.length();
        while (++i < len && ' ' >= s.charAt(i))
            ; // do nothing
        return i > 0 ? s.substring(i) : s;
    }

    /**
     * Right trim (String) - String#trim() specifications.
     * - Does not generate a new character string when missed.
     * - point. 空振りした時に新たな文字列を生成しない.
     */
    @SuppressWarnings("StatementWithEmptyBody")
    @NotNull
    static String rTrim(@NotNull String s) {
        int len, i = len = s.length();
        while (0 <= --i && ' ' >= s.charAt(i))
            ; // do nothing
        return ++i < len ? s.substring(0, i) : s;
    }

    /**
     * Right trim. (StringBuilder)
     * - High performance because no string generation is required.
     * - point. 文字列の生成が不要なため高性能.
     */
    @SuppressWarnings("StatementWithEmptyBody")
    @NotNull
    static StringBuilder rTrim(@NotNull StringBuilder sb) {
        int len, i = len = sb.length();
        while (0 <= --i && ' ' >= sb.charAt(i))
            ; // do nothing
        if (++i < len) sb.setLength(i);
        return sb;
    }

    /**
     * Right twin trim. (Source, Template)
     */
    static void rTrim(@NotNull StringBuilder sb, @NotNull StringBuilder sx) {
        int len = rTrim(sx).length(); // This is the first for DEL measures.
        sb.setLength(len);
    }

    /**
     * charAt - Gets the character of the specified index.
     *
     * @param cs Do not allow the empty string.
     * @throws AssertionError - IndexOutOfBoundsException '\\nl' Ref. Template.parse
     */
    static char charAt(@NotNull CharSequence cs, int ix) {
        assert 0 <= ix && ix < cs.length() :
                String.format("IndexOutOfBoundsException Index: %d/%d [%s]",
                        ix, cs.length(), cs);
        return toWhiteSpace(cs.charAt(ix));
    }

    /**
     * charAtLast - Get the character of the last index.
     *
     * @param cs Do not allow the empty string.
     * @throws AssertionError - Empty string
     */
    static char charAtLast(@NotNull CharSequence cs) {
        int ix = cs.length() - 1;
        assert 0 <= ix : "Empty string";
        return toWhiteSpace(cs.charAt(ix));
    }

    /**
     * Backward match. (char)
     *
     * @param cs Allows an empty string.
     */
    static boolean endsWith(@NotNull CharSequence cs, char c) {
        if (!cs.isEmpty())
            return toWhiteSpace(c) == charAtLast(cs);
        return false;
    }

    /*
     * @return a Blank if White Space.
     */
    private static final String WHITE_SPACES = " \t\n\r" + Template.RS;

    static char toWhiteSpace(char c) {
        return 0 <= WHITE_SPACES.indexOf(c) ? ' ' : c;
    }
}