/*
 * Copyright (C) 2011 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.channels.SocketChannel;

/**
 * [%inet%] クライアントソケットの実装 (java.net.Socket).
 *
 * @author kunio himei.
 */
final class InetSocket implements Connectable, ReadWriteable {

    /**
     * クライアントソケット.
     */
    private final Socket socket;
    /**
     * 接続待ち時間.
     */
    private final int timeout;
    /**
     * ソケットリーダ.
     */
    private InputStream reader;
    /**
     * ソケットライタ.
     */
    private OutputStream writer;

    /**
     * ソケットの構築.
     *
     * @param tout 接続待ち時間
     * @param sock クライアントソケット
     */
    InetSocket(final Socket sock, final int tout) {
        this.socket = sock;
        this.timeout = tout;
    }

    /**
     * ソケットの構築.
     *
     * @param soch ソケットチャネル
     * @param tout 接続待ち時間
     */
    InetSocket(final SocketChannel soch, final int tout) throws IOException {
        soch.configureBlocking(true);
        this.socket = soch.socket();
        this.timeout = tout;
    }

    /**
     * ソケットを閉じる.
     */
    @Override
    public void close() throws IOException {
        if (!this.socket.isClosed()) {
            // System.err.println("InetSocket#close");
            this.socket.close();
        }
    }

    /**
     * 接続されたソケットインターフェイスを返す.
     */
    @Override
    public ReadWriteable connect() {
        return this; // このクラスを返す
    }

    /**
     * ソケットリーダを返す.
     */
    @Override
    public InputStream getInputStream() throws IOException {
        this.socket.setSoTimeout(this.timeout); // 接続待ち時間
        if (null == this.reader) {
            this.reader = this.socket.getInputStream();
        }
        return this.reader;
    }

    /**
     * ソケットライタを返す.
     */
    @Override
    public OutputStream getOutputStream() throws IOException {
        if (null == this.writer) {
            this.writer = this.socket.getOutputStream();
        }
        return this.writer;
    }
}