/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.value;

import java.math.BigInteger;
import java.util.regex.Pattern;

import net.morilib.awk.namespace.AwkNamespace;

public class AwkInteger extends AwkValue {

	/**
	 * 
	 */
	public static final AwkInteger ONE = AwkInteger.valueOf(1);

	/**
	 * 
	 */
	public static final AwkInteger ZERO = AwkInteger.valueOf(0);

	//
	private BigInteger value;

	private AwkInteger(BigInteger v) {
		value = v;
	}

	public static AwkInteger valueOf(long v) {
		return new AwkInteger(BigInteger.valueOf(v));
	}

	public static AwkInteger valueOf(BigInteger v) {
		return new AwkInteger(v);
	}

	public static AwkValue booleanOf(boolean b) {
		return b ? TRUE : FALSE;
	}

	public double toFloat() {
		return value.doubleValue();
	}

	public BigInteger toInteger() {
		return value;
	}

	public boolean toBoolean() {
		return value.signum() != 0;
	}

	public Number toReal() {
		return value;
	}

	public Pattern toRegex() {
		return Pattern.compile(value.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.value.AwkValue#toObject()
	 */
	public Object toObject() {
		return value;
	}

	public boolean isFloat() {
		return false;
	}

	public boolean isInteger() {
		return true;
	}

	public boolean isReal() {
		return true;
	}

	public boolean isFloatValue() {
		return true;
	}

	public boolean isIntegerValue() {
		return true;
	}

	public boolean isRealValue() {
		return true;
	}

	@Override
	public boolean isNegativeValue() {
		return value.signum() < 0;
	}

	@Override
	public boolean isPositiveValue() {
		return value.signum() > 0;
	}

	@Override
	public boolean isZeroValue() {
		return value.signum() == 0;
	}

	public boolean isString() {
		return false;
	}

	public boolean isRegex() {
		return false;
	}

	public boolean isFunction() {
		return false;
	}

	@Override
	public boolean isEmpty() {
		return value.signum() == 0;
	}

	public int hashCode() {
		return value.hashCode();
	}

	public boolean equals(Object o) {
		if(o instanceof AwkInteger) {
			return value.equals(((AwkInteger)o).value);
		}
		return false;
	}

	public String toString(AwkNamespace ns) {
		return value.toString();
	}

	public int compareTo(AwkValue o, AwkNamespace ns) {
		if(o instanceof AwkUndefined) {
			return 1;
		} else if(o.isFloat()) {
			return toFloat() < o.toFloat() ?
					-1 : toFloat() > o.toFloat() ? 1 : 0;
		} else if(o.isInteger()) {
			return value.compareTo(o.toInteger());
		} else {
			return toString(ns).compareTo(o.toString(ns));
		}
	}

}
