/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.morilib.awk.builtin.AwkBuiltInLoader;
import net.morilib.awk.code.AwkProgram;
import net.morilib.awk.encoding.EncodingDetector;
import net.morilib.awk.encoding.EncodingDetectorFactory;
import net.morilib.awk.expr.AwkExpression;
import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.misc.FilesReader;
import net.morilib.awk.misc.RecordInputStream;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.namespace.AwkRootNamespace;
import net.morilib.awk.parser.AwkLexer;
import net.morilib.awk.parser.AwkParser;
import net.morilib.awk.value.AwkArray;
import net.morilib.awk.value.AwkInteger;
import net.morilib.awk.value.AwkString;
import net.morilib.awk.value.AwkUndefined;
import net.morilib.awk.value.AwkValue;
import net.morilib.c.pre.CPreprocessorReader;

/**
 * awkiumのFacadeです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public final class Awk {

	private static final Awk _INS = new Awk();

	/**
	 * awkiumのバージョンです。
	 */
	public static final String VERSION = "0.1.1";

	//
	private Awk() {}

	/**
	 * awkiumのFacadeを取得します。
	 * 
	 * @return awkiumのFacade
	 */
	public static Awk getInstance() {
		return _INS;
	}

	/**
	 * 名前空間を生成します。
	 * 
	 * @param filename ファイル名
	 * @param args     引数
	 * @return 名前空間
	 */
	public static AwkNamespace newNamespace(String filename,
			String... args) {
		AwkNamespace r = new AwkRootNamespace();
		AwkValue a;
		String s;

		AwkBuiltInLoader.load(r);
		r.assign("FILENAME", AwkString.valueOf(filename));
		r.assign("ARGC", AwkInteger.valueOf(args.length));
		r.assign("ARGV", new AwkArray(0, args));

		// environment
		a = new AwkArray();
		for(Map.Entry<String, String> e : System.getenv().entrySet()) {
			a.putArray(e.getKey(), AwkString.valueOf(e.getValue()));
		}
		r.assign("ENVIRON", a);

		// properties
		a = new AwkArray();
		for(Object o : System.getProperties().keySet()) {
			s = o.toString();
			a.putArray(s, AwkString.valueOf(System.getProperty(s)));
		}
		r.assign("PROPERTIES", a);
		return r;
	}

	/**
	 * awkiumプログラムをコンパイルします。
	 * 
	 * @param namespace 名前空間
	 * @param source ソースのReader
	 * @return コンパイル結果の中間表現
	 * @throws IOException IOエラー
	 */
	public AwkProgram compile(AwkNamespace namespace,
			Reader source) throws IOException {
		AwkProgram p;
		AwkLexer l;

		l = new AwkLexer(source);
		p = AwkParser.parse(namespace, l);
		return p;
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param namespace 名前空間
	 * @param program コンパイルされたawkiumプログラム
	 * @param filename 標準入力のファイル名
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(AwkNamespace namespace,
			AwkProgram program, String filename,
			InputStream stdin, Writer stdout, Writer stderr,
			Charset inputEncoding, Map<Object, Object> vars,
			String... args) throws IOException {
		RecordInputStream rd;
		AwkFiles f = null;
		String s;

		rd = new RecordInputStream(stdin, inputEncoding, namespace);
		if(vars != null) {
			for(Map.Entry<Object, Object> o : vars.entrySet()) {
				namespace.assign(o.getKey().toString(),
						AwkString.valueOf(o.getValue().toString()));
			}
		}

		try {
			f = new AwkFiles(rd, stdout, stderr);
			program.executeBegin(namespace, f);
			stdout.flush();  stderr.flush();
			if(!program.isExecuteOnce()) {
				while((s = rd.readRecord()) != null) {
					try {
						program.execute(namespace, f, s);
					} catch(AwkNextException e) {
						// ignore
					} catch(AwkExitException e) {
						program.executeEnd(namespace, f);
						return e.getValue();
					} finally {
						stdout.flush();  stderr.flush();
					}
				}
			}
			program.executeEnd(namespace, f);
			stdout.flush();  stderr.flush();
			return AwkInteger.ZERO;
		} finally {
			if(f != null)  f.closeAll();
		}
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソースのReader
	 * @param filename 標準入力のファイル名
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(Reader source, String filename,
			InputStream stdin, Writer stdout, Writer stderr,
			Charset inputEncoding, Map<Object, Object> vars,
			String... args) throws IOException {
		AwkNamespace r = newNamespace(filename, args);
		AwkProgram p;

		p = compile(r, source);
		return execute(r, p, filename, stdin, stdout, stderr,
				inputEncoding, vars, args);
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソースのReader
	 * @param inputFile  標準入力ファイル名
	 * @param outputFile 標準出力ファイル名
	 * @param stderr     標準エラー出力
	 * @param inputEncoding  入力の文字コード名
	 * @param outputEncoding 出力の文字コード名
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(Reader source,
			File inputFile,  File outputFile, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		Writer stdout = null;
		InputStream stdin = null;

		try {
			stdin = new FileInputStream(inputFile);
			if(outputEncoding != null) {
				stdout = new OutputStreamWriter(
						new FileOutputStream(outputFile),
						outputEncoding);
			} else {
				stdout = new OutputStreamWriter(
						new FileOutputStream(outputFile));
			}
			return execute(source, inputFile.toString(),
					stdin, stdout, stderr, inputEncoding, vars, args);
		} finally {
			if(stdin  != null)  stdin.close();
			if(stdout != null) {
				stdout.flush();
				stdout.close();
			}
		}
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			InputStream stdin, Writer stdout, Writer stderr,
			Charset inputEncoding, Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return execute(srd, "<no name>", stdin, stdout, stderr,
				inputEncoding, vars, args);
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			InputStream stdin, OutputStream stdout,
			OutputStream stderr,
			Charset inputEncoding, Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return execute(srd, "<no name>",
				stdin,
				new OutputStreamWriter(stdout),
				new OutputStreamWriter(stderr),
				inputEncoding, vars, args);
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			InputStream stdin, Writer stdout,
			Writer stderr, Charset inputEncoding) throws IOException {
		return execute(source, stdin, stdout, stderr, inputEncoding,
				new HashMap<Object, Object>());
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param inputEncoding 入力エンコード
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			InputStream stdin, OutputStream stdout,
			OutputStream stderr,
			Charset inputEncoding) throws IOException {
		return execute(source, stdin, stdout, stderr, inputEncoding,
				new HashMap<Object, Object>());
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			InputStream stdin, OutputStream stdout,
			OutputStream stderr) throws IOException {
		return execute(source, stdin, stdout, stderr,
				Charset.defaultCharset(),
				new HashMap<Object, Object>());
	}

	/**
	 * awkiumプログラムを実行します。
	 * 
	 * @param source ソース
	 * @param inputFile  標準入力ファイル名
	 * @param outputFile 標準出力ファイル名
	 * @param stderr     標準エラー出力
	 * @param inputEncoding  入力の文字コード名
	 * @param outputEncoding 出力の文字コード名
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue execute(String source,
			File inputFile, File outputFile, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return execute(srd, inputFile, outputFile, stderr,
				inputEncoding, outputEncoding, vars, args);
	}

	/**
	 * awkiumプログラムをファイル上書きで実行します。<br />
	 * fileの内容は上書きされます。
	 * 
	 * @param source ソースのReader
	 * @param file       標準入力および標準出力ファイル名
	 * @param stderr     標準エラー出力
	 * @param inputEncoding  入力の文字コード名
	 * @param outputEncoding 出力の文字コード名
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue overwrite(Reader source,
			File file, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		File tf = File.createTempFile("awkium", ".tmp");
		BufferedOutputStream ous = null;
		BufferedInputStream ins = null;
		byte[] b = new byte[4096];
		Writer stdout = null;
		InputStream stdin = null;
		AwkValue v;
		int l;

		try {
			stdin = new FileInputStream(file);
			if(outputEncoding != null) {
				stdout = new OutputStreamWriter(
						new FileOutputStream(tf), outputEncoding);
			} else {
				stdout = new OutputStreamWriter(
						new FileOutputStream(tf));
			}
			v = execute(source, file.toString(), stdin,
					stdout, stderr, inputEncoding, vars, args);
		} finally {
			if(stdin  != null)  stdin.close();
			if(stdout != null) {
				stdout.flush();
				stdout.close();
			}
		}

		try {
			ins = new BufferedInputStream(new FileInputStream(tf));
			ous = new BufferedOutputStream(
					new FileOutputStream(file));
			while((l = ins.read(b)) >= 0)  ous.write(b, 0, l);
			return v;
		} finally {
			if(ins != null)  ins.close();
			if(ous != null) {
				ous.flush();
				ous.close();
			}
			tf.delete();
		}
	}

	/**
	 * awkiumプログラムをファイル上書きで実行します。<br />
	 * fileの内容は上書きされます。
	 * 
	 * @param source ソース
	 * @param file       標準入力および標準出力ファイル名
	 * @param stderr     標準エラー出力
	 * @param inputEncoding  入力の文字コード名
	 * @param outputEncoding 出力の文字コード名
	 * @param vars   事前定義された変数
	 * @param args   引数
	 * @return 結果
	 * @throws IOException IOエラー
	 */
	public AwkValue overwrite(String source,
			File file, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return overwrite(srd, file, stderr, inputEncoding,
				outputEncoding, vars, args);
	}

	/**
	 * 使用法を出力します。
	 */
	public void usage() {
		System.err.print  ("awkium Ver. ");
		System.err.println(VERSION);
		System.err.println("usage:");
		System.err.println("awk [option] -f program [files ...]");
		System.err.println("awk [option] script [files ...]");
		System.err.println("option:");
		System.err.println("-F file-separator");
	}

	/**
	 * 引数を解析し、awkiumを実行します。
	 * 
	 * @param stdin  標準入力
	 * @param stdout 標準出力
	 * @param stderr 標準エラー出力
	 * @param args   引数
	 * @return 終了コード
	 * @throws IOException IOエラー
	 */
	public int invoke(InputStream stdin, Writer stdout,
			Writer stderr, String... args) throws IOException {
		AwkNamespace ns;
		List<String> a = new ArrayList<String>();
		List<String> f = new ArrayList<String>();
		String s, pfn = "<command line>";
		InputStream ins = null;
		Map<Object, Object> v;
		EncodingDetector en;
		boolean pre = false;
		AwkValue z = null;
		Reader r = null;
		AwkProgram p;
		int x, i = 1;
		String[] a2;
		Charset ch;

		v = new HashMap<Object, Object>();
		for(; i < args.length; i++) {
			s = args[i];
			if(s.equals("--with-cpp")) {
				pre = true;
			} else if(s.equals("-f") && i + 1 < args.length) {
				s = pfn = args[++i];
//				r = new InputStreamReader(new FileInputStream(s));
				f.add(s);
			} else if(s.equals("-F") && i + 1 < args.length) {
				v.put("FS", args[++i]);
			} else if(s.startsWith("-F")) {
				v.put("FS", s.substring(2));
			} else if(r == null) {
				r = new StringReader(s);
			} else {
				break;
			}
		}

		if(f.size() > 0)  r = FilesReader.newFiles(f);
		if(pre)  r = new CPreprocessorReader(pfn, r);
		if(args.length == 0)  throw new IllegalArgumentException();
		a.add(args[0]);
		for(; i < args.length; i++)  a.add(args[i]);

		a2 = a.toArray(new String[0]);
		ns = newNamespace("<stdin>", a2);
		en = EncodingDetectorFactory.getInstance();
		if(r == null) {
			usage();
			return 2;
		} else {
			p = compile(ns, r);
			for(int j = 1; j < a.size(); j++) {
				s = a.get(j);
				if((x = s.indexOf('=')) > 0) {
					v.put(s.substring(0, x), s.substring(x + 1));
				} else {
					try {
						ns.assign("FILENAME", AwkString.valueOf(s));
						ch  = en.detect(new File(s));
						ins = new FileInputStream(s);
						z   = execute(ns, p, s, ins, stdout, stderr,
								ch, v, a2);
					} finally {
						if(ins != null)  ins.close();
						ins = null;
					}
				}
			}

			if(z == null) {
				z = execute(ns, p, "<stdin>", stdin, stdout, stderr,
						Charset.defaultCharset(), v, a2);
			}
		}
		return z.toInteger().intValue();
	}

	/**
	 * REPL(Read-Eval-Print-Loop)を起動します。
	 */
	public void repl() {
		AwkNamespace r = newNamespace("<stdin>");
		AwkValue v = AwkUndefined.UNDEF;
		AwkFiles f = null;
		AwkExpression x;
		AwkLexer l;
		Writer so, se;
		InputStream rd;

		try {
			r.setField("");
			rd = System.in;
			so = new OutputStreamWriter(System.out);
			se = new OutputStreamWriter(System.err);
			l  = new AwkLexer("awk>", "  >>",
					new InputStreamReader(System.in));
			f  = new AwkFiles(rd, so, se);
			for(;;) {
				try {
					v = AwkUndefined.UNDEF;
					x = AwkParser.parseStatement(l);
					x = x.compileInternally();
					v = x.eval(r, f);
				} catch(AwkNextException e) {
					// do nothing
				} catch(AwkCompilationException e) {
					System.err.print  ("Error: ");
					System.err.println(e.getMessage());
				} catch(AwkException e) {
					System.err.print  ("Error: ");
					System.err.println(e.getMessage());
				} finally {
					so.flush();  se.flush();
					System.out.println(v.toString());
					l.resetPrompt();
				}
			}
		} catch(AwkExitException e) {
			System.exit(e.getValue().toInteger().intValue());
		} catch (IOException e) {
			e.printStackTrace(System.err);
			System.exit(2);
		} finally {
			if(f != null)  f.closeAll();
		}
	}

}
