/*******************************************************************************
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.engine.googletranslate;

import java.io.File;
import java.io.IOException;
import java.net.ProxySelector;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import net.sf.json.JSONArray;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.ProxySelectorRoutePlanner;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;

/**
 * このクラスは、Google Translate TM エンジン・クラスです。
 * 
 * @author IGA Tosiki
 * @author YAMAMOTO Koji
 */
public class GoogleTranslateTmEngine implements BentenTmEngine {
	/**
	 * 翻訳元の言語。
	 */
	protected String fSourceLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳先の言語。
	 */
	protected String fTargetLang = "ja-JP";//$NON-NLS-1$

	/**
	 * 翻訳言語を設定。
	 * 
	 * @param source 翻訳元の言語。例: en-US。
	 * @param target 翻訳先の言語。例: ja-JP。
	 */
	public void setLang(final String source, final String target) {
		fSourceLang = source;
		fTargetLang = target;
	}

	/**
	 * {@inheritDoc}
	 */
	public void loadTmx(final File dirTmx) throws IOException {
		// 何もしません。
	}

	/**
	 * {@inheritDoc}
	 */
	public List<BentenTmSearchResult> fuzzySearch(final String source) {
		final List<BentenTmSearchResult> result = new ArrayList<BentenTmSearchResult>();

		final BentenTmSearchResult append = new BentenTmSearchResult();
		append.setOrigin("google-translate");
		// TODO 一致率を MT (機械翻訳) の場合に何を設定すべきかの再考。
		// TODO XLIFF の state-qualifier もあわせて利用検討すべきと思われる。
		append.setMatchQuality("63%");
		append.setSource(source);

		String sourceLang = fSourceLang;
		String targetLang = fTargetLang;

		if (sourceLang.indexOf('-') > 0) {
			sourceLang = sourceLang.substring(0, sourceLang.indexOf('-'));
		}
		if (targetLang.indexOf('-') > 0) {
			targetLang = targetLang.substring(0, targetLang.indexOf('-'));
		}

		try {
			append.setTarget(json2text(translateByGoogle(sourceLang, targetLang, source)));
			result.add(append);
		} catch (IOException ex) {
			// あきらめます。
			throw new IllegalArgumentException(ex.toString());
		}

		return result;
	}

	/**
	 * Google Translate をつかって翻訳。
	 * @param sourceLang 翻訳元の言語。
	 * @param targetLang 翻訳先の言語。
	 * @param input 入力文字列。
	 * @return 翻訳後文字列。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	static String translateByGoogle(final String sourceLang, final String targetLang, final String input)
			throws IOException {
		System.out.println("trans: source[" + sourceLang + "], target[" + targetLang + "], input[" + input + "]");

		final String sourceLine = URLEncoder.encode(input, "UTF-8");

		final DefaultHttpClient httpClient = new DefaultHttpClient();
		{
			// Proxy 設定を Java 実行環境にゆだねる。
			final ProxySelectorRoutePlanner routePlanner = new ProxySelectorRoutePlanner(httpClient
					.getConnectionManager().getSchemeRegistry(), ProxySelector.getDefault());
			httpClient.setRoutePlanner(routePlanner);
		}

		final String uri = "http://translate.google.co.jp/translate_a/t?client=t&text=" + sourceLine + "&hl=ja"
				+ "&sl=" + sourceLang + "&tl=" + targetLang + "&pc=0";

		final HttpGet httpGet = new HttpGet(uri);

		httpGet.setHeader("User-Agent", "Mozilla/5.0 (Java) Benten Ext.");
		httpGet.setHeader("Accept-Language", "en-us");
		httpGet.setHeader("Accept-Charset", "UTF-8");

		final BasicResponseHandler responseHandler = new BasicResponseHandler();

		return httpClient.execute(httpGet, responseHandler);
	}

	/**
	 * JSON 検索結果をプレーンテキストに変換します。
	 * @param input 入力 JSON 文字列。
	 * @return プレーンな出力文字列。
	 */
	static String json2text(final String input) {
		final JSONArray sentences = JSONArray.fromObject(input);

		final JSONArray inner1 = (JSONArray) sentences.get(0);
		final JSONArray inner2 = (JSONArray) inner1.get(0);
		final String obj = (String) inner2.get(0);

		return obj;
	}

	/**
	 * {@inheritDoc}
	 */
	public void unload() {
		// 何もしません。
	}
}
