/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.engine.regex;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.glossary.core.BentenGlossarySearchResult;
import blanco.commons.util.BlancoNameUtil;

/**
 * 正規表現用語集エンジン。
 *
 * <UL>
 * <LI>このクラスは、用語集エンジンの切り替えテストのためのテストクラスです。
 * <LI>実際の Benten 基本セットには含まれません。
 * </UL>
 * 
 * @author IGA Tosiki
 */
public class RegexGlossaryEngine implements BentenGlossaryEngine {
	/**
	 * このクラスが保持している用語集
	 * 
	 * 検索結果クラスを利用していますが、これは検索結果ではありません。
	 */
	private final List<BentenGlossarySearchResult> fGlossaryList = Collections
			.synchronizedList(new ArrayList<BentenGlossarySearchResult>());
	/**
	 * 翻訳元の言語。
	 */
	protected String fSourceLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳先の言語。
	 */
	protected String fTargetLang = "ja-JP";//$NON-NLS-1$

	/**
	 * 基軸言語。
	 */
	protected String fPivotLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳言語を設定。
	 * 
	 * @param source 翻訳元の言語。例: en-US。
	 * @param target 翻訳先の言語。例: ja-JP。
	 */
	public void setLang(final String source, final String target) {
		fSourceLang = source;
		fTargetLang = target;
	}

	/**
	 * 基軸言語を設定。
	 * 
	 * @param pivotLang 基軸言語。例: en-US。
	 */
	public void setPivotLang(final String pivotLang) {
		fPivotLang = pivotLang;
	}

	/**
	 * {@inheritDoc}
	 */
	public void loadTsv(final File glossaryDirectory) throws IOException {
		final File[] files = glossaryDirectory.listFiles();
		if (files == null) {
			// 用語集ディレクトリにひとつもファイルがありません。
			return;
		}

		for (File file : files) {
			if (file.isFile() == false) {
				continue;
			}
			if (file.canRead() == false) {
				continue;
			}
			if (file.getName().toLowerCase().endsWith(".tsv") == false) { //$NON-NLS-1$
				continue;
			}

			// ロード処理。
			final BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8")); //$NON-NLS-1$
			for (;;) {
				final String line = reader.readLine();
				if (line == null) {
					break;
				}

				final String[] blocks = BlancoNameUtil.splitString(line, '\t'); //$NON-NLS-1$
				if (blocks.length < 2) {
					continue;
				}

				final BentenGlossarySearchResult item = new BentenGlossarySearchResult();
				fGlossaryList.add(item);

				if (fPivotLang.compareToIgnoreCase(fSourceLang) == 0) {
					// 基軸言語とソースが一致する。
					item.setSource(blocks[0]);
					item.setTarget(blocks[1]);
				} else {
					// 基軸言語とソースが一致しない。
					// これは、TSV としては逆転モード。左辺が target で右辺が source。
					item.setTarget(blocks[0]);
					item.setSource(blocks[1]);
				}

				if (blocks.length >= 3) {
					item.setNote(blocks[2]);
				}
			}
			reader.close();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void unload() {
		fGlossaryList.clear();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<BentenGlossarySearchResult> search(final String target) {
		final List<BentenGlossarySearchResult> list = new ArrayList<BentenGlossarySearchResult>();

		String regPatternString;
		boolean isRegexMode = false;
		if (target.startsWith("!!")) { //$NON-NLS-1$
			// !! の場合に、特殊モードに入ります。
			isRegexMode = true;
			regPatternString = target.substring(2);
		} else {
			regPatternString = ".*" + Pattern.quote(target.toLowerCase()) + ".*"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		try {
			final Pattern regPattern = Pattern.compile(regPatternString);

			for (BentenGlossarySearchResult item : fGlossaryList) {
				String searchString = null;
				if (isRegexMode) {
					searchString = item.getSource();
				} else {
					searchString = item.getSource().toLowerCase();
				}

				final Matcher regMatcher = regPattern.matcher(searchString);
				if (regMatcher.matches()) {
					list.add(item);
				}
			}
		} catch (PatternSyntaxException ex) {
			System.out.println("PatternSyntaxException occurs: " + ex.toString()); //$NON-NLS-1$
		}

		// 文字列長さでソート。
		Collections.sort(list, new Comparator<BentenGlossarySearchResult>() {
			public int compare(final BentenGlossarySearchResult tuLeft, final BentenGlossarySearchResult tuRight) {
				String left = tuLeft.getSource();
				String right = tuRight.getSource();
				return left.length() - right.length();
			}
		});

		return list;
	}
}
