/*
 * blanco Framework
 * Copyright (C) 2004-2006 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.dbmetadata;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;

import blanco.commons.util.BlancoStringUtil;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataColumnStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataKeyStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataTableStructure;

/**
 * f[^x[X烁^擾郆[eBeBB
 * 
 * \Ɋւ鏈܂܂܂B
 * 
 * @author IGA Tosiki
 */
public class BlancoDbMetaDataTable {
    private static final boolean IS_REPORT_COLUMN_NOT_FOUND = false;

    /**
     * f[^x[X̑SĂ̕\ɂāAځEvC}L[Ȃǂ̏tŎ擾܂B
     * 
     * @param conn
     *            f[^x[XRlNVB
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @param argTypes
     *            擾ނ̈ꗗBnullȂSāBnew String[] { "TABLE" }̂悤ɎwB
     * @return \̈ꗗB
     * @throws SQLException
     */
    public static ArrayList getTablesWithColumns(final Connection conn,
            final String argSchema, final String argTable,
            final String[] argTypes) throws SQLException {
        final DatabaseMetaData metadata = conn.getMetaData();

        // \擾܂B
        final ArrayList listTables = BlancoDbMetaDataTable.getTables(metadata,
                argSchema, argTable, argTypes);

        boolean isImportedKeysAvailable = true;
        boolean isExportedKeysAvailable = true;
        boolean isCrossReferenceKeysAvailable = true;

        // \ɂ܂SẴ^擾܂B
        for (int indexTable = 0; indexTable < listTables.size(); indexTable++) {
            final BlancoDbMetaDataTableStructure tableStructure = (BlancoDbMetaDataTableStructure) listTables
                    .get(indexTable);

            tableStructure.setColumns(BlancoDbMetaDataTable.getColumns(
                    metadata, tableStructure.getCatalog(), argSchema,
                    tableStructure.getName()));

            tableStructure.setPrimaryKeys(BlancoDbMetaDataTable.getPrimaryKeys(
                    metadata, tableStructure.getCatalog(), argSchema,
                    tableStructure.getName()));

            try {
                tableStructure.setImportedKeys(BlancoDbMetaDataTable
                        .getImportedKeys(metadata, tableStructure.getCatalog(),
                                argSchema, tableStructure.getName()));
            } catch (SQLException ex) {
                if (isImportedKeysAvailable) {
                    isImportedKeysAvailable = false;
                    System.out.println("C|[gL[擾ɗO:" + ex.toString());
                }
            }

            try {
                tableStructure.setExportedKeys(BlancoDbMetaDataTable
                        .getExportedKeys(metadata, tableStructure.getCatalog(),
                                argSchema, tableStructure.getName()));
            } catch (SQLException ex) {
                if (isExportedKeysAvailable) {
                    isExportedKeysAvailable = false;
                    System.out.println("GNX|[gL[擾ɗO:" + ex.toString());
                }
            }

            try {
                tableStructure.setCrossReferenceKeys(BlancoDbMetaDataTable
                        .getCrossReference(metadata, tableStructure
                                .getCatalog(), argSchema, tableStructure
                                .getName(), null, null, null));
            } catch (SQLException ex) {
                if (isCrossReferenceKeysAvailable) {
                    isCrossReferenceKeysAvailable = false;
                    System.out.println("NXQƎ擾ɗO:" + ex.toString());
                }
            }
        }

        return listTables;
    }

    private static String getStringSafety(final ResultSet resultSet,
            final String fieldName) {
        try {
            return resultSet.getString(fieldName);
        } catch (SQLException ex) {
            if (IS_REPORT_COLUMN_NOT_FOUND) {
                System.out.println("ResultSet當擾ۂɗOB:"
                        + ex.toString());
            }
            return null;
        }
    }

    /**
     * f[^x[X^񂩂\̈ꗗ擾܂B
     * 
     * \ $ ܂܂̂͏XLbv܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @param argTypes
     *            擾ނ̈ꗗBnullȂSāBnew String[] { "TABLE" }̂悤ɎwB
     * @return \̈ꗗB
     * @throws SQLException
     */
    public static ArrayList getTables(final DatabaseMetaData metadata,
            final String argSchema, final String argTable,
            final String[] argTypes) throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getTables(null, argSchema,
                argTable, argTypes);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataTableStructure tableStructure = new BlancoDbMetaDataTableStructure();

                tableStructure.setName(resultSet.getString("TABLE_NAME"));

                // if (tableStructure.getName().toLowerCase().startsWith("sys"))
                // {
                // continue;
                // }

                // System.out.println("\[" + tableStructure.getName() + "]");

                tableStructure.setType(resultSet.getString("TABLE_TYPE"));

                tableStructure.setCatalog(getStringSafety(resultSet,
                        "TABLE_CAT"));
                tableStructure.setSchema(getStringSafety(resultSet,
                        "TABLE_SCHEM"));
                tableStructure
                        .setRemarks(getStringSafety(resultSet, "REMARKS"));

                if (BlancoStringUtil.null2Blank(tableStructure.getName())
                        .length() == 0) {
                    // \̂͏XLbv܂B
                    continue;
                }
                if (BlancoStringUtil.null2Blank(tableStructure.getType())
                        .length() == 0) {
                    // \^Cv̂͏XLbv܂B
                    continue;
                }
                if (tableStructure.getName().indexOf('$') >= 0) {
                    // \ $ ܂܂Ă̂͏XLbv܂B
                    continue;
                }

                result.add(tableStructure);
            }
        } finally {
            resultSet.close();
        }

        return result;
    }

    /**
     * f[^x[X^񂩂\̃Jꗗ擾܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @return ڂ̈ꗗ߂B
     * @throws SQLException
     */
    public static ArrayList getColumns(final DatabaseMetaData metadata,
            final String argCat, final String argSchema,
            final String argTableName) throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getColumns(argCat, argSchema,
                argTableName, null);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataColumnStructure columnStructure = new BlancoDbMetaDataColumnStructure();

                columnStructure.setName(resultSet.getString("COLUMN_NAME"));

                columnStructure.setDataType(resultSet.getShort("DATA_TYPE"));
                if (columnStructure.getDataType() == Types.OTHER) {
                    BlancoDbMetaDataUtil.mapTypeName2DataType(columnStructure);
                }

                columnStructure.setDataTypeDisplayName(BlancoDbMetaDataUtil
                        .convertJdbcDataTypeToString(columnStructure
                                .getDataType()));
                columnStructure.setTypeName(getStringSafety(resultSet,
                        "TYPE_NAME"));
                try {
                    columnStructure.setColumnSize(resultSet
                            .getInt("COLUMN_SIZE"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                columnStructure.setBufferLength(getStringSafety(resultSet,
                        "BUFFER_LENGTH"));
                try {
                    columnStructure.setDecimalDigits(resultSet
                            .getInt("DECIMAL_DIGITS"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                try {
                    columnStructure.setNumPrecRadix(resultSet
                            .getInt("NUM_PREC_RADIX"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                try {
                    columnStructure.setNullable(resultSet.getInt("NULLABLE"));
                    columnStructure.setNullableDisplayName(BlancoDbMetaDataUtil
                            .convertJdbcNullableToString(columnStructure
                                    .getNullable()));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }

                columnStructure
                        .setRemarks(getStringSafety(resultSet, "REMARKS"));
                columnStructure.setColumnDef(getStringSafety(resultSet,
                        "COLUMN_DEF"));

                try {
                    columnStructure.setSqlDataType(resultSet
                            .getInt("SQL_DATA_TYPE"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                try {
                    columnStructure.setSqlDatetimeSub(resultSet
                            .getInt("SQL_DATETIME_SUB"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                try {
                    columnStructure.setCharOctetLength(resultSet
                            .getInt("CHAR_OCTET_LENGTH"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                try {
                    columnStructure.setOrdinalPosition(resultSet
                            .getInt("ORDINAL_POSITION"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetint擾ۂɗOB:"
                                + ex.toString());
                    }
                }
                columnStructure.setIsNullable(getStringSafety(resultSet,
                        "IS_NULLABLE"));
                columnStructure.setScopeCatlog(getStringSafety(resultSet,
                        "SCOPE_CATLOG"));
                columnStructure.setScopeSchema(getStringSafety(resultSet,
                        "SCOPE_SCHEMA"));
                columnStructure.setScopeTable(getStringSafety(resultSet,
                        "SCOPE_TABLE"));
                try {
                    columnStructure.setSourceDataType(resultSet
                            .getShort("SOURCE_DATA_TYPE"));
                } catch (SQLException ex) {
                    if (IS_REPORT_COLUMN_NOT_FOUND) {
                        System.out.println("ResultSetshort擾ۂɗOB:"
                                + ex.toString());
                    }
                }

                result.add(columnStructure);
            }
        } finally {
            resultSet.close();
        }
        return result;
    }

    /**
     * f[^x[X^񂩂\̃vC}L[擾܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @return vC}L[̈ꗗ߂B
     * @throws SQLException
     */
    public static ArrayList getPrimaryKeys(final DatabaseMetaData metadata,
            final String argCat, final String argSchema,
            final String argTableName) throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getPrimaryKeys(argCat, argSchema,
                argTableName);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataKeyStructure primaryKeyStructure = new BlancoDbMetaDataKeyStructure();

                primaryKeyStructure.setPkcolumnName(resultSet
                        .getString("COLUMN_NAME"));
                primaryKeyStructure.setKeySeq(resultSet.getShort("KEY_SEQ"));
                primaryKeyStructure.setPkName(getStringSafety(resultSet,
                        "PK_NAME"));

                result.add(primaryKeyStructure);
            }
        } finally {
            resultSet.close();
        }
        return result;
    }

    /**
     * f[^x[X^񂩂\̊OL[QƂL[Ɋւ擾܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @return L[̈ꗗ߂B
     * @throws SQLException
     */
    public static ArrayList getImportedKeys(final DatabaseMetaData metadata,
            final String argCat, final String argSchema,
            final String argTableName) throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getImportedKeys(argCat, argSchema,
                argTableName);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataKeyStructure importedKeyStructure = getKeyStructureFromResultSet(resultSet);
                result.add(importedKeyStructure);
            }
        } finally {
            resultSet.close();
        }
        return result;
    }

    /**
     * f[^x[X^񂩂\̊OL[QƂL[Ɋւ擾܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argTable
     *            \Bw肵Ȃꍇɂnull^B
     * @return L[̈ꗗ߂B
     * @throws SQLException
     */
    public static ArrayList getExportedKeys(final DatabaseMetaData metadata,
            final String argCat, final String argSchema,
            final String argTableName) throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getExportedKeys(argCat, argSchema,
                argTableName);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataKeyStructure exportedKeyStructure = getKeyStructureFromResultSet(resultSet);
                result.add(exportedKeyStructure);
            }
        } finally {
            resultSet.close();
        }
        return result;
    }

    /**
     * f[^x[X^񂩂\̊OL[Ɋւ擾܂B
     * 
     * @param metadata
     *            f[^x[X^f[^B
     * @param argPrimaryCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argPrimarySchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argPrimaryTableName
     *            \Bw肵Ȃꍇɂnull^B
     * @param argForeignCat
     *            J^OBw肵Ȃꍇɂnull^B
     * @param argForeignSchema
     *            XL[}Bw肵Ȃꍇɂnull^B
     * @param argForeignTableName
     *            \Bw肵Ȃꍇɂnull^B
     * @return L[̈ꗗ߂B
     * @throws SQLException
     */
    public static ArrayList getCrossReference(final DatabaseMetaData metadata,
            final String argPrimaryCat, final String argPrimarySchema,
            final String argPrimaryTableName, final String argForeignCat,
            final String argForeignSchema, final String argForeignTableName)
            throws SQLException {
        final ArrayList result = new ArrayList();

        final ResultSet resultSet = metadata.getCrossReference(argPrimaryCat,
                argPrimarySchema, argPrimaryTableName, argForeignCat,
                argForeignSchema, argForeignTableName);
        try {
            for (; resultSet.next();) {
                final BlancoDbMetaDataKeyStructure exportedKeyStructure = getKeyStructureFromResultSet(resultSet);
                result.add(exportedKeyStructure);
            }
        } finally {
            resultSet.close();
        }
        return result;
    }

    /**
     * ResultSetL[\擾܂B
     * 
     * getImportedKeys, getExportedKeysȂǂResultSetł邱ƂAʉĂ܂B
     * 
     * @param resultSet
     * @return
     * @throws SQLException
     */
    private static BlancoDbMetaDataKeyStructure getKeyStructureFromResultSet(
            final ResultSet resultSet) throws SQLException {
        final BlancoDbMetaDataKeyStructure keyStructure = new BlancoDbMetaDataKeyStructure();

        keyStructure.setPktableCat(resultSet.getString("PKTABLE_CAT"));
        keyStructure.setPktableSchem(resultSet.getString("PKTABLE_SCHEM"));
        keyStructure.setPktableName(resultSet.getString("PKTABLE_NAME"));
        keyStructure.setPkcolumnName(resultSet.getString("PKCOLUMN_NAME"));
        keyStructure.setFktableCat(resultSet.getString("FKTABLE_CAT"));
        keyStructure.setFktableSchem(resultSet.getString("FKTABLE_SCHEM"));
        keyStructure.setFktableName(resultSet.getString("FKTABLE_NAME"));
        keyStructure.setFkcolumnName(resultSet.getString("FKCOLUMN_NAME"));
        keyStructure.setKeySeq(resultSet.getShort("KEY_SEQ"));

        try {
            keyStructure.setUpdateRule(resultSet.getShort("UPDATE_RULE"));
        } catch (SQLException ex) {
            if (IS_REPORT_COLUMN_NOT_FOUND) {
                System.out.println("ResultSetshort擾ۂɗOB:"
                        + ex.toString());
            }
        }
        try {
            keyStructure.setDeleteRule(resultSet.getShort("DELETE_RULE"));
        } catch (SQLException ex) {
            if (IS_REPORT_COLUMN_NOT_FOUND) {
                System.out.println("ResultSetshort擾ۂɗOB:"
                        + ex.toString());
            }
        }
        keyStructure.setFkName(getStringSafety(resultSet, "FK_NAME"));
        keyStructure.setPkName(getStringSafety(resultSet, "PK_NAME"));
        try {
            keyStructure.setDeferrability(resultSet.getShort("DEFERRABILITY"));
        } catch (SQLException ex) {
            if (IS_REPORT_COLUMN_NOT_FOUND) {
                System.out.println("ResultSetshort擾ۂɗOB:"
                        + ex.toString());
            }
        }

        return keyStructure;
    }
}
