require 'net/https'
require 'uri'
require 'yaml'
require 'rexml/document'
require 'fileutils'
require 'find'
require 'tempfile'
require 'cgi'
require 'xmlrpc/client'
begin
  require 'rubygems'
rescue LoadError
end

require 'bloggerpost/base.rb'
require 'bloggerpost/config.rb'
require 'bloggerpost/service.rb'
require 'bloggerpost/entry.rb'

Version = '0.1.8'

class String
  def is_binary_data?
    false
  end
end

module BloggerPost
  PATH = File::dirname(__FILE__)

  @@service_list = []
  @@style_list = []
  @@filter_list = []

  def self.register_service(service_class)
    @@service_list << service_class unless @@service_list.include?(service_class)
  end

  def self.register_style(style_class)
    @@style_list << style_class unless @@style_list.include?(style_class)
  end

  def self.register_filter(filter_class)
    @@filter_list << filter_class unless @@filter_list.include?(filter_class)
  end

  def self.load_service
    Find.find(self::PATH + '/bloggerpost/service') do |service_file|
      require File.expand_path(service_file) if service_file =~ /\.rb$/
    end
  end

  def self.load_style
    Find.find(self::PATH + '/bloggerpost/style') do |style_file|
      require File.expand_path(style_file) if style_file =~ /\.rb$/
    end
  end

  def self.load_filter
    Find.find(self::PATH + '/bloggerpost/filter') do |filter_file|
      require File.expand_path(filter_file) if filter_file =~ /\.rb$/
    end
    self.register_filter(PreFilter)
  end

  def self.service_new(config, service_name)
    @@service_list.each do |service_class|
      return service_class.new(config) if service_class.service_name == service_name
    end
    raise "The #{service_name} service is not found."
  end

  def self.style_new(style_name)
    @@style_list.each do |style_class|
      return style_class.new if style_class.style_name == style_name
    end
    raise "The #{style_name} style is not found."
  end

  def self.style_new_filename(filename)
    @@style_list.each do |style_class|
      return style_class.new if style_class.extname == File.extname(filename)
    end
    raise "The #{filename} style is not found."
  end

  def self.each_filter(config, &block)
    @@filter_list.each do |filter_class|
      block.call(filter_class.new(config))
    end
  end

  class Request
    def initialize(config)
      @config = config
    end

    def new_post(entry)
      api = entry.blog.service.api
      result = api.new_post(entry)
      entry.dump_backup
      entry.dump_cache
      send_trackback(entry)
      dump_log(entry, result)
      entry.status = STATUS_POSTED
      entry.save
      send_ping(entry)
      result
    end

    def edit_post(entry)
      raise 'The entry is not posted.' unless entry.posted?
      api = entry.blog.service.api
      result = api.edit_post(entry)
      entry.dump_backup
      entry.dump_cache
      dump_log(entry, result)
      entry.status = STATUS_POSTED
      entry.save
      result
    end

    def delete_post(entry)
      raise 'The entry is not posted.' unless entry.posted?
      api = entry.blog.service.api
      result = api.delete_post(entry)
      dump_log(entry, result)
      entry.status = STATUS_DELETED
      entry.save
      result
    end

    private

    def make_log(entry, result)
      entry_hash = {'method' => result.req_method }
      entry_hash['editid'] = entry.edit_id
      entry_hash['blog'] = entry.blog.blog_name
      if result.req_method == 'delete'
        entry_hash['date'] = entry.updated.to_s
      else
        entry_hash['date'] = entry.date.to_s
        entry_hash['updated'] = entry.updated.to_s
        entry_hash['title'] = entry.title
        entry_hash['tag'] = entry.tag_list.collect { |tag| tag.toutf8 }
        entry_hash['category'] = entry.category_list.collect { |category| category.toutf8 }
        entry_hash['trackback'] = entry.trackback_list
        entry_hash['style'] = entry.style.style_name
        entry_hash['url'] = entry.url
      end
      yaml_decode(YAML.dump(entry_hash))
    end

    def dump_log(entry, result)
      File.open(@config.log_file, 'a') do |file|
        file.write(make_log(entry, result))
      end
    end

    def send_trackback(entry)
      entry.trackback_list.each do |trackback|
        url = URI.parse(trackback['url'])
        req = Net::HTTP::Post.new(url.path)
        req.form_data = {
          'url' => entry.url,
          'title' => entry.title,
          'blog_name' => entry.blog.title,
          'excerpt' => entry.excerpt
        }
        res = http_request(@config, url, req)
        if res.code == '200'
          doc = REXML::Document.new(res.body)
          error_element = doc.elements['/response/error']
          if error_element && error_element.text == '0'
            trackback['status'] = 'sent'
          else
            trackback['status'] = 'failed'
          end
        else
          trackback['status'] = 'failed'
        end
      end
    end

    def send_ping(entry)
      entry.blog.ping_list.each do |ping_url|
        uri = URI.parse(ping_url)
        server = XMLRPC::Client.new(uri.host, uri.path)
        result = server.call("weblogUpdates.ping", entry.blog.title, entry.blog.url)
        puts "ping: " + uri.host + ": " + result["message"]
      end
    end

    def parse_atom(src)
      doc = REXML::Document.new(src)
      published_element = doc.elements['/entry/published']
      published_element = doc.elements['/entry/issued'] unless published_element
      published_element = doc.elements['/entry/created'] unless published_element
      updated_element = doc.elements['/entry/updated']
      updated_element = doc.elements['/entry/modified'] unless updated_element
      updated_element = published_element unless updated_element
      edit_url_element = doc.elements["/entry/link[@rel='service.edit']"]
      edit_url_element = doc.elements["/entry/link[@rel='edit']"] unless edit_url_element
      if edit_url_element
        edit_url = edit_url_element.attributes['href']
      else
        edit_url = nil
      end
      [published_element.text, updated_element.text, doc.elements["/entry/link[@rel='alternate']"].attributes['href'], edit_url]
    end

    def yaml_decode(src)
      src.gsub(/\\x(\w{2})/) { [Regexp.last_match.captures.first.to_i(16)].pack("C") }
    end
  end

  class Result
    attr_reader :req_method, :api_type

    def initialize(req_method, api_type)
      @req_method = req_method # 'new' or 'edit' or 'delete'
      @api_type = api_type # 'xmlrpc' or 'atom'
    end
  end

  class Blog
    attr_reader :blog_name, :title, :url, :user, :passwd, :service, :post_url, :api_url, :blog_id, :ping_list

    def self.new_config(config, blog_name)
      title = nil
      user = nil
      passwd = nil
      service_name = nil
      url = nil
      post_url = nil
      api_url = nil
      blog_id = nil
      ping_list = []
      config.each_blog do |blog_config|
        next unless blog_config['name'] == blog_name
        title = blog_config['title'].toutf8
        url = blog_config['url']
        service_name = blog_config['service']
        post_url = blog_config['posturl'] if blog_config['posturl']
        api_url = blog_config['apiurl'] if blog_config['apiurl']
        blog_id = blog_config['blogid'] if blog_config['blogid']
        config.account_list.each do |account|
          next unless account['name'] == blog_config['account']
          user = account['user']
          passwd = account['passwd']
          break
        end
        config.ping_list.each do |ping|
          next unless ping['name'] == blog_config['ping']
          ping_list = ping['list']
          break
        end
      end
      self.new(config, blog_name, title, url, user, passwd, service_name, post_url, api_url, blog_id, ping_list)
    end

    def initialize(config, blog_name, title, url, user, passwd, service_name, post_url, api_url, blog_id, ping_list)
      @config = config
      @blog_name = blog_name
      @title = title
      @url = url
      @user = user
      @passwd = passwd
      @service = BloggerPost.service_new(@config, service_name) if service_name
      @post_url = post_url
      @api_url = api_url
      @blog_id = blog_id
      @ping_list = ping_list
    end
  end

  class Style
    include BloggerPost::Common

    def self.style_name
      'none'
    end

    def self.extname
      ''
    end

    def style_name
      self.class.style_name
    end

    def extname
      self.class.extname
    end

    def entry_to_html(entry)
      entry.body
    end

    def delete_lf?
      true
    end

    private

    def external_convert(command, src)
      tempfile = make_tempfile_in(src)
      `#{command} #{tempfile.path}`
    end

    def tidy_html(html)
      external_convert('tidy -q -raw -asxhtml', html)
    end

    def escape_pre_text(html)
      html = html.gsub(/<pre>(.*)<\/pre>/m) do |pattern|
        '<pre>' + CGI.escapeHTML($1) + '</pre>'
      end
      html
    end

    def wrap_html(title, body)
      "<html><head><title>#{title}</title></head><body>#{body}</body></html>"
    end

    def wrap_html_first_h1(body)
      title = get_h1_text(body)
      body = delete_first_h1(body)
      wrap_html(title, body)
    end
  end

  class Filter
    include BloggerPost::Common

    def initialize(config)
      @config = config
    end

    def translate(html)
      html
    end

    private

    def filter_inline(html, filtername, &block)
      html = html.gsub(/([^:])#{filtername}/) do
        $1 + block.call
      end
    end

    def filter_inline_text(html, filtername, &block)
      html = html.gsub(/[\(\[\{]#{filtername}\s+(.*?)\s*bp[\)\]\}]/m) do
        block.call(CGI.unescapeHTML($1))
      end
    end

    def filter_inline_attr(html, filtername, &block)
      html = html.gsub(/([^:])#{filtername}:([^\s<]*)/) do
        $1 + block.call($2)
      end
    end

    def filter_inline_text_attr(html, filtername, &block)
      html = html.gsub(/[\(\[\{]#{filtername}:([^\s<]*)\s*(.*?)\s*bp[\)\]\}]/m) do
        block.call(CGI.unescapeHTML($2), $1)
      end
    end

    def filter_pre(html, filtername, &block)
      html = html.gsub(/<pre>\s*#{filtername}\s*<\/pre>/m) do
        block.call
      end
    end

    def filter_pre_text(html, filtername, &block)
      html = html.gsub(/<pre>\s*#{filtername}\s*\n(.*?)<\/pre>/m) do
        block.call(CGI.unescapeHTML($1))
      end
    end

    def filter_pre_attr(html, filtername, &block)
      html = html.gsub(/<pre>\s*#{filtername}\s*<\/pre>/m) do
        block.call
      end
    end

    def filter_pre_text_attr(html, filtername, &block)
      html = html.gsub(/<pre>\s*#{filtername}:([^\s<]*)\s*\n(.*?)<\/pre>/m) do
        block.call(CGI.unescapeHTML($2), $1)
      end
    end

    def filter(filter_list, html, filtername, &block)
      filter_list.inject(html) {|result, item| send(item, result, filtername, &block) }
    end

    def filter_attr(html, filtername, &block)
      filter([:filter_inline_attr, :filter_pre_attr], html, filtername, &block)
    end

    def filter_text(html, filtername, &block)
      filter([:filter_inline_text, :filter_pre_text], html, filtername, &block)
    end

    def filter_text_attr(html, filtername, &block)
      filter([:filter_inline_text_attr, :filter_pre_text_attr], html, filtername, &block)
    end
  end

  class PreFilter < Filter
    def translate(html)
      html = filter_inline_attr(html, 'bppre') do |param|
        CGI.escapeHTML(param)
      end
      html = filter_inline_text_attr(html, 'bppre') do |src, param|
        CGI.escapeHTML("[#{param} #{src} bp]")
      end
      html = filter_pre_text(html, 'bppre') do |src|
        '<pre>' + CGI.escapeHTML(src) + '</pre>'
      end
      html
    end
  end
end

BloggerPost.load_service
BloggerPost.load_style
BloggerPost.load_filter
