require 'depot'

module BloggerPost
  DEFAULT_DIR = '~/.bloggerpost'
  DEFAULT_CONFIG_FILE = 'config.yml'
  DEFAULT_LOG_FILE = 'log.yml'
  DEFAULT_DB_FILE = 'db'
  DEFAULT_BACKUP_DIR = 'backup'
  DEFAULT_CACHE_DIR = 'cache'
  DEFAULT_DRAFT_DIR = 'draft'
  DEFAULT_PREVIEW_FILE = 'preview.html'

  class Config
    attr_reader :log_file, :backup_dir, :cache_dir, :work_dir, :config_file_hash, :use_proxy
    attr_reader :http_proxy_host, :http_proxy_port, :http_proxy_user, :http_proxy_passwd
    attr_reader :https_proxy_host, :https_proxy_port, :https_proxy_user, :https_proxy_passwd
    attr_reader :default_blog, :account_list, :ping_list
    attr_reader :draft_dir, :default_style_name
    attr_reader :preview_file

    def initialize
      @work_dir = DEFAULT_DIR
      @work_dir = File.expand_path(@work_dir)
      @work_dir << '/'
      @config_file = @work_dir + DEFAULT_CONFIG_FILE
      @config_file = File.expand_path(@config_file)
      load_config
      @log_file = @work_dir + DEFAULT_LOG_FILE
      @log_file = File.expand_path(@log_file)
      @db_file = @work_dir + DEFAULT_DB_FILE
      @db_file = File.expand_path(@db_file)
      @backup_dir = @work_dir + DEFAULT_BACKUP_DIR
      @backup_dir = File.expand_path(@backup_dir)
      @cache_dir = @work_dir + DEFAULT_CACHE_DIR
      @cache_dir = File.expand_path(@cache_dir)
      @draft_dir = @work_dir + DEFAULT_DRAFT_DIR
      @draft_dir = File.expand_path(@draft_dir)
      @preview_file = @work_dir + DEFAULT_PREVIEW_FILE
      @preview_file = File.expand_path(@preview_file)
      FileUtils.makedirs(@draft_dir)
      unless File.exist?(@db_file)
        db_open_writer {}
      end
    end

    def each_blog(&block)
      return unless @blog_info
      @blog_info['list'].each {|blog| block.call(blog) }
    end

    def blog
      blog_by_name(@blog_name)
    end

    def db_open_reader(&block)
      Depot::new(@db_file, Depot::OREADER) do |db|
        block.call(db)
      end
    end

    def db_open_writer(&block)
      Depot::new(@db_file, Depot::OWRITER | Depot::OCREAT) do |db|
        block.call(db)
      end
    end

    private

    def blog_by_name(name)
      each_blog do |blog|
        return blog if blog['name'] == name
      end
      nil
    end
      
    def load_config
      @config_file_hash = YAML.load_file(@config_file)
      @account_list = @config_file_hash['account']
      @account_list = [] unless @account_list
      @ping_list = @config_file_hash['ping']
      @ping_list = [] unless @ping_list
      @blog_info = @config_file_hash['blog']
      if @blog_info
        @default_blog = Blog.new_config(self, @blog_info['default']) if @blog_info['default']
      end
      proxy_hash = @config_file_hash['proxy']
      if proxy_hash
        @use_proxy = proxy_hash['use']
        http_proxy_hash = proxy_hash['http']
        @http_proxy_host = http_proxy_hash['host']
        @http_proxy_port = http_proxy_hash['port']
        @http_proxy_user = http_proxy_hash['user']
        @http_proxy_passwd = http_proxy_hash['passwd']
        https_proxy_hash = proxy_hash['https']
        @https_proxy_host = https_proxy_hash['host']
        @https_proxy_port = https_proxy_hash['port']
        @https_proxy_user = https_proxy_hash['user']
        @https_proxy_passwd = https_proxy_hash['passwd']
      else
        @use_proxy = false
      end
      @default_style_name = @config_file_hash['defaultstyle']
    end

    def load_log
      unless @yaml_cache
        @yaml_cache = []
        YAML.each_document(File.read(@log_file)) do |log|
          @yaml_cache << log
        end
      end
    end
  end
end
