/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;

/**
 * Indicates that a field is an element of enumerations.
 * 
 * <p>
 * If a field annotated as {@code @ASN1Enumeration} is present in a definition
 * of a sub-class of {@code INTEGER} or {@code ENUMERATED}, value that can be
 * contained in the type is limited to values of {@code @ASN1Enumeration}
 * fields. An {@code @ASN1Enumeration} field must be a {@code public}
 * {@code static} {@code final} field and the type shall be {@code int},
 * {@code long} or {@code BigInteger}. Each value of {@code @ASN1Enumeration}
 * fields appearing in definition of a type must be distinct from the others. A
 * sub-class of {@code ENUMERATED} must have at least one
 * {@code @ASN1Enumeration} field.
 * </p>
 * <p>
 * For example, following ASN.1 definition :
 * 
 * <pre>
 * Sample ::= ENUMERATED {
 * 	zero(0),
 * 	one(1) }
 * </pre>
 * 
 * can be translated as :
 * 
 * <code><pre>
 * public class Sample extends ENUMERATED {
 * 
 * 	&#064;ASN1Enumeration
 * 	public static final int zero = 0;
 * 
 * 	&#064;ASN1Enumeration
 * 	public static final int one = 1;
 * 
 * }
 * </pre></code>
 * 
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see INTEGER
 * @see ENUMERATED
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ASN1Enumeration {
}
