/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

import jp.bitmeister.asn1.annotation.ASN1Alternative;
import jp.bitmeister.asn1.annotation.ASN1Element;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;

/**
 * Specifications of an element that contained in a {@code StructuredType}.
 * 
 * <p>
 * An instance of this class contains static information of an element of a
 * structured type which applied by annotations and type definitions. Each
 * element is a {@code Field} object declared in a {@code StructuredType} and
 * annotated as {@code @ASN1Element} or {@code @ASN1Alternative}.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Element
 * @see ASN1Alternative
 * @see StructuredType
 */
public class NamedTypeSpecification implements
		Comparable<NamedTypeSpecification> {

	/**
	 * The field annotated as {@code @ASN1Element} or {@code @ASN1Alternative}.
	 */
	private Field field;

	/**
	 * ASN.1 identifier of this element.
	 */
	private String identifier;

	/**
	 * ASN.1 tag of this element.
	 */
	private ASN1TagValue tag;

	/**
	 * Appearing order of this element.
	 */
	private int order;

	/**
	 * Instantiate a {@code NamedTypeSpecification} from the field.
	 * 
	 * @param field
	 *            The field annotated as {@code @ASN1Element} or
	 *            {@code @ASN1Alternative}.
	 */
	NamedTypeSpecification(int order, Field field) {

		if (field.isAnnotationPresent(ASN1Identifier.class)) {
			identifier = field.getAnnotation(ASN1Identifier.class).value();
		} else {
			identifier = field.getName();
		}

		this.field = field;
		final int modifier = Modifier.FINAL | Modifier.PUBLIC | Modifier.STATIC;
		if (!ASN1Type.class.isAssignableFrom(field.getType())) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"An element of structured type must be a sub-class of ASN1Type.",
					null, enclosingType(), identifier, null);
			throw ex;
		}
		if ((field.getModifiers() & modifier) != Modifier.PUBLIC) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"An element of structured type must be a public mutable instance field.",
					null, enclosingType(), identifier, null);
			throw ex;
		}

		if (field.isAnnotationPresent(ASN1Tag.class)) {
			tag = new ASN1TagValue(this);
		}
		this.order = order;
	}

	/**
	 * Returns the identifier of this element.
	 * 
	 * @return The identifier.
	 */
	public String identifier() {
		return identifier;
	}

	/**
	 * Returns the tag of this element.
	 * 
	 * @return The tag.
	 */
	public ASN1TagValue tag() {
		return tag;
	}

	/**
	 * Returns the field object of this element.
	 * 
	 * @return The field.
	 */
	Field field() {
		return field;
	}

	/**
	 * Tests if the element represented by the {@code NamedTypeSpecification} is
	 * the same ASN.1 type of this element.
	 * 
	 * @param obj
	 *            The {@code NamedTypeSpecification} to be compared.
	 * @return {@code true} when the field is the same ASN.1 type.
	 */
	public boolean isSameType(NamedTypeSpecification obj) {
		return field.getType().equals(obj.field.getType());
	}

	/**
	 * Tests if the ASN.1 tag matches this element.
	 * 
	 * @param tagClass
	 *            ASN.1 tag class.
	 * @param tagNumber
	 *            ASN.1 tag number.
	 * @return {@code true} when the tag class and tag number matches this
	 *         element.
	 */
	public boolean matches(ASN1TagClass tagClass, int tagNumber) {
		if (tag != null) {
			return tagClass == tag.tagClass() && tagNumber == tag.tagNumber();
		}
		@SuppressWarnings("unchecked")
		TypeSpecification typeSpec = TypeSpecification
				.getSpecification((Class<? extends ASN1Type>) field.getType());
		if (typeSpec.tagged()) {
			return typeSpec.matches(tagClass, tagNumber);
		}
		return instantiate().matches(tagClass, tagNumber);
	}

	/**
	 * Creates a new instance of this element type.
	 * 
	 * @return An empty instance of this element type.
	 */
	@SuppressWarnings("unchecked")
	public ASN1Type instantiate() {
		return ASN1Type
				.instantiate((Class<? extends ASN1Type>) field.getType());
	}

	/**
	 * Returns the type of this field.
	 * 
	 * @return The type of this field.
	 */
	@SuppressWarnings("unchecked")
	Class<? extends ASN1Type> type() {
		return (Class<? extends ASN1Type>) field.getType();
	}

	/**
	 * Returns the enclosing type of this field.
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	Class<? extends ASN1Type> enclosingType() {
		return (Class<? extends ASN1Type>) field.getDeclaringClass();
	}

	/**
	 * Generates automatic tag and assigns it to this element.
	 * 
	 * @param order
	 *            The appearing order of this element in the type definition.
	 */
	void generateAutomaticTag(int order) {
		tag = new ASN1TagValue(order, type());
	}

	/**
	 * Retrieves an ASN.1 data assigned to the field specified by this
	 * {@code NamedTypeSpecification} from the {@code StructuredType} instance.
	 * 
	 * @param enclosure
	 *            The {@code StructuredType} instance.
	 * @return An ASN.1 data.
	 */
	ASN1Type retrieve(StructuredType enclosure) {
		try {
			return (ASN1Type) field.get(enclosure);
		} catch (IllegalAccessException e) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"Failed to retreave value from the field. The field might be inaccessible.",
					e, enclosure.getClass(), identifier, null);
			throw ex;
		}
	}

	/**
	 * Assigns the ASN.1 data to a field specified by this
	 * {@code NamedTypeSpecification} of the {@code StructuredType} instance.
	 * 
	 * @param enclosure
	 *            The {@code StructuredType} instance.
	 * @param data
	 *            The ASN.1 data to be assigned.
	 */
	void assign(StructuredType enclosure, ASN1Type data) {
		if (data != null && !field.getType().equals(data.getClass())) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"The type '"
							+ data.specification().identifier()
							+ "' of the data to be assigned is not the same type of this element.",
					null, enclosure.getClass(), identifier, null);
			throw ex;
		}
		try {
			field.set(enclosure, data);
		} catch (IllegalAccessException e) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"Failed to assign value to the field. The field might be inaccessible.",
					e, enclosure.getClass(), identifier, null);
			throw ex;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Comparable#compareTo(T)
	 */
	@Override
	public int compareTo(NamedTypeSpecification compared) {
		return order - compared.order;
	}

}
