/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.PrimitiveType;
import jp.bitmeister.asn1.type.ValueComparable;

/**
 * Represents ASN.1 'REAL' type.
 * 
 * <p>
 * An instance of this class represents a 'REAL' type data, and contains a
 * {@link java.lang.Double} value. Encoding mode will be used when the data is
 * encoded with BER or DER. Default value for encoding mode is decimal and it
 * can be set to binary.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 9, tagMode = ASN1TagMode.IMPLICIT)
public class REAL extends PrimitiveType<Double> implements ValueComparable {

	private boolean isBinary = false;

	/**
	 * Instantiates an empty {@code REAL}.
	 */
	public REAL() {
	}

	/**
	 * Instantiates an empty {@code REAL} and sets encoding mode.
	 * 
	 * @param isBinary
	 *            The encoding mode. {@code true} means this REAL data will be
	 *            encoded in binary encoding.
	 */
	public REAL(boolean isBinary) {
		this.isBinary = isBinary;
	}

	/**
	 * Instantiates a {@code REAL} and initialize it with the {@code double}
	 * value.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public REAL(double value) {
		set(value);
	}

	/**
	 * Instantiates a {@code REAL}, initializes it with the {@code double} value
	 * and sets encoding mode.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 * @param isBinary
	 *            The encoding mode.
	 */
	public REAL(double value, boolean isBinary) {
		this(isBinary);
		set(value);
	}

	/**
	 * Instantiates a {@code REAL} and initializes it with the value specified
	 * by the M*B^E formula.
	 * 
	 * @param mantissa
	 *            Mantissa.
	 * @param base
	 *            Base. It can take values 2 or 10.
	 * @param exponent
	 *            Exponent.
	 */
	public REAL(long mantissa, int base, int exponent) {
		set(mantissa, base, exponent);
	}

	/**
	 * Sets the {@code Float} value to the instance.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public void set(Float value) {
		set((double) value);
	}

	/**
	 * Sets the value specified by the M*B^E formula to the instance.
	 * 
	 * @param mantissa
	 *            Mantissa.
	 * @param base
	 *            Base. It can take values 2 or 10.
	 * @param exponent
	 *            Exponent.
	 */
	public void set(long mantissa, int base, int exponent) {
		if (base == 2) {
			isBinary = true;
		} else if (base == 10) {
			isBinary = false;
		} else {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage("Invalid base value '" + base
					+ "'. Base must be '2' or '10'.", null, getClass(), null,
					null);
			throw ex;
		}
		set(mantissa * Math.pow(base, exponent));
	}

	/**
	 * Sets binary encoding mode.
	 */
	public void setBinaryEncoding() {
		isBinary = true;
	}

	/**
	 * Sets decimal encoding mode.
	 */
	public void setDecimalEncoding() {
		isBinary = false;
	}

	/**
	 * Returns encoding mode for this data.
	 * 
	 * @return The encoding mode.
	 */
	public boolean isBinary() {
		return isBinary;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ValueComparable#compareTo(jp.bitmeister.asn1.
	 * type.ASN1Type)
	 */
	@Override
	public int compareTo(ASN1Type other) {
		return value().compareTo(((REAL) other).value());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
