/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.useful;

import java.nio.charset.Charset;
import java.util.regex.Pattern;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.StringType;

/**
 * Represents ASN.1 'UniversalString' type.
 * 
 * <p>
 * An instance of this class represents a 'UniversalString' type data.
 * Characters which can appear in the {@code UniversalString} are ISO/IEC
 * 10646-1 characters. In this library, UTF-16 encoding is used for this type
 * instead of ISO/IEC 10646-1.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 28, tagMode = ASN1TagMode.IMPLICIT)
public class UniversalString extends StringType {

	/**
	 * The name of character set that is used for converting value to
	 * {@code String}. Default value for this type is 'UTF-16'.
	 */
	public static String charsetName = "UTF-16";

	/**
	 * Instantiates an empty {@code UniversalString}.
	 */
	public UniversalString() {
	}

	/**
	 * Instantiates a {@code UniversalString} and initialize it with the
	 * {@code String} value.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public UniversalString(String value) {
		set(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StringType#charset()
	 */
	@Override
	public Charset charset() {
		return Charset.forName(charsetName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StringType#pattern()
	 */
	@Override
	protected Pattern pattern() {
		return null;
	}

}
