/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1ModuleTags;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET_OF;
import jp.bitmeister.asn1.type.useful.BMPString;
import jp.bitmeister.asn1.type.useful.CHARACTER_STRING;
import jp.bitmeister.asn1.type.useful.EMBEDDED_PDV;
import jp.bitmeister.asn1.type.useful.EXTERNAL;
import jp.bitmeister.asn1.type.useful.GeneralString;
import jp.bitmeister.asn1.type.useful.GeneralizedTime;
import jp.bitmeister.asn1.type.useful.GraphicString;
import jp.bitmeister.asn1.type.useful.IA5String;
import jp.bitmeister.asn1.type.useful.NumericString;
import jp.bitmeister.asn1.type.useful.ObjectDescriptor;
import jp.bitmeister.asn1.type.useful.PrintableString;
import jp.bitmeister.asn1.type.useful.TeletexString;
import jp.bitmeister.asn1.type.useful.UTCTime;
import jp.bitmeister.asn1.type.useful.UTF8String;
import jp.bitmeister.asn1.type.useful.UniversalString;
import jp.bitmeister.asn1.type.useful.VideotexString;
import jp.bitmeister.asn1.type.useful.VisibleString;

/**
 * The built-in ASN.1 module.
 * 
 * <p>
 * An instance of this class provides {@code instantiate} method that
 * instantiates an ASN.1 data from a UNIVERSAL class tag.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1ModuleTags(ASN1TagDefault.AUTOMATIC_TAGS)
public class BuiltInModule extends ASN1Module {

	/**
	 * Registers all of the UNIVERSAL ASN.1 types to this module.
	 */
	BuiltInModule() {
		super(true);
		register(BOOLEAN.class); // UNIVERSAL 1
		register(INTEGER.class); // UNIVERSAL 2
		register(BIT_STRING.class); // UNIVERSAL 3
		register(OCTET_STRING.class); // UNIVERSAL 4
		register(NULL.class); // UNIVERSAL 5
		register(OBJECT_IDENTIFIER.class); // UNIVERSAL 6
		register(ObjectDescriptor.class); // UNIVERSAL 7
		register(EXTERNAL.class); // UNIVERSAL 8
		register(REAL.class); // UNIVERSAL 9
		// UnkonwnEnumerated is registered automatically for UNIVERSAL 10.
		register(EMBEDDED_PDV.class); // UNIVERSAL 11
		register(UTF8String.class); // UNIVERSAL 12
									// 13-15 reserved
		// UnkownSequence is registered automatically for UNIVERSAL16
		// UnkownSet is registered automatically for UNIVERSAL 17
		register(NumericString.class); // UNIVERSAL 18
		register(PrintableString.class); // UNIVERSAL 19
		register(TeletexString.class); // UNIVERSAL 20
		register(VideotexString.class); // UNIVERSAL 21
		register(IA5String.class); // UNIVERSAL 22
		register(UTCTime.class); // UNIVERSAL 23
		register(GeneralizedTime.class); // UNIVERSAL 24
		register(GraphicString.class); // UNIVERSAL 25
		register(VisibleString.class); // UNIVERSAL 26
		register(GeneralString.class); // UNIVERSAL27
		register(UniversalString.class); // UNIVERSAL 28
		register(CHARACTER_STRING.class); // UNIVERSAL 29
		register(BMPString.class); // UNIVERSAL 30
	}

	/**
	 * Represents unknown 'ENUMERATED' type.
	 * 
	 * <p>
	 * If the ASN.1 tag indicates 'UNIVERSAL 10' the {@code instantiate} method
	 * of built-in module will return an instance of this class because the data
	 * must be an 'ENUMERATED' type but can't specify the definition. An
	 * instance of this type behaves as an instance of {@code INTEGER}.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 * 
	 * @see ENUMERATED
	 * @see INTEGER
	 */
	@ASN1BuiltIn
	@ASN1Identifier("ENUMERATED(unknown)")
	@ASN1Tag(value = 10, tagClass = ASN1TagClass.UNIVERSAL, tagMode = ASN1TagMode.IMPLICIT)
	public static class UnknownEnumerated extends INTEGER {
	}

	/**
	 * Represents unknown 'SEQUENCE' type.
	 * 
	 * <p>
	 * If the ASN.1 tag indicates 'UNIVERSAL 16' the {@code instantiate} method
	 * of built-in module will return an instance of this class because the data
	 * must be a 'SEQUENCE' or 'SEQUENCE OF' type but can't specify the
	 * definition. An instance of this type behaves as an instance of
	 * {@code SEQUENCE_OF<ANY>}.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 * 
	 * @see SEQUENCE_OF
	 */
	@ASN1BuiltIn
	@ASN1Identifier("SEQUENCE(unknown)")
	@ASN1Tag(value = 16, tagClass = ASN1TagClass.UNIVERSAL, tagMode = ASN1TagMode.IMPLICIT)
	public static class UnknownSequence extends SEQUENCE_OF<ANY> {

		public UnknownSequence() {
			super(ANY.class);
		}

	}

	/**
	 * Represents unknown 'SET' type.
	 * 
	 * <p>
	 * If the ASN.1 tag indicates 'UNIVERSAL 17' the {@code instantiate} method
	 * of built-in module will return an instance of this class because the data
	 * must be a 'SET' or 'SET OF' type but can't specify the definition. An
	 * instance of this type behaves as an instance of {@code SET_OF<ANY>}.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 * 
	 * @see SET_OF
	 */
	@ASN1BuiltIn
	@ASN1Identifier("SET(unknown)")
	@ASN1Tag(value = 17, tagClass = ASN1TagClass.UNIVERSAL, tagMode = ASN1TagMode.IMPLICIT)
	public static class UnknownSet extends SET_OF<ANY> {

		public UnknownSet() {
			super(ANY.class);
		}

	}

}
