/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1NamedBit;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;
import jp.bitmeister.asn1.exception.ASN1RuntimeException;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.Concatenatable;
import jp.bitmeister.asn1.type.PrimitiveType;
import jp.bitmeister.asn1.type.SizeCountable;
import jp.bitmeister.asn1.value.StringItem;

/**
 * Represents ASN.1 'BIT STRING' type.
 * 
 * <p>
 * An instance of this class represents a 'BIT STRING' type data, and has an
 * array of {@code boolean} value.
 * </p>
 * <p>
 * A sub-class of {@code BIT_STRING} can contain one or more {@code public}
 * {@code static} {@code final} {@code int} fields which annotated as
 * {@code @ASN1NamedBit}. Each of these fields represents an index number of a
 * bit in a bitstring value.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1NamedBit
 */
@ASN1BuiltIn
@ASN1Identifier("BIT STRING")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 3, tagMode = ASN1TagMode.IMPLICIT)
public class BIT_STRING extends PrimitiveType<boolean[]> implements
		Concatenatable<BIT_STRING>, SizeCountable {

	/**
	 * Contains the maps of named bits for all of the sub-types of 'BIT STRING'.
	 */
	private static final Map<Class<? extends BIT_STRING>, Map<Integer, String>> NAMED_BIT_MAP = new HashMap<Class<? extends BIT_STRING>, Map<Integer, String>>();

	static {
		NAMED_BIT_MAP.put(BIT_STRING.class, new HashMap<Integer, String>());
	}

	/**
	 * Returns the map of named bits for the type. The {@code Map} maps value to
	 * identifier.
	 * 
	 * @param type
	 *            The type.
	 * @return The map of enumerations.
	 */
	private static Map<Integer, String> getNamedBitMap(
			Class<? extends BIT_STRING> type) {
		if (NAMED_BIT_MAP.containsKey(type)) {
			return NAMED_BIT_MAP.get(type);
		}
		Map<Integer, String> map = new HashMap<Integer, String>();
		for (Field f : type.getDeclaredFields()) {
			if (!f.isAnnotationPresent(ASN1NamedBit.class)) {
				continue;
			}
			final int modifier = Modifier.PUBLIC | Modifier.STATIC
					| Modifier.FINAL;
			ASN1Identifier id = f.getAnnotation(ASN1Identifier.class);
			String fieldId = id != null ? id.value() : f.getName();
			if ((f.getModifiers() & modifier) != modifier) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"A named bit must be a public static final field.",
						null, type, fieldId, null);
				throw ex;
			}
			Integer value;
			if (f.getType() == int.class) {
				try {
					value = f.getInt(null);
				} catch (Exception e) {
					ASN1RuntimeException ex = new ASN1RuntimeException();
					ex.setMessage(
							"Failed to retrieve the value from the field.", e,
							type, fieldId, null);
					throw ex;
				}
				if (value < 0) {
					ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
					ex.setMessage(
							"A value of named bit must be a non-negative value.",
							null, type, fieldId, null);
					throw ex;
				}
			} else {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("A named bit must be an 'int' field.", null,
						type, fieldId, null);
				throw ex;
			}
			if (map.containsKey(value)) {
				ASN1RuntimeException ex = new ASN1RuntimeException();
				ex.setMessage(
						"A value of named bit shall be distinct from all other named bits in the type.",
						null, type, fieldId, null);
				throw ex;
			}
			map.put(value, fieldId);
		}
		@SuppressWarnings("unchecked")
		Class<? extends BIT_STRING> parent = (Class<? extends BIT_STRING>) type
				.getSuperclass();
		if (parent != BIT_STRING.class) {
			if (!map.isEmpty()) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"A class that does not extend 'BIT_STRING' directly, can not have own named bits.",
						null, type, null, null);
				throw ex;
			}
			map = getNamedBitMap(parent);
		}
		NAMED_BIT_MAP.put(type, map);
		return map;
	}

	/**
	 * Instantiates an empty {@code BIT_STRING}.
	 */
	public BIT_STRING() {
	}

	/**
	 * Instantiates a {@code BIT_STRING} and initialize it with the array of
	 * {@code boolean} value.
	 * 
	 * @param array
	 *            The value to be assigned.
	 */
	public BIT_STRING(boolean... array) {
		set(array);
	}

	/**
	 * Instantiates a {@code BIT_STRING} and initialize it with the
	 * {@link StringItem}.
	 * 
	 * @param item
	 *            The value to be assigned.
	 */
	public BIT_STRING(StringItem item) {
		set(item);
	}

	/**
	 * Instantiates a {@code BIT_STRING} and sets {@code true} to bits which
	 * specified by the indexes.
	 * 
	 * @param indexes
	 *            Indexes of bits to be set true.
	 */
	public BIT_STRING(int... indexes) {
		set(indexes);
	}

	/**
	 * Sets the {@link StringItem} value to this data.
	 * 
	 * @param item
	 *            The value to be set.
	 */
	public void set(StringItem item) {
		set(item.toBinArray());
	}

	/**
	 * Sets {@code true} to bits specified by index numbers. If the
	 * {@code boolean} array of the value of this data does not have enough size
	 * to hold the bit specified by the biggest index, it will be expanded
	 * automatically.
	 * 
	 * @param index
	 *            Indexes of bits to be set true.
	 */
	public void set(int ...indexes) {
		Arrays.sort(indexes);
		expand(indexes[indexes.length - 1]);
		for (int e: indexes) {
			value()[e] = true;
		}
	}

	/**
	 * Sets {@code false} to bits specified by index numbers. If the
	 * {@code boolean} array of the value of this data does not have enough size
	 * to hold the bit specified by the biggest index, it will be expanded
	 * automatically.
	 * 
	 * @param index
	 *            Indexes of bits to be set false.
	 */
	public void unset(int ...indexes) {
		Arrays.sort(indexes);
		expand(indexes[indexes.length - 1]);
		for (int e: indexes) {
			value()[e] = false;
		}
	}
	
	/**
	 * Contracts the {@code boolean} array of the value of this data to enough
	 * size to hold the last {@code true} bit.
	 */
	public void contract() {
		if (value() == null) {
			set(new boolean[0]);
			return;
		}
		int index = size();
		for (index--; index >= 0; index--) {
			if (value()[index]) {
				break;
			}
		}
		if (index + 1 != size()) {
			boolean[] newValue = new boolean[index + 1];
			System.arraycopy(value(), 0, newValue, 0, newValue.length);
			set(newValue);
		}
	}

	/**
	 * Returns the value of a bit specified by the index number.
	 * 
	 * @param index
	 *            The index number.
	 * @return The value of the bit.
	 */
	public boolean bit(int index) {
		if (index < size()) {
			return value()[index];
		}
		return false;
	}

	/**
	 * Tests if the type has named bits definition.
	 * 
	 * @return {@code true} when the type has named bits.
	 */
	public boolean hasNamedBits() {
		return !getNamedBitMap(getClass()).isEmpty();
	}

	/**
	 * Returns the indentifier of the bit specified by the index number.
	 * 
	 * @param index
	 *            The index number of a bit.
	 * @return The identifier of the bit.
	 */
	public String nameOfBit(int index) {
		return getNamedBitMap(getClass()).get(index);
	}

	/**
	 * Expands the {@code boolean} array of the value of this data to
	 * enough size to hold the bit specified by the index.
	 * 
	 * @param index
	 *            The index of a bit.
	 */
	private void expand(int index) {
		if (size() <= index) {
			boolean[] newValue = new boolean[index + 1];
			if (size() > 0) {
				System.arraycopy(value(), 0, newValue, 0, size());
			}
			set(newValue);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.Concatenatable#concatenate(null)
	 */
	@Override
	public void concatenate(BIT_STRING data) {
		if (data == null) {
			return;
		}
		if (!getClass().equals(data.getClass())) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"The type '"
							+ data.specification().identifier()
							+ "' of the data to be concatenated is not the same type of this data.",
					null, getClass(), null, null);
			throw ex;
		}
		if (data.hasValue()) {
			if (!hasValue()) {
				set(data.value());
			} else {
				boolean[] newValue = new boolean[value().length
						+ data.value().length];
				System.arraycopy(value(), 0, newValue, 0, value().length);
				System.arraycopy(data.value(), 0, newValue, value().length,
						data.value().length);
				set(newValue);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.SizeCountable#size()
	 */
	@Override
	public int size() {
		if (value() == null) {
			return 0;
		}
		return value().length;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#cloneValue()
	 */
	@Override
	protected boolean[] cloneValue() {
		if (value() == null) {
			return null;
		}
		return value().clone();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof BIT_STRING) {
			boolean[] otherValue = ((BIT_STRING) other).value();
			if (value() != null) {
				return Arrays.equals(value(), otherValue);
			}
			return otherValue == null;
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
