/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.useful;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1RuntimeException;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.TimeType;

/**
 * Represents ASN.1 "GeneralizedTime" type.
 * 
 * <p>
 * An instance of this class represents a 'GeneralizedTime' type data, and
 * contains an array of {@code byte} value. The value represents a character
 * sequence of a calendar date with 4-digits year, a time of day with
 * millisecond precision and a time differential from GMT.
 * </p>
 * <p>
 * Time differential is represented with a 4 digits number that follows plus(+)
 * or minus(-) sign, or a character 'Z' means GMT. If time differential is
 * omitted, the time zone is local time.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 24, tagMode = ASN1TagMode.IMPLICIT)
public class GeneralizedTime extends TimeType {

	private static final Pattern pattern = Pattern.compile("\\d{10}(\\d{2}(\\d{2}((\\.|\\,)\\d*)?)?)?(Z|(\\+|\\-)\\d{4})?");

	/**
	 * The date format for {@code GeneralizedTime}.
	 */
	private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT = new ThreadLocal<SimpleDateFormat>() {

		@Override
		protected SimpleDateFormat initialValue() {
			return new SimpleDateFormat("yyyyMMddHHmmss.SSSZ");
		}

	};

	/**
	 * Instantiates an empty {@code GeneralizedTime}.
	 */
	public GeneralizedTime() {
	}

	/**
	 * Instantiates a {@code GeneralizedTime} and initialize it with the
	 * {@code String} value.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public GeneralizedTime(String value) {
		set(value);
	}

	/**
	 * Instantiate a {@code GeneralizedTime} and initialize it with the
	 * {@code Date} value.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public GeneralizedTime(Date value) {
		set(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.TimeType#form()
	 */
	@Override
	public DateFormat form() {
		return DATE_FORMAT.get();
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StringType#pattern()
	 */
	@Override
	protected Pattern pattern() {
		return pattern;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.TimeType#parseDate(java.lang.String,
	 * java.lang.String)
	 */
	@Override
	protected Date parseDate(String time, String differential) {
		StringBuilder builder = new StringBuilder(time);
		if (builder.length() == 10) {
			builder.append("00");
		}
		if (builder.length() == 12) {
			builder.append("00");
		}
		if (builder.length() > 14) {
			if (builder.charAt(14) == ',') {
				builder.setCharAt(14, '.');
			}
			if (builder.length() > 18) {
				builder.setLength(18);
			}
			else {
				while (builder.length() < 18) {
					builder.append('0');
				}
			}
		}
		else {
			builder.append(".000");
		}
		builder.append(differential);
		try {
			return DATE_FORMAT.get().parse(builder.toString());
		} catch (ParseException e) {
			ASN1RuntimeException ex = new ASN1RuntimeException();
			ex.setMessage("Failed to parse the argument string '" + builder
					+ "'.", e, getClass(), null, null);
			throw ex;
		}
	}

}
