/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;
import jp.bitmeister.asn1.exception.ASN1InvalidDataValue;
import jp.bitmeister.asn1.exception.ASN1RuntimeException;
import jp.bitmeister.asn1.processor.ASN1Processor;
import jp.bitmeister.asn1.processor.ASN1StringBuilder;
import jp.bitmeister.asn1.processor.ASN1Visitor;

/**
 * The base class for all ASN.1 types.
 * 
 * <p>
 * This class provides general interfaces and common methods for all classes
 * that represent ASN.1 types in this library. Any sub-classes of
 * {@code ASN1Type} which to be instantiated must be declared as {@code public}
 * (and {@code static} if the class is a member class), and have a
 * {@code public} constructor with no argument.
 * </p>
 * <p>
 * User defined ASN.1 types must not extend {@code ASN1Type} directly, but use
 * one of built-in (or useful) types as the base class.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class ASN1Type implements Cloneable {

	/**
	 * An {@link ASN1Processor} that returns a {@code String}, is used in the
	 * {@code toString()} method. In default, an instance of
	 * {@code ASN1StringBuilder} is set. To modify output of {@code toString()}
	 * method of ASN.1 types, user defined processor can be assigned. If
	 * {@code null} is assigned to this field, {@code toString()} returns the
	 * result of {@code Object.toString()}.
	 * 
	 * @see ASN1Processor
	 * @see ASN1StringBuilder
	 */
	public static ASN1Processor<String, ASN1RuntimeException> stringBuilder = new ASN1StringBuilder();

	/**
	 * Instantiates an ASN.1 data of the type specified by the {@code Class}
	 * parameter.
	 * 
	 * @param type
	 *            The ASN.1 type to be instantiated.
	 * @return An instance of the ASN.1 type.
	 */
	public static <T extends ASN1Type> T instantiate(Class<T> type) {
		try {
			return type.newInstance();
		} catch (InstantiationException e) {
			ASN1RuntimeException ex = new ASN1IllegalDefinition();
			ex.setMessage("Failed to instanciate the class.", e, type, null,
					null);
			throw ex;
		} catch (IllegalAccessException e) {
			ASN1RuntimeException ex = new ASN1IllegalDefinition();
			ex.setMessage("Default constructor might be not accessable.", e,
					type, null, null);
			throw ex;
		}
	}
	
	/**
	 * The constructor with no argument.
	 */
	protected ASN1Type() {
	}

	/**
	 * Returns a {@link TypeSpecification} instance that associated to this
	 * ASN.1 type.
	 * 
	 * @return A {@code TypeSpecification} instance.
	 */
	public TypeSpecification specification() {
		return TypeSpecification.getSpecification(getClass());
	}

	/**
	 * Tests if the ASN.1 tag matches this ASN1 type.
	 * 
	 * @param tagClass
	 *            The ASN.1 tag class.
	 * @param tagNumber
	 *            The ASN.1 tag number.
	 * @return {@code true} when the tag class and the tag number matches this
	 *         ASN.1 type.
	 */
	public boolean matches(ASN1TagClass tagClass, int tagNumber) {
		return specification().matches(tagClass, tagNumber);
	}

	/**
	 * Tests if this ASN.1 data has valid value.
	 * 
	 * @throws ASN1InvalidDataValue
	 *             When the value of this data is invalid.
	 */
	public void validate() throws ASN1InvalidDataValue {
		if (!hasValue()) {
			ASN1InvalidDataValue ex = new ASN1InvalidDataValue();
			ex.setMessage("This ASN.1 data has no value.", null, getClass(),
					null, this);
			throw ex;
		}
	}

	/**
	 * Clears the value of this ASN.1 data.
	 */
	public abstract void clear();

	/**
	 * Tests if this ASN.1 data has value.
	 * 
	 * @return {@code true} when this ASN.1 data has value.
	 */
	public abstract boolean hasValue();
	
	/**
	 * Accepts the {@link ASN1Visitor} and calls a {@code visit} method of the
	 * visitor.
	 * 
	 * @param visitor 
	 *            The visitor.
	 * @return Result.
	 * @throws E 
	 *             When an error occured in the {@code visit} method of the
	 *             visitor.
	 */
	public abstract <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor)
			throws E;

	/**
	 * Tests if the value of this ASN.1 data equals the value of the other ASN.1
	 * data. This method returns {@code true} when they have same value even if
	 * their types are different.
	 * 
	 * @param other
	 *            The ASN.1 data which to be compared.
	 * @return {@code true} when they have same value.
	 */
	public abstract boolean valueEquals(Object other);

	/**
	 * Tests if the type and the value of this ASN.1 data equals the other ASN.1
	 * data.
	 * 
	 * @param other
	 *            The ASN.1 data which to be compared.
	 * @return {@code true} when they have same type and value.
	 */
	@Override
	public boolean equals(Object other) {
		if (other != null && getClass().equals(other.getClass())) {
			return valueEquals(other);
		}
		return false;
	}

	/**
	 * Returns a hash code value for this ASN.1 data.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public abstract int hashCode();

	/**
	 * Creates and returns a copy of this ASN.1 data. This method performs a
	 * 'deep copy' operation.
	 * 
	 * @see java.lang.Object#clone()
	 */
	@Override
	public abstract Object clone();

	/**
	 * Returns a string representation of this ASN.1 data.
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		if (stringBuilder == null) {
			return super.toString();
		}
		return stringBuilder.process(this);
	}

}
