/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import java.util.ArrayList;
import java.util.Collection;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.annotation.ASN1XmlTypeName;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;

/**
 * Represents ASN.1 'SEQUENCE OF' type.
 * 
 * <p>
 * This is the class used for defining 'SEQUENCE OF' types. Generic type
 * parameter {@code T} indicates component type of the 'SEQUENCE OF' type.
 * Constructors of a sub-class of {@code SEQUENCE_OF} must call parent constructor
 * with {@code componentType} parameter that is the class object of {@code T} in
 * them.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Identifier("SEQUENCE")
@ASN1XmlTypeName("SEQUENCE_OF")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 16, tagMode = ASN1TagMode.IMPLICIT)
public abstract class SEQUENCE_OF<T extends ASN1Type> extends CollectionType<T> {

	/**
	 * Instantiate an empty {@code SEQUENCE_OF}.
	 * 
	 * @param componentType
	 *            The class instance of component type.
	 */
	public SEQUENCE_OF(Class<T> componentType) {
		super(componentType);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.CollectionType#newCollection()
	 */
	@Override
	protected Collection<T> newCollection() {
		return new ArrayList<T>();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
