/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.useful;

import jp.bitmeister.asn1.annotation.ASN1Alternative;
import jp.bitmeister.asn1.annotation.ASN1Anonymous;
import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Element;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.annotation.ASN1XmlTypeName;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;

/**
 * Represents ASN.1 'EMBEDDED PDV' type.
 * 
 * <p>
 * An instance of this class represents an 'EmbeddedPDV' type data and the class
 * is a sub-class of {@code SEQUENCE}. 'EmbeddedPDV' type is defined with
 * following ASN.1 definition :
 * 
 * <pre>
 * EmbeddedPDV ::=  [UNIVERSAL 11] IMPLICIT SEQUENCE {
 *  identification CHOICE {
 *    syntaxes SEQUENCE { 
 *      abstract OBJECT IDENTIFIER,
 *      transfer OBJECT IDENTIFIER },
 *    syntax OBJECT IDENTIFIER,
 *    presentation-context-id INTEGER,
 *    context-negotiation SEQUENCE {
 *      presentation-context-id INTEGER,
 *      transfer-syntax OBJECT IDENTIFIER },
 *    transfer-syntax OBJECT IDENTIFIER,
 *    fixed NULL
 *  },
 *  data-value-descriptor ObjectDescriptor OPTIONAL,
 *  data-value OCTET STRING }
 *  ( WITH COMPONENTS {
 *            ... ,
 *            data-value-descriptor ABSENT } )
 * </pre>
 * 
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Identifier("EMBEDDED PDV")
@ASN1XmlTypeName("SEQUENCE")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 11, tagMode = ASN1TagMode.IMPLICIT)
public class EMBEDDED_PDV extends SEQUENCE {

	/**
	 * Sequence type definition for 'syntaxes' field.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	@ASN1Anonymous
	public static class Syntaxes extends SEQUENCE {

		/**
		 * 'abstract' field.
		 */
		@ASN1Element(0)
		@ASN1Identifier("abstract")
		public OBJECT_IDENTIFIER _abstract;

		/**
		 * 'transfer' field.
		 */
		@ASN1Element(1)
		public OBJECT_IDENTIFIER transfer;

		/**
		 * Instantiate an empty {@code Syntaxes}
		 */
		public Syntaxes() {
		}

		/**
		 * Instantiate a {@code Syntaxes} and initialize it with parameters.
		 * 
		 * @param _abstract
		 *            'abstract' field.
		 * @param transfer
		 *            'transfer' field.
		 */
		public Syntaxes(OBJECT_IDENTIFIER _abstract, OBJECT_IDENTIFIER transfer) {
			this._abstract = _abstract;
			this.transfer = transfer;
		}

	}

	/**
	 * Sequence type definition for 'context-negotiation' field.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	@ASN1Anonymous
	public static class ContextNegotiation extends SEQUENCE {

		/**
		 * 'presentation-context-id' field.
		 */
		@ASN1Element(0)
		@ASN1Identifier("presentation-context-id")
		public INTEGER presentation_context_id;

		/**
		 * 'transfer-syntax' field.
		 */
		@ASN1Element(1)
		@ASN1Identifier("transfer-syntax")
		public OBJECT_IDENTIFIER transfer_syntax;

		/**
		 * Instantiate an empty {@code ContextNegotiation}.
		 */
		public ContextNegotiation() {
		}

		/**
		 * Instantiate a {@code ContextNegotiation} and initialize it with
		 * parameters.
		 * 
		 * @param presentation_context_id
		 *            'presentation-context-id' field.
		 * @param transfer_syntax
		 *            'transfer-syntax' field.
		 */
		public ContextNegotiation(INTEGER presentation_context_id,
				OBJECT_IDENTIFIER transfer_syntax) {
			this.presentation_context_id = presentation_context_id;
			this.transfer_syntax = transfer_syntax;
		}

	}

	/**
	 * Choice type definition for 'identification' field.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	@ASN1Anonymous
	public static class Identification extends CHOICE {

		/**
		 * 'syntaxes' field.
		 */
		@ASN1Alternative(0)
		public Syntaxes syntaxes;

		/**
		 * 'syntax' field.
		 */
		@ASN1Alternative(1)
		public OBJECT_IDENTIFIER syntax;

		/**
		 * 'presentation-context-id' field.
		 */
		@ASN1Alternative(2)
		@ASN1Identifier("presentation-context-id")
		public INTEGER presentation_context_id;

		/**
		 * 'context-negotiation' field.
		 */
		@ASN1Alternative(3)
		@ASN1Identifier("context-negotiation")
		public ContextNegotiation context_negotiation;

		/**
		 * 'transfer-syntax' field.
		 */
		@ASN1Alternative(4)
		@ASN1Identifier("transfer-syntax")
		public OBJECT_IDENTIFIER transfer_syntax;

		/**
		 * 'fixed' field.
		 */
		@ASN1Alternative(5)
		public NULL fixed;

		/**
		 * Instantiates an empty {@code Identification}.
		 */
		public Identification() {
		}

		/**
		 * Instantiates an {@code Identification} and initialize it with
		 * parameters.
		 * 
		 * @param tagClass
		 *            The tag class used for select a field.
		 * @param tagNumber
		 *            The tag number used for select a field.
		 * @param data
		 *            The data to be assigned.
		 */
		public Identification(ASN1TagClass tagClass, int tagNumber,
				ASN1Type data) {
			super(tagClass, tagNumber, data);
		}

	}

	/**
	 * 'identification' field.
	 */
	@ASN1Element(0)
	public Identification identification;

	/**
	 * 'data-value' field.
	 */
	@ASN1Element(1)
	@ASN1Identifier("data-value")
	public OCTET_STRING data_value;

	/**
	 * Instantiates an empty {@code EMBEDDED_PDV}.
	 */
	public EMBEDDED_PDV() {
		super();
	}

	/**
	 * Instantiates an {@code EMBEDDED_PDV} and initialize it with prameters.
	 * 
	 * @param identification
	 *            'identification' field.
	 * @param data_value
	 *            'data-value' field.
	 */
	public EMBEDDED_PDV(Identification identification, OCTET_STRING data_value) {
		this.identification = identification;
		this.data_value = data_value;
	}

}
