/*
** LuaJIT frontend. Runs commands, scripts, read-eval-print (REPL) etc.
** Copyright (C) 2005-2021 Mike Pall. See Copyright Notice in luajit.h
**
** Major portions taken verbatim or adapted from the Lua interpreter.
** Copyright (C) 1994-2008 Lua.org, PUC-Rio. See Copyright Notice in lua.h
*/

/* Copyright (C) 2020 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
static void bootlj_setval(void* vm);
/* copied from luajit*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define luajit_c

#include "lua.h"
#include "lauxlib.h"
#include "lualib.h"
#include "luajit.h"

#include "lj_arch.h"

#if LJ_TARGET_POSIX
#include <unistd.h>
#define lua_stdin_is_tty()	isatty(0)
#elif LJ_TARGET_WINDOWS
#include <io.h>
#ifdef __BORLANDC__
#define lua_stdin_is_tty()	isatty(_fileno(stdin))
#else
#define lua_stdin_is_tty()	_isatty(_fileno(stdin))
#endif
#else
#define lua_stdin_is_tty()	1
#endif

#if !LJ_TARGET_CONSOLE
#include <signal.h>
#endif

static lua_State *globalL = NULL;
static const char *progname = LUA_PROGNAME;

#if !LJ_TARGET_CONSOLE
static void lstop(lua_State *L, lua_Debug *ar)
{
  (void)ar;  /* unused arg. */
  lua_sethook(L, NULL, 0, 0);
  /* Avoid luaL_error -- a C hook doesn't add an extra frame. */
  luaL_where(L, 0);
  lua_pushfstring(L, "%sinterrupted!", lua_tostring(L, -1));
  lua_error(L);
}

static void laction(int i)
{
  signal(i, SIG_DFL); /* if another SIGINT happens before lstop,
			 terminate process (default action) */
  lua_sethook(globalL, lstop, LUA_MASKCALL | LUA_MASKRET | LUA_MASKCOUNT, 1);
}
#endif

static void print_usage(void)
{
  fputs("usage: ", stderr);
  fputs(progname, stderr);
  fputs(" [options]... [script [args]...].\n"
  "Available options are:\n"
  "  -e chunk  Execute string " LUA_QL("chunk") ".\n"
  "  -l name   Require library " LUA_QL("name") ".\n"
  "  -b ...    Save or list bytecode.\n"
  "  -j cmd    Perform LuaJIT control command.\n"
  "  -O[opt]   Control LuaJIT optimizations.\n"
  "  -i        Enter interactive mode after executing " LUA_QL("script") ".\n"
  "  -v        Show version information.\n"
  "  -E        Ignore environment variables.\n"
  "  --        Stop handling options.\n"
  "  -         Execute stdin and stop handling options.\n", stderr);
  fflush(stderr);
}

static void l_message(const char *pname, const char *msg)
{
  if (pname) { fputs(pname, stderr); fputc(':', stderr); fputc(' ', stderr); }
  fputs(msg, stderr); fputc('\n', stderr);
  fflush(stderr);
}

static int bootlj_flg=1;
static int bootlj_rc=0;
static int report(lua_State *L, int status)
{
  //edit bootlj
  if(bootlj_flg){
//printf("%d\n", lua_gettop(L) );
    int rc=0;
    // 1はcpcallで何か入ってるようだ
    // isnum hits "123" and 456. needs collect typeck
    if( lua_type(L, 2)==LUA_TNUMBER ){ rc = (int)lua_tointeger (L, 2); }
    else if( lua_isnil(L, 2) ){ rc=1; }
    else if( lua_isboolean(L, 2) ){ rc= !lua_toboolean(L, 2); }
    //false rc=1
    if(status){ rc=1; fprintf(stderr, "%s\n", lua_tostring (L, 2));}
    
    //isstring()は数値も混じるので純粋な文字が欲しければこれしかない
    if( lua_type(L, 3)==LUA_TSTRING){ fprintf(stderr, "%s\n", lua_tostring (L, 3));}
    fflush(stderr);
    bootlj_flg=0;
    lua_settop(L, 0);
    bootlj_rc=rc;
    return status;
  }
  
  if (status && !lua_isnil(L, -1)) {
    const char *msg = lua_tostring(L, -1);
    if (msg == NULL) msg = "(error object is not a string)";
    l_message(progname, msg);
    lua_pop(L, 1);
  }
  return status;
}

static int traceback(lua_State *L)
{
  if (!lua_isstring(L, 1)) { /* Non-string error object? Try metamethod. */
    if (lua_isnoneornil(L, 1) ||
	!luaL_callmeta(L, 1, "__tostring") ||
	!lua_isstring(L, -1))
      return 1;  /* Return non-string error object. */
    lua_remove(L, 1);  /* Replace object by result of __tostring metamethod. */
  }
  luaL_traceback(L, L, lua_tostring(L, 1), 1);
  return 1;
}

static int docall(lua_State *L, int narg, int clear)
{
  int status;
  int base = lua_gettop(L) - narg;  /* function index */
  lua_pushcfunction(L, traceback);  /* push traceback function */
  lua_insert(L, base);  /* put it under chunk and args */
#if !LJ_TARGET_CONSOLE
  signal(SIGINT, laction);
#endif
  status = lua_pcall(L, narg, (clear ? 0 : LUA_MULTRET), base);
#if !LJ_TARGET_CONSOLE
  signal(SIGINT, SIG_DFL);
#endif
  lua_remove(L, base);  /* remove traceback function */
  /* force a complete garbage collection in case of errors */
  if (status != LUA_OK) lua_gc(L, LUA_GCCOLLECT, 0);
  return status;
}

static void print_version(void)
{
  fputs(LUAJIT_VERSION " -- " LUAJIT_COPYRIGHT ". " LUAJIT_URL "\n", stdout);
}

static void print_jit_status(lua_State *L)
{
  int n;
  const char *s;
  lua_getfield(L, LUA_REGISTRYINDEX, "_LOADED");
  lua_getfield(L, -1, "jit");  /* Get jit.* module table. */
  lua_remove(L, -2);
  lua_getfield(L, -1, "status");
  lua_remove(L, -2);
  n = lua_gettop(L);
  lua_call(L, 0, LUA_MULTRET);
  fputs(lua_toboolean(L, n) ? "JIT: ON" : "JIT: OFF", stdout);
  for (n++; (s = lua_tostring(L, n)); n++) {
    putc(' ', stdout);
    fputs(s, stdout);
  }
  putc('\n', stdout);
  lua_settop(L, 0);  /* clear stack */
}

static void createargtable(lua_State *L, char **argv, int argc, int argf)
{
  int i;
  lua_createtable(L, argc - argf, argf);
  for (i = 0; i < argc; i++) {
    lua_pushstring(L, argv[i]);
    lua_rawseti(L, -2, i - argf);
  }
  lua_setglobal(L, "arg");
}

static int dofile(lua_State *L, const char *name)
{
  int status = luaL_loadfile(L, name) || docall(L, 0, 1);
  return report(L, status);
}

static int dostring(lua_State *L, const char *s, const char *name)
{
  int status = luaL_loadbuffer(L, s, strlen(s), name) || docall(L, 0, 1);
  return report(L, status);
}

static int dolibrary(lua_State *L, const char *name)
{
  lua_getglobal(L, "require");
  lua_pushstring(L, name);
  return report(L, docall(L, 1, 1));
}

static void write_prompt(lua_State *L, int firstline)
{
  const char *p;
  lua_getfield(L, LUA_GLOBALSINDEX, firstline ? "_PROMPT" : "_PROMPT2");
  p = lua_tostring(L, -1);
  if (p == NULL) p = firstline ? LUA_PROMPT : LUA_PROMPT2;
  fputs(p, stdout);
  fflush(stdout);
  lua_pop(L, 1);  /* remove global */
}

static int incomplete(lua_State *L, int status)
{
  if (status == LUA_ERRSYNTAX) {
    size_t lmsg;
    const char *msg = lua_tolstring(L, -1, &lmsg);
    const char *tp = msg + lmsg - (sizeof(LUA_QL("<eof>")) - 1);
    if (strstr(msg, LUA_QL("<eof>")) == tp) {
      lua_pop(L, 1);
      return 1;
    }
  }
  return 0;  /* else... */
}

static int pushline(lua_State *L, int firstline)
{
  char buf[LUA_MAXINPUT];
  write_prompt(L, firstline);
  if (fgets(buf, LUA_MAXINPUT, stdin)) {
    size_t len = strlen(buf);
    if (len > 0 && buf[len-1] == '\n')
      buf[len-1] = '\0';
    if (firstline && buf[0] == '=')
      lua_pushfstring(L, "return %s", buf+1);
    else
      lua_pushstring(L, buf);
    return 1;
  }
  return 0;
}

static int loadline(lua_State *L)
{
  int status;
  lua_settop(L, 0);
  if (!pushline(L, 1))
    return -1;  /* no input */
  for (;;) {  /* repeat until gets a complete line */
    status = luaL_loadbuffer(L, lua_tostring(L, 1), lua_strlen(L, 1), "=stdin");
    if (!incomplete(L, status)) break;  /* cannot try to add lines? */
    if (!pushline(L, 0))  /* no more input? */
      return -1;
    lua_pushliteral(L, "\n");  /* add a new line... */
    lua_insert(L, -2);  /* ...between the two lines */
    lua_concat(L, 3);  /* join them */
  }
  lua_remove(L, 1);  /* remove line */
  return status;
}

static void dotty(lua_State *L)
{
  int status;
  const char *oldprogname = progname;
  progname = NULL;
  while ((status = loadline(L)) != -1) {
    if (status == LUA_OK) status = docall(L, 0, 0);
    report(L, status);
    if (status == LUA_OK && lua_gettop(L) > 0) {  /* any result to print? */
      lua_getglobal(L, "print");
      lua_insert(L, 1);
      if (lua_pcall(L, lua_gettop(L)-1, 0, 0) != 0)
	l_message(progname,
	  lua_pushfstring(L, "error calling " LUA_QL("print") " (%s)",
			      lua_tostring(L, -1)));
    }
  }
  lua_settop(L, 0);  /* clear stack */
  fputs("\n", stdout);
  fflush(stdout);
  progname = oldprogname;
}

static int handle_script(lua_State *L, char **argx)
{
  int status;
  const char *fname = argx[0];
  if (strcmp(fname, "-") == 0 && strcmp(argx[-1], "--") != 0)
    fname = NULL;  /* stdin */
  status = luaL_loadfile(L, fname);
  if (status == LUA_OK) {
    /* Fetch args from arg table. LUA_INIT or -e might have changed them. */
    int narg = 0;
    lua_getglobal(L, "arg");
    if (lua_istable(L, -1)) {
      do {
	narg++;
	lua_rawgeti(L, -narg, narg);
      } while (!lua_isnil(L, -1));
      lua_pop(L, 1);
      lua_remove(L, -narg);
      narg--;
    } else {
      lua_pop(L, 1);
    }
    status = docall(L, narg, 0);
  }
  return report(L, status);
}

/* Load add-on module. */
static int loadjitmodule(lua_State *L)
{
  lua_getglobal(L, "require");
  lua_pushliteral(L, "jit.");
  lua_pushvalue(L, -3);
  lua_concat(L, 2);
  if (lua_pcall(L, 1, 1, 0)) {
    const char *msg = lua_tostring(L, -1);
    if (msg && !strncmp(msg, "module ", 7))
      goto nomodule;
    return report(L, 1);
  }
  lua_getfield(L, -1, "start");
  if (lua_isnil(L, -1)) {
  nomodule:
    l_message(progname,
	      "unknown luaJIT command or jit.* modules not installed");
    return 1;
  }
  lua_remove(L, -2);  /* Drop module table. */
  return 0;
}

/* Run command with options. */
static int runcmdopt(lua_State *L, const char *opt)
{
  int narg = 0;
  if (opt && *opt) {
    for (;;) {  /* Split arguments. */
      const char *p = strchr(opt, ',');
      narg++;
      if (!p) break;
      if (p == opt)
	lua_pushnil(L);
      else
	lua_pushlstring(L, opt, (size_t)(p - opt));
      opt = p + 1;
    }
    if (*opt)
      lua_pushstring(L, opt);
    else
      lua_pushnil(L);
  }
  return report(L, lua_pcall(L, narg, 0, 0));
}

/* JIT engine control command: try jit library first or load add-on module. */
static int dojitcmd(lua_State *L, const char *cmd)
{
  const char *opt = strchr(cmd, '=');
  lua_pushlstring(L, cmd, opt ? (size_t)(opt - cmd) : strlen(cmd));
  lua_getfield(L, LUA_REGISTRYINDEX, "_LOADED");
  lua_getfield(L, -1, "jit");  /* Get jit.* module table. */
  lua_remove(L, -2);
  lua_pushvalue(L, -2);
  lua_gettable(L, -2);  /* Lookup library function. */
  if (!lua_isfunction(L, -1)) {
    lua_pop(L, 2);  /* Drop non-function and jit.* table, keep module name. */
    if (loadjitmodule(L))
      return 1;
  } else {
    lua_remove(L, -2);  /* Drop jit.* table. */
  }
  lua_remove(L, -2);  /* Drop module name. */
  return runcmdopt(L, opt ? opt+1 : opt);
}

/* Optimization flags. */
static int dojitopt(lua_State *L, const char *opt)
{
  lua_getfield(L, LUA_REGISTRYINDEX, "_LOADED");
  lua_getfield(L, -1, "jit.opt");  /* Get jit.opt.* module table. */
  lua_remove(L, -2);
  lua_getfield(L, -1, "start");
  lua_remove(L, -2);
  return runcmdopt(L, opt);
}

/* Save or list bytecode. */
static int dobytecode(lua_State *L, char **argv)
{
  int narg = 0;
  lua_pushliteral(L, "bcsave");
  if (loadjitmodule(L))
    return 1;
  if (argv[0][2]) {
    narg++;
    argv[0][1] = '-';
    lua_pushstring(L, argv[0]+1);
  }
  for (argv++; *argv != NULL; narg++, argv++)
    lua_pushstring(L, *argv);
  report(L, lua_pcall(L, narg, 0, 0));
  return -1;
}

/* check that argument has no extra characters at the end */
#define notail(x)	{if ((x)[2] != '\0') return -1;}

#define FLAGS_INTERACTIVE	1
#define FLAGS_VERSION		2
#define FLAGS_EXEC		4
#define FLAGS_OPTION		8
#define FLAGS_NOENV		16

static int collectargs(char **argv, int *flags)
{
  int i;
  for (i = 1; argv[i] != NULL; i++) {
    if (argv[i][0] != '-')  /* Not an option? */
      return i;
    switch (argv[i][1]) {  /* Check option. */
    case '-':
      notail(argv[i]);
      return i+1;
    case '\0':
      return i;
    case 'i':
      notail(argv[i]);
      *flags |= FLAGS_INTERACTIVE;
      /* fallthrough */
    case 'v':
      notail(argv[i]);
      *flags |= FLAGS_VERSION;
      break;
    case 'e':
      *flags |= FLAGS_EXEC;
      /* fallthrough */
    case 'j':  /* LuaJIT extension */
    case 'l':
      *flags |= FLAGS_OPTION;
      if (argv[i][2] == '\0') {
	i++;
	if (argv[i] == NULL) return -1;
      }
      break;
    case 'O': break;  /* LuaJIT extension */
    case 'b':  /* LuaJIT extension */
      if (*flags) return -1;
      *flags |= FLAGS_EXEC;
      return i+1;
    case 'E':
      *flags |= FLAGS_NOENV;
      break;
    default: return -1;  /* invalid option */
    }
  }
  return i;
}

static int runargs(lua_State *L, char **argv, int argn)
{
  int i;
  for (i = 1; i < argn; i++) {
    if (argv[i] == NULL) continue;
    lua_assert(argv[i][0] == '-');
    switch (argv[i][1]) {
    case 'e': {
      const char *chunk = argv[i] + 2;
      if (*chunk == '\0') chunk = argv[++i];
      lua_assert(chunk != NULL);
      if (dostring(L, chunk, "=(command line)") != 0)
	return 1;
      break;
      }
    case 'l': {
      const char *filename = argv[i] + 2;
      if (*filename == '\0') filename = argv[++i];
      lua_assert(filename != NULL);
      if (dolibrary(L, filename))
	return 1;
      break;
      }
    case 'j': {  /* LuaJIT extension. */
      const char *cmd = argv[i] + 2;
      if (*cmd == '\0') cmd = argv[++i];
      lua_assert(cmd != NULL);
      if (dojitcmd(L, cmd))
	return 1;
      break;
      }
    case 'O':  /* LuaJIT extension. */
      if (dojitopt(L, argv[i] + 2))
	return 1;
      break;
    case 'b':  /* LuaJIT extension. */
      return dobytecode(L, argv+i);
    default: break;
    }
  }
  return LUA_OK;
}

static int handle_luainit(lua_State *L)
{
#if LJ_TARGET_CONSOLE
  const char *init = NULL;
#else
  const char *init = getenv(LUA_INIT);
#endif
  if (init == NULL)
    return LUA_OK;
  else if (init[0] == '@')
    return dofile(L, init+1);
  else
    return dostring(L, init, "=" LUA_INIT);
}

static struct Smain {
  char **argv;
  int argc;
  int status;
} smain;

static int pmain(lua_State *L)
{
  struct Smain *s = &smain;
  char **argv = s->argv;
  int argn;
  int flags = 0;
  globalL = L;
  if (argv[0] && argv[0][0]) progname = argv[0];
  LUAJIT_VERSION_SYM();  /* Linker-enforced version check. */

  argn = collectargs(argv, &flags);
  if (argn < 0) {  /* Invalid args? */
    print_usage();
    s->status = 1;
    return 0;
  }

  if ((flags & FLAGS_NOENV)) {
    lua_pushboolean(L, 1);
    lua_setfield(L, LUA_REGISTRYINDEX, "LUA_NOENV");
  }

  /* Stop collector during library initialization. */
  lua_gc(L, LUA_GCSTOP, 0);
  luaL_openlibs(L);
  lua_gc(L, LUA_GCRESTART, -1);

  createargtable(L, argv, s->argc, argn);
  ;	bootlj_setval(L);		// add bootlj.c

  if (!(flags & FLAGS_NOENV)) {
    s->status = handle_luainit(L);
    if (s->status != LUA_OK) return 0;
  }

  if ((flags & FLAGS_VERSION)){
	  print_version();
      s->status = 1;
	  return 0;
  }
  s->status = runargs(L, argv, argn);
  if (s->status != LUA_OK) return 0;

  if (s->argc > argn) {
    s->status = handle_script(L, argv + argn);
    if (s->status != LUA_OK) return 0;
  }

  if ((flags & FLAGS_INTERACTIVE)) {
    print_jit_status(L);
    dotty(L);
  } else if (s->argc == argn && !(flags & (FLAGS_EXEC|FLAGS_VERSION))) {
    if (lua_stdin_is_tty()) {
      print_version();
      print_jit_status(L);
      dotty(L);
    } else {
      dofile(L, NULL);  /* Executes stdin as a file. */
    }
  }
  return 0;
}

/* add,edit bootlj.c */
static int ljmain(int argc, char **argv)
{
  //luajit -a -b -d -- name -g -h 方式になってる
  int status;
  lua_State *L = lua_open();
  if (L == NULL) {
    l_message(argv[0], "cannot create state: not enough memory");
    return EXIT_FAILURE;
  }
  smain.argc = argc;
  smain.argv = argv;
  status = lua_cpcall(L, pmain, NULL);
  status = status;
  lua_close(L);
  return bootlj_rc;
  
//  report(L, status);
//  lua_close(L);
//  return (status || smain.status > 0) ? EXIT_FAILURE : EXIT_SUCCESS;
}


#include <features.h>
#if ( _POSIX_C_SOURCE +0 < 200112L )
	# include	"needs compiler posix-2001 or upper(c99+)"
#endif

/* tool macros */
#ifndef ERRact
#include <stdio.h>
 #if (199901L <= __STDC_VERSION__ +0)	/* nealy 200112L, _POSIX_C_SOURCE	c99*/
	#include <sys/types.h>
	#include <unistd.h>
	#define ERRactag	__func__, getpid()
 #else
	#define ERRactag	"func:c99+", 0
 #endif
 #include <string.h>
 #include <errno.h>
 #define ERRact(xpr, msg, act)	if(xpr){ fprintf(stderr, \
	"ERR: %s %d %s() pid:%d %s msg:%s sys:%s\n",__FILE__,__LINE__, ERRactag \
	, "hit(" #xpr ")", msg, strerror(errno) ); act; }
 #define STOP(xpr, msg)	ERRact(xpr, msg, fputs("STOP\n",stderr);exit(1) )
#endif
#define loop(a)		for(int lpcnt=0;lpcnt<a;lpcnt++)
/*tool end*/

#include "bootlj.h"	/*SH_co*	libluajit.a	-ldl -lm */
static void bootlj_setval(void* vm);
//	#include "lua.h"	/*本体*/
//	#include "lualib.h"	/*printf 系基本関数 */
//	#include "lauxlib.h"	/*luaLとかの少し拡張系？ */
//	
//	#include "luaconf.h"	/*多分env*/
//	#include "luajit.h"		

typedef lua_State lua_t;
#define lua_new(a)	luaL_newstate(a)
#define lua_free(a) lua_close(a)

#include <stdio.h>	
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <libgen.h>		/*basename()*/

typedef struct bootinfo_tag{
	char* p_full;
	char* p_dir;	//ラストの/は含まない
	char* p_base;	// /は含まない単純名
	char* p_cwd;
	char* p_sep;
} btini_t;
static btini_t btini;

static void bootlj_setval(void* p){
	lua_t* vm = p;
	//set lua global value
	lua_pushstring(vm, btini.p_dir);
	lua_setglobal(vm, "FULL_BOOTDIR");
	lua_pushstring(vm, btini.p_cwd);
	lua_setglobal(vm, "FULL_CWD");
	lua_pushstring(vm, btini.p_sep);
	lua_setglobal(vm, "DIRSEP");
	lua_pushstring(vm, btini.p_base);
	lua_setglobal(vm, "BASE_BOOTFILE");
	lua_pushstring(vm, btini.p_full);
	lua_setglobal(vm, "FULL_BOOTFILE");

	/* グローバルにpathとexewd, cwd, sepを登録 */
	lua_getglobal(vm, "package" );
    lua_getfield(vm, -1, "path" );
	p = (char*)lua_tostring(vm, -1);
	lua_pushfstring(vm, "%s%s?.lua;%s", btini.p_dir, btini.p_sep, p);
	lua_setfield(vm, -3, "path");
	lua_pop(vm, 1);	/*// path_strが邪魔 */

	char* ldsuf = strrchr(LUA_CPATH_DEFAULT, '.')+1;
	lua_getfield(vm, -1, "cpath");
	p = (char*)lua_tostring(vm, -1);
	lua_pushfstring(vm, "%s%s?.%s;%s", btini.p_dir, btini.p_sep, ldsuf, p);
	lua_setfield(vm, -3, "cpath");
	lua_pop(vm, 1);	/*/cpath */
	lua_pop(vm, 1);	/*/package */
}

#include "msgp.h"
int bootlj_main_impl(int argc, char** argv, const char* namestr, const char* sepstr){
	if(namestr==NULL){ namestr="main.lua"; }
	if(sepstr==NULL){ sepstr="-J"; }
	int ac = argc+3;	//av[ac]==NULL, c-standard. -- + end NULL
	// main.lua + -- + \0 で最低3つ以上は増える
	char** av=NULL;
	int rc=0;
	//調査するのはPATH実行とlnの二つ
	//PATHの時はファイル名にdirpath/を含まない
	//リンク起動はlstatでファイル検査で検出する
	//判明したらrealpathを取得してファイル位置を割り出す
	char* exename = strdup(argv[0]);
	errno=0;
	//extpath
	char* ename = strdup(exename);
	if(strcmp(basename(ename), argv[0])==0){ errno=EINVAL; }
	free(ename);

	//basename: a.out etc. nodirinfo >>> PATH boot
	if(errno){
		//PATH系。linuxメインで進めて失敗したらdir付き起動しろmsg
		errno=0;
		const char* pstr = getenv("PATH");
		if(pstr==NULL){pstr="";}
		char* spath = strdup(pstr);
		; STOP(errno, "fatal err");
		char* p = NULL;
		while(1){
			p = strtok(p==NULL?spath:NULL, ":");
			if(p==NULL){
				free(exename);
				exename=NULL;
				break;
			}
			char* sbuf=0;
			int sz = snprintf(sbuf, 0, "%s/%s", p, exename);
			sz++;
			errno=0;
			sbuf = malloc(sz);
			; STOP(errno, "fatal err");
			rc=snprintf(sbuf, sz, "%s/%s", p, exename);
			struct stat stbuf;
			int rc = stat(sbuf, &stbuf);
			if(rc==0){
				free(exename);
				exename=sbuf;
				break;
			}
			free(sbuf);
		}
		free(spath);
	}

//memcheck
//	exit(0);
	ERRact(exename==NULL, 
	"\nfailed to get execfile dir. this pg needs dirinfo. run as below\n"
	" ~$ /foo/bar/a.out	:run with fullpath \n"
	" ~$ ../../bin/a.out	:relative path \n"
	" ~$ cd /aa/bb; ./a.out	:move to bindir \n", free(exename);return 1);
	//一応PATHから再検索する方法もあるけどwin/linuxで:;使い分けが必要になる
	//linuxのみ対応してwinはラッパーを作らせる方式でいいか

	//normalize
	char* pbuf = malloc(4096);
	char* p = realpath(exename, pbuf);
	//char* p = realpath(exename, NULL);
	//depends on sys in posix 2001 依存動作だけど多分大丈夫だろう
	//ダメなら適当にmalloc() 4000する
	free(exename);
	exename=strdup(pbuf);
	free(pbuf);
//exenameのみmem

	//argv[0]は必ずパス情報を含む
	btini.p_sep = strdup(LUA_DIRSEP);
	btini.p_full = strdup(exename);
	
	p = strdup(exename);
	btini.p_dir = strdup(dirname(p));
	free(p);
	
	// 頭の/は邪魔しかしないので削る
	p = strdup(exename);
	btini.p_base = strdup(basename(p) );
	free(p);
	//用済み
	free(exename);
	
	//常にfullpath実行でいいか cwdはとれるから
	int cwdsz=128;
	p = NULL;
	while(1){
		errno=0;
		p = (char*)realloc(p, cwdsz);
		getcwd(p, cwdsz);
		if(errno==ERANGE){ cwdsz=cwdsz*2; continue;}
		ERRact(errno, "getcwd() err", rc=1; goto lb_END;);
		break;
	}
	btini.p_cwd = p;
	//下準備ok

	//割り込み起動用にargv系のコピー作成
	av = (char**)calloc(ac, sizeof(char*) );
	int avpos=0;
	av[avpos]=strdup(argv[0]);
	avpos++;
	int flg=0;
	//初めにluajitへのoptを集める
	for(int i=1;i<argc;i++){
		if(strcmp(argv[i], sepstr) == 0 && flg==0){flg=1;continue;}
		if(strcmp(argv[i], sepstr) == 0 && flg==1){flg=0;continue;}
		if(flg==1){ av[avpos]=strdup(argv[i]); avpos++; }
	}
	ERRact(flg==1, "bad opt. 'sep' must be even ( -J a b c -J [end] etc)", rc=1; goto lb_END; );
	av[avpos]=strdup("--");
	avpos++;
	
	int sz = strlen(btini.p_dir)+strlen(btini.p_sep)+strlen(namestr)+1;
	char* ljname = (char*)calloc(sz, sizeof(char));
	sprintf(ljname, "%s%s%s", btini.p_dir, btini.p_sep, namestr);
//
	av[avpos]=ljname;	// /home/abc/main.lua etc
	avpos++;
	//通常の引数を追加していく
	for(int i=1;i<argc;i++){
		if(strcmp(argv[i], sepstr) == 0 && flg==0){flg=1;continue;}
		if(strcmp(argv[i], sepstr) == 0 && flg==1){flg=0;continue;}
		if(flg!=1){ av[avpos]=strdup(argv[i]); avpos++; }
	}
	av[avpos]=NULL;

	//main.lua系調査 こっちはリンクでも構わない
	struct stat stbuf;
	rc = stat(ljname, &stbuf);
	ERRact(rc, "", fprintf(stderr, "file: %s not found\n", ljname) );
	if(rc==0){ rc = ljmain(avpos, av);}

lb_END:;
	free(btini.p_full);	btini.p_full=NULL;
	free(btini.p_dir);	btini.p_dir=NULL;
	free(btini.p_base);	btini.p_base=NULL;
	free(btini.p_cwd );	btini.p_cwd=NULL;
	free(btini.p_sep );	btini.p_sep=NULL;
	//avは割り込みmalloc()してるので開放
	for(int i=0;av&&i<avpos;i++){ free(av[i]); }
	free(av);av=NULL;
//	lua_free(vm);	//ljでclose()してる
	return rc;
}



/*SH_SMP
#include "bootlj.h"
int main(){
	return 0;
}
//	~$ gcc smpl.c bootlj.c

//SH_SMPE*/


