;
; rcsID=$Id: Braille.lisp,v 1.19 2008/04/29 16:25:15 tmatsugaki Exp $
;
(proclaim '(inline
;                   encodeBrailleFile
                   encodeBrailleStr
                   encodeBrailleChunk
                   encodeBrailleChunkSubstr
;                   decodeBrailleFile
                   decodeBrailleStr
                   decodeBrailleChunk
                   decodeBrailleChunkSubstr
                   g1WordList2String
                   g2WordList2String
                   maintainBrace
                   initLang
                   getLang
                   getMaxItemLength
                   g1
                   g2
;                   study
;                   extract-err
;                   print-usage
))
(proclaim '(optimize speed))
; 語の定義
; 【行】　　　　復帰改行を含まない一行のデータ。
;              eg. "ABC DEF -- G-H-I"
; 【チャンク】　デリミタを末端に従えた文字列。
;              eg. "ABC ", "DEF ", "--", " ", "G-", "H-", "I"
; 【トークン】　チャンクの末端のデリミタを除いた部分。
;              eg. "ABC", "DEF", "G", "H", "I"
; 【デリミタ】　区切り文字。
;              eg. " ", "--", "-"
; 【部分文字列】【行】〜【デリミタ】について、上位に対する下位の呼称。
;             ここでは、主にチャンクに対するものを指している。
;
;;(load "profile.lisp")
;;(load "cl-ppcre-1.3.2/load.lisp")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 encodeBrailleFile
;; 【機能　】 墨字ファイル→点字ファイルへの変換をする。
;; 【入力　】 infn       : 墨字ファイル名
;;            outfn      : 点字ファイル名(BSE)
;;            nthLine    : 当該ファイル上での行番号（現在未使用）
;;            nthToken   : 当該行での n 番目のトークン
;;            rwBytes    : 一行のバイト数
;;            pgLines    : 一頁の行数
;;            spPrepInf  : 入力文字列の前置詞が分割されているか否か
;;            putCapSign : 大文字符自動付与の要否
;;            emHdr      : ヘッダ（含改ページ）を出力の要否
;;            emBlankPad : 空行への空白パディング出力の要否
;; 【出力　】 点字ファイル
;; 【例外　】 なし
;; 【使用例】 (jp) (encodeBrailleFile "../txt/test/test.txt" "../bse/test/test.bse")
;;          (g2us) (encodeBrailleFile "../txt/pink floyd/time.txt")
;;          (g2us) (encodeBrailleFile "a.txt" "b.txt" nil nil 32 22 t)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun encodeBrailleFile (infn
         &optional (outfn nil) (nthLine nil) (nthToken nil)
                   (rwBytes _bseRowBytes_) (pgLines _bsePageLines_) (spPrepInf t)
                   (putCapSign nil) (emHdr nil) (emBlankPad t))
  (let (extension buff header (mode *defaultMode*) (padding "  ") (lineNo 1)
    errList hasFF)
    ;; BSE のヘッダーは 512バイトで構造は下記？
    ;; char version[4]      = "0001" （固定？）
    ;; char bytesPerLine[2] = "32"   （デフォルトは32バイト）
    ;; char magicNo[2]      = "22"   （デフォルトは22行/頁）
    ; 行カウンタの初期化
    (setf *gInputLineCnt* 0 *gOutputLineCnt* 0)
    ; 行頁情報の初期化
    (setf *rowBytes* rwBytes *pageLines* pgLines)
    ; 括弧管理情報を初期化する。
    (setf *bracketStats* 0)
    (if outfn
      (setf extension
        (nth (1- (length (split-string outfn "."))) (split-string outfn ".")))
      nil)
    (if (and extension (string= (string-upcase extension) "BSE"))
      (setf header
        (format nil "~504A~4,'0D~2,'0D~2,'0D" " " 1 rwBytes pgLines))
        nil)
    ;; 必要に応じて、空白行に 2個の空白文字を挿入する。
    (if (not emBlankPad)
      (setf padding "")
      nil)
    ;; 入力ファイル（墨字）をオープンする。
    (with-open-file (instream infn :direction :input)
      (if (null outfn)
        (let (lin e)
          ; BSEヘッダを画面出力する。
          (if header
            (format t "~A~%" header)
            nil)
          (do ()
            ((null (setf buff (read-line instream nil))) nil)
            ; 入力行カウンタをインクリメントする。
            (incf *gInputLineCnt*)
            ; ヘッダ（含改ページ）の有無を検出する。
            (setf hasFF (and (> (length buff) 0) (char= (char buff 0) _sym_FF_)))
            ; 1行エンコードし、画面出力する。
            (if (or (null nthLine) (= nthLine lineNo))
              (if (> (length buff) 2)
                (progn
                  (multiple-value-setq
                    (lin mode e)
                    (encodeBrailleStr buff nthToken mode lineNo
                      spPrepInf putCapSign))
                  (if (= *debug* 3)
                    (format t "~80A~80A~%" buff lin)
                    (if hasFF ; ヘッダ（含改ページ）を出力する。
                      (format t "~C~A~%" _sym_FF_ lin)
                      (format t "~A~%" lin)))
                  (if e (push e errList) nil))
                (progn
                  ; 英語グレード２で空白行を検出したので、括弧管理情報を初期化する。
                  (if (= *grade* 2) (setf *bracketStats* 0) nil)
                  (format t "~A~%" padding)))
              nil)
            (incf lineNo)))
        (let (lin e)
          ;; 出力ファイル（点字）を削除する。
          (if (probe-file outfn)
            (delete-file outfn)
            nil)
          ;; 出力ファイル（点字）をオープンする。
          (with-open-file (outstream outfn :direction :output :if-exists :new-version)
            ; BSEヘッダを書き出す。
            (if header
              (format outstream "~A~%" header)
              nil)
            (do ()
              ((null (setf buff (read-line instream nil))) nil)
              ; 入力行カウンタをインクリメントする。
              (incf *gInputLineCnt*)
              ; ヘッダ（含改ページ）の有無を検出する。
              (setf hasFF (and (> (length buff) 0) (char= (char buff 0) _sym_FF_)))
              ; 1行エンコードし、ファイルに書き出す。
              (if (> (length buff) 2)
                (progn
                  (multiple-value-setq
                    (lin mode e)
                    (encodeBrailleStr buff nthToken mode lineNo
                      spPrepInf putCapSign))
                  (if hasFF (princ "|") (princ "*"))
                  (if (= *debug* 3)
                    (format outstream "~80A~80A~%" buff lin)
                    (if (and hasFF emHdr) ; ヘッダ（含改ページ）がある場合は、出力する。
                      (format outstream "~C~A~%" _sym_FF_ lin)
                      (format outstream "~A~%" lin)))
                  (if e (push e errList) nil))
                (progn
                  ; 英語グレード２で空白行を検出したので、括弧管理情報を初期化する。
                  (if (= *grade* 2) (setf *bracketStats* 0) nil)
                  (format outstream "~A~%" padding)))
              (incf lineNo))
            ;; 出力ファイル（点字）をクローズする。
            (close outstream))))
      ;; 入力ファイル（墨字）をクローズする。
      (close instream))
    errList))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 encodeBrailleStr
;; 【機能　】 1行の墨字文字列→点字文字列の変換をする。
;; 【入力　】 istr       : 墨字文字列(デリミタ以外は全角)
;;            nthToken   : 当該行での n 番目のトークン
;;            iMode      : 言語モードの初期値
;;            lineNo     : 行番号
;;            spPrepInf  : 入力文字列の前置詞が分割の要否
;;            putCapSign : 大文字符自動付与の要否
;;            dicGen     : 辞書生成用モードか否か
;; 【出力　】 (点字文字列 以前の言語モード エラーリスト)
;; 【例外　】 なし
;; 【使用例】 (jp) (encodeBrailleStr "；，，ＮＨＫ　ノ　テレビ　プログラムデ、")
;;            → ";,,NHK S QG\"V ,XJ\"%EY\"Q;"
;;            (g2us 0) (encodeBrailleStr "technical material simply building new")
;;            → "TE*NICAL MAT]IAL SIMPLY BUILD+ NEW"
;;               "us"
;;               NIL
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun encodeBrailleStr (istr
         &optional (nthToken nil)
                   (i_mode *defaultMode*) (lineNo 0) (spPrepInf t)
                   (putCapSign nil) (dicGen nil))
  (let (str (offset 0)
        (n_mode i_mode) (o_mode "") (wlst nil) errList)
    ;---------------------------------------------------------------------------
    (if (= *debug* 5)
      (progn
        (debug5-maintainSubstrCount 5 1 nil nil nil nil _dbgHeaderMask_))
      nil)
    (if (= *debug* 8)
      (progn
        (debug8-decideCombination 8 1 nil nil nil nil nil nil nil nil _dbgHeaderMask_))
      nil)
    ;---------------------------------------------------------------------------
    (if (string= (string-upcase i_mode) "FQ")
      (setf o_mode i_mode)
      (setf o_mode *defaultMode*))
    ; 1行エンコード前処理
    (setf str (linePreProcessInEncoding istr putCapSign dicGen *grade*))
    (if (> (length str) 0)
      (let (lst slen (last_s "") (last_b "") (tokenNo 1) (hyphenStack nil) (hyphenOffset 0))
        ; 言語モードを決定し、適切な言語を選択する。
        (setf slen (length str))
        ; 半角空白区切り（英語グレード２の場合はダッシュも）で一行を分割する。
        ; eg. "abc def " --> "abc " "def "
        (if (= *grade* 2)
          (setf lst (explode-string str (list _blank_s_ _dash_s_ _dash_w_) nil))
          (setf lst (explode-string str (list _blank_s_) nil)))
        (setf lst (flatten (mapcar #'(lambda (x) (explode-delims x #\Space)) lst)))
        ;-----------------------------------------------------------------------
        (if (= *debug* 1)
          (progn
            (debug1-mode 1 1 o_mode n_mode lst (logior _dbgHeaderMask_ _dbgContentMask_)))
          nil)
        (if (= *debug* 1)
          (progn
            (debug1-desc 1 3 nil nil nil nil nil nil nil nil _dbgHeaderMask_))
          nil)
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        ; 【注意事項】
        ; チャンクは全角化されたトークンとそれに続くデリミタ（半角空白）で構成されている。
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        (dolist (w lst)
          (tagbody
            start
            (let (w2 (tlst nil))
              ; ハイフンスタックのトークンをエンコード対象とする。
              (if (/= (length hyphenStack) 0)
                (setf w (pop hyphenStack))
                nil)
              ; 英語グレード２の場合は、数字だけで構成されたトークンの冒頭に ＃（全角） を付与する。
              (if (and (= *grade* 2) (digit-stringp (string-right-trim '(#\Space) w)))
                (setf w2 (concatenate 'string _numeralSign_w_ w))
                (setf w2 w))
              ; 1チャンクの墨字文字列→点字文字列の変換をする。
              (if (or (null nthToken) (= nthToken tokenNo))
                (multiple-value-setq
                  (tlst n_mode o_mode last_s last_b offset errList)
                  (encodeBrailleChunk w2 n_mode o_mode last_s last_b offset
                    errList lineNo spPrepInf dicGen))
                nil)
              (if (and *hyphenate* *justificate* (= (length hyphenStack) 0))
                ; ハイフネーションとジャスティフィケーションを実施する。
                (let (rc token0 token1)
                  (multiple-value-setq
                    (rc token0 token1)
                    (hyphenateAndJustificate w2 wlst tlst hyphenOffset))
                  (cond
                    ((and rc (= rc 1))
                      ; ハイフネーションを実施する。
                      (progn
                        ; 当該トークンを差し戻し、リトライする。
                        (push (concatenate 'string token1) hyphenStack)
                        (push (concatenate 'string token0 "-" _NewLine_) hyphenStack)
                        ; 以降のハイフネーションを抑止する。
                        (setf hyphenOffset (wlstBrailleLength wlst))
                        (go start)))
                    ((and rc (= rc 2))
                      ; ジャスティフィケーションを実施する。
                      (progn
                        ; 当該トークンを差し戻し、リトライする。
                        (push (concatenate 'string token1) hyphenStack)
                        (push (concatenate 'string token0 _NewLine_) hyphenStack)
                        ; 以降のジャスティフィケーションを抑止する。
                        (setf hyphenOffset (wlstBrailleLength wlst))
                        (go start)))
                    (t
                      ; ハイフネーションとジャスティフィケーションは必要なかった。
                      ; チャンクのワードリストを連結する。
                      (progn
                        (setf wlst (append wlst tlst))))))
                ; ハイフネーションとジャスティフィケーションを実施しないモードであるか、
                ; または、ハイフネーションスタックの文字列処理中である。
                ; チャンクのワードリストを連結する。
                (setf wlst (append wlst tlst)))

;(format t "~%~S ~S ~S ~S ~S ~S" tlst n_mode o_mode last_s last_b offset)

              (incf tokenNo)
              (if (> (length hyphenStack) 0)
                ; dolist のブロックの終端に到達させずに、タグボディーの冒頭にジャンプする。
                (go start)
                ; 通常の処理。
                nil))
          ) ; tagbody の終端
        ) ; dolist の終端
        (if (= *debug* 9)
          (let (texts brailles)
            (setf texts (mapcar #'(lambda (x) (narrow-string (nth _idxText_ x) *grade*)) wlst))
            (setf brailles (mapcar #'(lambda (x) (nth _idxBraille_ x)) wlst))
            (if (= *grade* 2)
              (format t "~%~S~%~S~%~S~%~S"
                        wlst
                        (join-string texts "")
                        (join-string brailles "")
                        (g2WordList2String wlst t))
              (format t "~%~S~%~S~%~S~%~S"
                        wlst
                        (join-string texts "")
                        (join-string brailles "")
                        (g1WordList2String wlst t))))
          nil))
      nil)
    (if (listp wlst)
      ; 1行エンコード後処理
      (let (tmp)
        (if (= *grade* 2)
          (setf tmp (g2WordList2String wlst t))
          (setf tmp (g1WordList2String wlst t)))
        (values (linePostProcessInEncoding istr tmp *grade*) o_mode errList))
      (values (format nil "~%") o_mode errList))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 encodeBrailleChunk
;; 【機能　】 1チャンクの墨字文字列→点字文字列の変換をする。
;; 【入力　】 w       : 墨字文字列(デリミタ以外は全角。末尾に半角空白がある場合もある）
;;            i_n_mode   : 次回の言語モード
;;            i_o_mode   : 前回の言語モード
;;            i_last_s   : 前回の墨字
;;            i_last_b   : 前回の点字
;;            i_offset   : オフセット
;;            i_errList  : エラーリスト
;;            lineNo     : 行番号
;;            spPrepInf  : 入力文字列の前置詞が分割の要否
;;            dicGen     : 辞書生成用モードか否か
;; 【出力　】 (ワードリスト 次回の言語モード 前回の言語モード 前回の墨字 前回の点字 オフセット エラーリスト)
;; 【例外　】 なし
;; 【使用例】 (jp) (encodeBrailleChunk "テレビ" "us" "us" nil nil 0 nil 32 nil nil)
;;            → ((1 "JP" 1 "テ" "Q" NIL) (2 "JP" 1 "レ" "G" NIL) (3 "JP" 1 "ビ" "\"V" NIL)) ;
;;            (g2us 0) (encodeBrailleChunk "technical" "us" "us" nil nil 0 nil 32 nil nil)
;;            → ((1 "us" 2 "t" "T" NIL) (2 "us" 2 "e" "E" NIL) (3 "us" 2 "ch" "*" NIL) (4 "us" 2 "n" "N" NIL) (5 "us" 2 "i" "I" NIL) (6 "us" 2 "c" "C" NIL) (7 "us" 2 "a" "A" NIL) (8 "us" 2 "l" "L" NIL)) ;
;;            → ((1 "us" 2 "technical" "TE*NICAL" NIL)) ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun encodeBrailleChunk (w i_n_mode i_o_mode i_last_s i_last_b i_offset i_errList
         lineNo spPrepInf dicGen)
  (let ((wlst nil) (wlst_leader nil) (n_mode i_n_mode) (o_mode i_o_mode)
        (last_s i_last_s) (last_b i_last_b) (offset i_offset) (errList i_errList)
        (substr_cnt 0))
    ;---------------------------------------------------------------------------
    (if (and (= *debug* 2) (= *grade* 2))
      (debug2-enc 2 1 nil nil nil nil nil nil nil nil nil nil nil nil nil nil _dbgHeaderMask_)
      nil)
    ;---------------------------------------------------------------------------
    ; チャンクの部分文字列カウンタを初期化する。
    (if (> (length w) 0)
      (let (word_s (word_b "") (effective_last_s "") (chunkErr nil) chunk chunkLen)
        (setf chunk w chunkLen (length chunk))
        (setf word_s (substitute-string w " " ""))
        ; 英語グレード２の場合、エンコード時に行をチャンクに分割する際のデリミタは " ", "－"(ダッシュ) である。
        ; "－"(ダッシュ) によるハイフネーションを適切に処理するため、事前の点字が "-" であることを保持する。
        (if (and (= *grade* 2)
                 (> (length last_b) 0)
                 (string= (subseq last_b (1- (length last_b))) _hyphen_s_))
          nil
          (setf last_s "" last_b ""))
        (progn
          (do ((left 0 (1+ left))) ; 左のインデクスのループ
            ((>= left chunkLen) nil)
            (incf offset) ; カラムインデクスをインクリメントする。
            (do ((right chunkLen (1- right))) ; 右のインデクスのループ
              ((>= left right) nil)
              (let (s b hankakuStr g2info effective dummy)
                (multiple-value-setq
                  (s b hankakuStr g2info n_mode)
                  (encodeBrailleChunkSubstr chunk left right o_mode last_s last_b
                    substr_cnt dicGen))
                ; 当該ワードが存在する。
                (if (stringp b)
                  (let ((dic_mode nil))
                    (if g2info
                      (progn
;                        (setf dic_mode (nth _dic_mode_index_ g2info))
                        ; 辞書から取得したモードを採用する。
                        (if dic_mode
                          (setf n_mode dic_mode)
                          nil)
                        ; 辞書のアクセス状況を更新する。
                        (if (string= (string-downcase (subseq *defaultEnglish* 1)) "k")
                          (updateUkG2DicAccessInfo g2info)
                          (updateUsG2DicAccessInfo g2info)))
                      nil)
                    ; 【１】当該の部分文字列が括弧などでない場合、チャンクの部分文字列カウンタをインクリメントする。
                    (if (and (string/= b "")
                             (or (null (us_ignoreableLeft s t t))
                                 (us_prep-inf-conj-p hankakuStr nil t)))
                      (incf substr_cnt)
                      nil)
                    ; 【２】1語エンコード後処理
                    (if (null dic_mode)
                      (progn
                        (multiple-value-setq
                          (n_mode dummy)
                          (wordPostProcess n_mode b s nil *grade* t))
                        ; 【３】エンコード結果をワードホルダーに追加する。
                        ; 英語グレード２で前置詞・不定詞・接続詞を分割する場合は適宜、
                        ; substr_cnt を設定する。（substr_cnt への効果があるのは、エンコード時のみ）
                        (if (and spPrepInf (= *grade* 2) (null g2info))
                          (setf substr_cnt (maintainSubstrCount b s effective_last_s substr_cnt t))
                          nil))
                      nil)
                    (setf word_b (concatenate 'string word_b b))
                    ; 【４】変換対象を次の部分文字列に移す。
                    (setf left (1- right))
                    ; 【５】直近の各種状態を保持する。
                    (setf last_s s last_b b o_mode n_mode)
                    ; 【６】直近の有効な部分文字列を保持する。
                    (setf effective (and (= *grade* 2) (not (formatter-decode-p b))))
                    (if effective
                      (setf effective_last_s s)
                      nil)
                    ; 結果をワードリストに連結する。
                    ; 英語グレード2 の場合は、墨字を半角小文字に変換して格納する。
                    (if (= *grade* 2)
                      (setf s (string-downcase (narrow-string s *grade*)))
                      nil)
                    ; 英語グレード2でリーダーがある場合は辞書作成時を除いて、リーダーリストを使用する。
                    (if (and (not dicGen) (= *grade* 2) (= substr_cnt 0))
                      (setf wlst_leader (append wlst_leader (list (list substr_cnt n_mode *grade* s b nil))))
                      (setf wlst (append wlst (list (list substr_cnt n_mode *grade* s b nil))))))
                  ; 入力行末が "、"  "。" の場合は辞書には無いが、エラーではない。
                  ; 一行処理の最後に適宜データを補填する。
                  ; 変換対象が 1バイトになっても変換に失敗するのは、内部エラー。
                  (if (and (= (length s) 1)
                           (not (string= s _kuten_w_))
                           (not (string= s _touten_w_)))
                    ; エラー
                    (progn
                      (setf chunkErr t)
                      (setf errList (push (list lineNo offset n_mode s) errList)))
                    nil))))))
        ; エンコード用辞書から得たものでない複合した語の場合は、エンコード用辞書に登録する。
        (if (and *regist_dict*
                 (= *grade* 2)
                 (null *forceGrade1*)
                 (null chunkErr)
                 word_s
                 word_b
                 (> substr_cnt 1)
                 (> (length word_b) 1))
          (let ((ls_sym (intern (narrow-string word_s 2))))
            (if (and (null (assoc ls_sym *grade2_enc_dict*))
                     (null (find-if #'(lambda (x) (string= x (subseq word_b 0 (length x)))) _inhibit_leading_brl_list_))
                     (null (search-chars word_b _inhibit_include_brl_list_)))
              ; 英語グレード２辞書の結合も、LiteraryUS の *us_grade2* に準拠する。
              ; ※トークンなので、_wholeWordMask_ で登録する。
              (progn
                (push
                  (list ls_sym n_mode _wholeWordMask_ (string-right-trim '(#\Space) word_b) 0)
                  *grade2_enc_dict*)
                (if (= *debug* 4)
                  (format t "~%エンコード用辞書登録[~S]→[~S]" ls_sym (string-right-trim '(#\Space) word_b))
                  nil))
              nil))
          nil))
      nil)
    (if wlst
      ; リーダーがあった場合は、ワードリストの先頭のメンバにリーダーリストを格納する。
      (if wlst_leader
        (setf (nth _idxLeaderList_ (nth 0 wlst)) wlst_leader)
        nil)
      (setf wlst wlst_leader))
    (values wlst n_mode o_mode last_s last_b offset errList)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 encodeBrailleChunkSubstr
;; 【機能　】 1チャンクの部分文字列の墨字文字列→点字文字列の変換をする。
;; 【入力　】 w       : 墨字文字列(デリミタ以外は全角。末尾に半角空白がある場合もある）
;;            i_left     : チャンクの冒頭からの部分文字列の開始オフセット
;;            i_right    : チャンクの冒頭からの部分文字列の終了オフセット
;;            i_o_mode   : 前回の言語モード
;;            i_last_s   : 前回の墨字
;;            i_last_b   : 前回の点字
;;            substr_cnt : チャンクの部分文字列カウンタ
;;            dicGen     : 辞書生成用モードか否か
;; 【出力　】 (点字文字列 墨字 半角の墨字 英語グレード2情報 言語モード)
;; 【例外　】 なし
;; 【使用例】 (jp) (encodeBrailleChunkSubstr "テ" 0 1 "us" "" "" 0 nil)
;;            → "Q"
;;            (g2us 0) (encodeBrailleChunkSubstr "t" 0 1 "us" "" "" 0 nil)
;;            → "T"
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun encodeBrailleChunkSubstr (w i_left i_right i_o_mode i_last_s i_last_b
         substr_cnt dicGen)
  (let ((o_mode i_o_mode) (last_s i_last_s) (last_b i_last_b) (chunkLen (length w))
        n_mode s (b nil) hankakuStr (g2info nil))
    (setf s (subseq w i_left i_right))
    (if (or (null *maxDicSumijiLength*) (<= (length s) *maxDicSumijiLength*))
      (progn
        (setf hankakuStr (narrow-string s *grade*))
        (let ((whole nil) (init nil) (mid nil) (fin nil))
          ; 1語エンコード前処理（言語モードを決定し、適切な言語を選択する。）
          (setf n_mode (wordPreProcess o_mode s last_s *grade* t))
          (if (constant-char-stringp s #\Space)
            ; 半角のスペースのみで構成された文字列は、変換しない。
            (setf b s)
            (progn
;(format t "~%~S,~S,~S,~S,~S,~S,~S,~S~%" w i_left i_right i_o_mode i_last_s last_b substr_cnt dicGen)
              (if (and (= *grade* 2) *useG2dic* (null *forceGrade1*)
                       (string= (string-downcase (subseq n_mode 0 1)) "u"))
                (if dicGen
                  ; 英語グレード２辞書データ生成
                  (setf b (brailleEncoder (intern hankakuStr) (getLang n_mode)))
                  ; 通常のエンコード処理
                  (let (r1 r2 hyphenated (len (length s)))
                    ; ハイフネーションの場合は、続く語を無条件で末尾結合とする。
                    ; ハイフンで連結された語の場合は、続く語の冒頭が構成符号である場合を除き末尾結合とする。
                    (setf hyphenated
                           (and (> (length last_b) 0)
                                ; 先行ワードの末端が "-" である。
                                (string= (subseq last_b (1- (length last_b))) _hyphen_s_)
                                ; 当該の墨字の冒頭は構成符号ではない。
                                (not
                                  (or
                                    (and (> len 0) (formatter-encode-p (subseq s 0 1)))
                                    (and (> len 1) (formatter-encode-p (subseq s 0 2)))))))
                    ; 英語グレード２エンコード時は、先ず結合モード決定と辞書検索をする。
                    (multiple-value-setq
                      (whole init mid fin b r1 r2 g2info)
                      (decideCombination
                        i_left (subseq w 0 i_left) (subseq w i_right chunkLen)
                        nil s hyphenated substr_cnt nil *grade* t))))
                ; 英語グレード２以外の場合は、効果が薄いので辞書検索をしない。
                nil)
              ; （ここの (length s) は必須！！）
              (if (and (null b) (<= (length s) *maxLangSumijiLength*))
                (let (lang)
                  ; 強制グレード１の場合は、グレード１を使用する。
                  (if *forceGrade1*
                    (setf lang (getLang n_mode t 1))
                    (setf lang (getLang n_mode)))
                  (cond
                    ; 1語エンコード主処理（半角化できない場合）
                    ; 墨字が半角化されると正しく検索できないものは半角化しない。
                    ((prohibit-hankaku-p s n_mode)
                      (setf b (brailleEncoder (intern s) lang)))
                    ; 1語エンコード主処理（半角化できる場合）
                    (t (setf b (brailleEncoder (intern hankakuStr) lang whole init mid fin)))))
                nil)
              (if (and (= *debug* 2) (= *grade* 2))
                (let ((sLen (length s)))
                  (debug2-enc 2 1 *forceGrade1* (not (null g2info)) whole init mid fin sLen (subseq s (1- sLen) sLen) i_left
                           (subseq w 0 i_left) s (subseq w i_right chunkLen) substr_cnt b _dbgContentMask_))
                nil)))))
      nil)
    (if (= *debug* 1)
      (debug1-desc 1 3 *grade* o_mode n_mode b s nil
        (seeksym *represent* (cdr (assoc (intern s) (getLang n_mode))))
        *bracketStats* _dbgContentMask_)
      nil)
    (values s b hankakuStr g2info n_mode)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 decodeBrailleFile
;; 【機能　】 点字ファイル→墨字ファイルへの変換をする。
;; 【入力　】 infn       : 点字ファイル名(BSE)
;;            outfn      : 墨字ファイル名
;;            hankaku    : 出力文字列半角化の要否
;;            spPrepInf  : 前置詞・不定詞・接続詞を分割の要否
;;            emSigns    : 数符・外字符・ACBC符・大文字符を出力の要否
;;            jpPrep     : 濁音、半濁音を前置きの要否
;;            emHdr      : ヘッダ（含改ページ）を出力の要否
;;            emBlankPad : 空行への空白パディング出力の要否
;; 【出力　】 墨字ファイル
;; 【例外　】 なし
;; 【使用例】 (jp) (decodeBrailleFile "../bse/biblio/1-01.bse" "../txt/test/a.txt")
;;            (g2us 3) (decodeBrailleFile "../bse/pink floyd/time.bse" nil nil nil t t nil nil nil t)
;;            (g2us 3) (decodeBrailleFile "../brf/Resources/monbas.brf" nil nil nil t t nil nil t)
;;            (g2us 3) (decodeBrailleFile "../brf/Mark Twain/The_Adventures_of_Tom_Sawyer_f1.brf" nil nil nil t t nil nil nil t)
;;            (g2us 3) (decodeBrailleFile "../brf/Alexandre Dumas/Count_of_Monte_Cristo_The_f1.brf" nil nil nil t t nil nil nil t)
;;            (g2us 3) (decodeBrailleFile "../brf/Mark Twain/The_Adventures_of_Tom_Sawyer_f1.brf" nil nil nil t t t nil nil t)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun decodeBrailleFile(infn
         &optional (outfn nil) (nthLine nil) (nthToken nil) (hankaku nil) (spPrepInf nil)
                   (emSigns t) (jpPrep nil) (emHdr nil) (emBlankPad t))
  (let (extension buff header (mode *defaultMode*) (lineNo 1) (padding "")
        errList hasFF)
    ; 行カウンタの初期化
    (setf *gInputLineCnt* 0 *gOutputLineCnt* 0)
    (setf extension
          (nth (1- (length (split-string infn "."))) (split-string infn ".")))
    ;; 必要に応じて、空白行に 2個の空白文字を挿入する。
    (if emBlankPad
      (if (string= *defaultMode* "JP")
        (setf padding _blank_w2_)
        (setf padding _blank_s2_))
      nil)
    ; 括弧管理情報を初期化する。
    (setf *bracketStats* 0)
    ;; 入力ファイル（点字）をオープンする。
    (with-open-file (instream infn :direction :input)
      (if (null outfn)
        (let (lin e)
          ; BSEヘッダを読み込む。
          (if (string= (string-upcase extension) "BSE")
            (setf header (read-line instream nil))
            nil)
          (do ()
            ((null (setf buff (read-line instream nil))) nil)
            ; 入力行カウンタをインクリメントする。
            (incf *gInputLineCnt*)
            ; ヘッダ（含改ページ）の有無を検出する。
            (setf hasFF (and (> (length buff) 0) (char= (char buff 0) _sym_FF_)))
            ; 1行デコードし、画面出力する。
            (if (or (null nthLine) (= nthLine lineNo))
              (if (> (length buff) 2)
                (progn
                  (multiple-value-setq
                    (lin mode e)
                    (decodeBrailleStr buff nthToken mode lineNo hankaku spPrepInf
                      emSigns jpPrep))
                  (if (null lin) (setf lin "") nil)
                  (if (= *debug* 3)
                    (progn
                      (if hasFF
                        (format t "~80A~80A~%" (subseq buff 1) lin)
                        (format t "~80A~80A~%" buff lin)))
                    (cond
                      ; ヘッダを出力する。
                      ((and hasFF emHdr) (format t "~C~A~%" _sym_FF_ lin))
                      ; ヘッダを出力しない。
                      ((and hasFF (not emHdr))
                        (progn
                          (setf lin (strip-header lin))
                          (format t "~A~%" lin)))
                      ; ヘッダでない。
                      (t (format t "~A~%" lin))))
                  (if e (push e errList) nil))
                (progn
                  ; 英語グレード２で空白行を検出したので、括弧管理情報を初期化する。
                  (if (= *grade* 2) (setf *bracketStats* 0) nil)
                  (format t "~A~%" padding)))
              nil)
            (incf lineNo)))
        (let (lin e)
          ;; 出力ファイル（墨字）を削除する。
          (if (probe-file outfn)
            (delete-file outfn)
            nil)
          ;; 出力ファイル（墨字）をオープンする。
          (with-open-file (outstream outfn :direction :output :if-exists :new-version)
            ; BSEヘッダを読み込む。
            (if (string= (string-upcase extension) "BSE")
              (setf header (read-line instream nil))
              nil)
            (do ()
              ((null (setf buff (read-line instream nil))) nil)
              ; 入力行カウンタをインクリメントする。
              (incf *gInputLineCnt*)
              ; ヘッダ（含改ページ）の有無を検出する。
              (setf hasFF (and (> (length buff) 0) (char= (char buff 0) _sym_FF_)))
              ; 1行デコードし、ファイルに書き出す。
              (if (> (length buff) 2)
                (progn
                  (multiple-value-setq
                    (lin mode e)
                    (decodeBrailleStr buff nthToken mode lineNo hankaku spPrepInf
                      emSigns jpPrep))
                  (if (null lin) (setf lin "") nil)
                  (if hasFF (princ "|") (princ "*"))
                  (if (= *debug* 3)
                    (progn
                      (if hasFF
                        (format outstream "~80A~80A~%" (subseq buff 1) lin)
                        (format outstream "~80A~80A~%" buff lin)))
                    (cond
                      ; ヘッダを出力する。
                      ((and hasFF emHdr)
                        (format outstream "~C~A~%" _sym_FF_ lin))
                      ; ヘッダを出力しない。
                      ((and hasFF (not emHdr))
                        (progn
                          (setf lin (strip-header lin))
                          (format outstream "~A~%" lin)))
                      ; ヘッダでない。
                      (t (format outstream "~A~%" lin))))
                  (if e (push e errList) nil))
                (progn
                  ; 英語グレード２で空白行を検出したので、括弧管理情報を初期化する。
                  (if (= *grade* 2) (setf *bracketStats* 0) nil)
                  (format outstream "~A~%" padding)))
              (incf lineNo))
            ;; 出力ファイル（墨字）をクローズする。
            (close outstream))))
      ;; 入力ファイル（点字）をクローズする。
      (close instream))
    errList))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 decodeBrailleStr
;; 【機能　】 1行の点字文字列→墨字文字列の変換をする。
;; 【入力　】 istr      : 点字文字列
;;            nthLine   : 当該ファイル上での行番号（現在未使用）
;;            nthToken  : 当該行での n 番目のトークン（現在未使用）
;;            iMode     : 言語モードの初期値
;;            lineNo    : 行番号
;;            hankaku   : 出力文字列半角化の要否
;;            spPrepInf : 前置詞・不定詞・接続詞分割の要否
;;            emSigns   : 数符・外字符・ACBC符・大文字符出力の要否
;;            jpPrep    : 濁音、半濁音前置の要否
;; 【出力　】 (墨字文字列 前回の言語モード エラーリスト)
;; 【例外　】 なし
;; 【使用例】 (jp) (decodeBrailleStr ";,,NHK S QG\"V ,XJ\"%EY\"Q;" nil "JP" 0 nil nil nil nil)
;;            → "；，，ＮＨＫ　ノ　テレビ　プログラムデ、"
;;            (g2us) (decodeBrailleStr "K ^U ! OLD )\\T HAV+ 6/OP 6LE>N A NEW" nil *defaultMode* 0 t t nil nil)
;;            → "knowledge upon the old without having to stop to learn a new"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun decodeBrailleStr(istr
         &optional (nthToken nil) (i_mode *defaultMode*) (lineNo 0) (hankaku nil)
                   (spPrepInf nil) (emSigns t) (jpPrep nil))
  (let ((o_mode "")
        str (offset 0) errList n_mode (wlst nil))
    ;---------------------------------------------------------------------------
    (if (= *debug* 1)
      (debug1-mode 1 2 nil nil nil _dbgHeaderMask_)
      nil)
    (if (= *debug* 5)
      (debug5-maintainSubstrCount 5 1 nil nil nil nil _dbgHeaderMask_)
      nil)
    (if (= *debug* 8)
      (debug8-decideCombination 8 1 nil nil nil nil nil nil nil nil _dbgHeaderMask_)
      nil)
    ;---------------------------------------------------------------------------
    (if (string= (string-upcase i_mode) "FQ")
      (setf o_mode i_mode)
      (setf o_mode *defaultMode*))
    ; 1行デコード前処理
    (setf str (linePreProcessInDecoding istr *grade*))
    (if (> (length str) 0)
      (let (lst slen
           (last_b "") (last_s "") (tokenNo 1))
        ; 言語モードを決定し、適切な言語を選択する。
        (setf slen (length str))
;
        ; 半角空白区切り（英語グレード２の場合はさらに "-", "--", "－"）で一行を分割する。
        ; eg. "abc def " --> "abc " "def "
        (if (= *grade* 2)
          (setf lst (explode-string str (list _blank_s_ _dash_s_) nil))
          (setf lst (explode-string str (list _blank_s_) nil)))
        ; 連結された空白をさらに分割する。
        (setf lst (flatten (mapcar #'(lambda (x) (explode-delims x #\Space)) lst)))
        ; 英語グレード２の場合は、"-" をデリミタとして、さらに分割する。
        ; 【注意】最初の有効な文字が "-" の場合はデリミタとしては扱わない。
        (if (= *grade* 2)
          (setf lst (flatten (mapcar #'(lambda (x) (split-except x "-" "--" (us_effectiveLeft x nil nil))) lst)))
          nil)
#|
        ; 半角空白区切り（英語グレード２の場合はさらに "-"）で一行を分割する。
        ; eg. "abc def " --> "abc " "def "
        (setf lst (explode-string str (list _blank_s_) nil))
        ; 連結された空白をさらに分割する。
        (setf lst (flatten (mapcar #'(lambda (x) (explode-delims x #\Space)) lst)))
        ; 英語グレード２の場合は、"-" をデリミタとして、さらに分割する。
        ; 【注意】最初の有効な文字が "-" の場合はデリミタとしては扱わない。
        (if (= *grade* 2)
          (setf lst (flatten (mapcar #'(lambda (x) (split-except x "-" nil (us_effectiveLeft x nil nil))) lst)))
          nil)
|#
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        ; 【注意事項】
        ; チャンクは全角化されたトークンとそれに続くデリミタ（半角空白）で構成されている。
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        (dolist (w lst)
          (let ((tlst nil))
            ; 1チャンクの点字文字列→墨字文字列の変換をする。
            (if (or (null nthToken) (= nthToken tokenNo))
              (multiple-value-setq
                (tlst n_mode o_mode last_s last_b offset errList)
                (decodeBrailleChunk w o_mode n_mode last_s last_b offset errList
                  lineNo spPrepInf emSigns jpPrep))
              nil)
            ; チャンクのワードリストを連結する。
            (setf wlst (append wlst tlst))
            (incf tokenNo)))
        (if (= *debug* 1)
          (debug1-mode 1 2 o_mode n_mode lst _dbgContentMask_)
          nil)
        (if (= *debug* 9)
          (let (texts brailles)
            (setf texts (mapcar #'(lambda (x) (nth _idxText_ x)) wlst))
            (setf brailles (mapcar #'(lambda (x) (nth _idxBraille_ x)) wlst))
            (if (= *grade* 2)
              (format t "~%~S~%~S~%~S~%~S"
                        wlst
                        (join-string texts "")
                        (join-string brailles "")
                        (g2WordList2String wlst nil))
              (format t "~%~S~%~S~%~S~%~S"
                        wlst
                        (join-string texts "")
                        (join-string brailles "")
                        (g1WordList2String wlst nil))))
          nil))
      nil)
    (if (listp wlst)
      ; 1行デコード後処理
      (let (tmp)
        (if (= *grade* 2)
          (setf tmp (g2WordList2String wlst nil))
          (setf tmp (g1WordList2String wlst nil)))
        (values (linePostProcessInDecoding tmp hankaku *grade*) o_mode errList))
      (values (format nil "~%") o_mode errList))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 decodeBrailleChunk
;; 【機能　】 1チャンクの点字文字列→墨字文字列の変換をする。
;; 【入力　】 w      : 点字文字列
;;            i_n_mode   : 次回の言語モード
;;            i_o_mode   : 前回の言語モード
;;            i_last_s   : 前回の墨字
;;            i_last_b   : 前回の点字
;;            i_offset   : オフセット
;;            i_errList  : エラーリスト
;;            lineNo     : 行番号
;;            spPrepInf  : 前置詞・不定詞・接続詞分割の要否
;;            emSigns    : 数符・外字符・ACBC符・大文字符出力の要否
;;            jpPrep     : 濁音、半濁音前置の要否
;; 【出力　】 (ワードリスト 次回の言語モード 前回の言語モード 前回の墨字 前回の点字 オフセット エラーリスト)
;; 【例外　】 なし
;; 【使用例】 (jp) (decodeBrailleChunk "QG\"V" nil "JP" "JP" "" "" 0 nil 0 nil nil nil)
;;            → "テレビ"
;;            (g2us) (decodeBrailleChunk ")\\T" nil "us" "us" "" "" 0 nil 0 t t nil)
;;            → "without"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun decodeBrailleChunk (w i_o_mode i_n_mode i_last_s i_last_b i_offset i_errList
         lineNo spPrepInf emSigns jpPrep)
  (let ((wlst nil) (wlst_leader nil) (o_mode i_o_mode) (n_mode i_n_mode)
        (last_s i_last_s) (last_b i_last_b) (offset i_offset) (errList i_errList)
        (substr_cnt 0))
    ;---------------------------------------------------------------------------
    (if (and (= *debug* 2) (= *grade* 2))
      (debug2-dec 2 2 nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil _dbgHeaderMask_)
      nil)
    (if (= *debug* 1)
      (debug1-desc 1 5 nil nil nil nil nil nil nil nil _dbgHeaderMask_)
      nil)
    ;---------------------------------------------------------------------------
    (if (> (length w) 0)
      (let ((inhibitMedialDot nil) word_b (word_s "") (chunkErr nil)
            (effective_last_s "") chunk chunkLen)
        (setf chunk w chunkLen (length chunk))
        (setf word_b (substitute-string w " " ""))
        ; 英語グレード２の場合、デコード時に行をチャンクに分割する際のデリミタは " ", "-"(ハイフン) である。
        ; "-"(ハイフン) によるハイフネーションを適切に処理するため、事前の点字が "-" であることを保持する。
;;;        (if (and (= *grade* 2) (string/= last_b _dash_s_))
        (if (and (= *grade* 2)
                 (> (length last_b) 0)
                 (string= (subseq last_b (1- (length last_b))) _hyphen_s_))
          nil
          (setf last_b "" last_s ""))
        ; 複数の "."[4] がある場合は、URL とみなし "." の中間結合をインヒビットする。
        (if (us_url-p w)
          (setf inhibitMedialDot t)
          nil)
        (progn
          (do ((left 0 (1+ left))) ; 左のインデクスのループ
            ((>= left chunkLen) nil)
            (incf offset) ; カラムインデクスをインクリメントする。
            (do ((right chunkLen (1- right))) ; 右のインデクスのループ
              ((>= left right) nil)
              (let (s b g2info modified underlyingCapital effective)
                (multiple-value-setq
                  (s b g2info n_mode modified underlyingCapital)
                  (decodeBrailleChunkSubstr chunk left right n_mode o_mode last_b
                    substr_cnt inhibitMedialDot))
                ; 当該ワードが存在する。
                (if (stringp s)
                  (let ((dic_mode nil))
                    (if g2info
                      (progn
;                        (setf dic_mode (nth _dic_mode_index_ g2info))
                        ; 辞書から取得したモードを採用する。
                        (if dic_mode
                          (setf n_mode dic_mode)
                          nil)
                        ; 辞書のアクセス状況を更新する。
                        (if (string= (string-downcase (subseq *defaultEnglish* 1)) "k")
                          (updateUkG2DicAccessInfo g2info)
                          (updateUsG2DicAccessInfo g2info)))
                      nil)
                    ; 【１】英語グレード２の場合は、全角小文字に変換する。
                    (if (= *grade* 2)
                      (progn
                        (if (null dic_mode)
                          (setf s (string-downcase s))
                          nil)
                        ; *us_grade2* から取得した墨字に大文字符が内在している場合は、大文字符の前処理をポストする。
                        (if underlyingCapital
                          (setf n_mode (wordPreProcess o_mode _capitalSign_s_ "" *grade* nil))
                          nil))
                      nil)
                    ; 【２】当該の部分文字列が括弧などでない場合、チャンクの部分文字列カウンタをインクリメントする。
                    (if (and (string/= s "")
                             (or (null (us_ignoreableLeft s nil t))
                                 (us_prep-inf-conj-p s nil t)))
                      (incf substr_cnt)
                      nil)
                    ; 【３】デコード後処理
                    (if (null dic_mode)
                      (multiple-value-setq
                        (n_mode s)
                        (wordPostProcess n_mode b s jpPrep *grade* nil))
                      nil)
                    ; 【４】us_wholeWordCandidate でチャンクの末端（デリミタ部分）が空白化された場合の処理。
                    (if modified
                      (multiple-value-setq
                        (s right) (wholeWordPostProcess s right))
                      nil)
                    ; 【５】適宜、数符・外字符・ACBC符・大文字符の出力を抑止する。
                    (if (null emSigns)
                      (setf s (suppressSigns s b))
                      (if underlyingCapital
                        (setf s (concatenate 'string _capitalSign_s_ s))
                        nil))
                    ; 【６】デコード結果をワードホルダーに追加し、substr_cnt を設定する。
                    ;      （substr_cnt への効果があるのは、エンコード時のみ）
                    (if (and spPrepInf (= *grade* 2) (null g2info))
                      (setf substr_cnt (maintainSubstrCount b s effective_last_s substr_cnt nil))
                      nil)
                    ; 前置詞・不定詞・接続詞の直後に空白が追加された場合（分割）は、ワードホルダーにも反映させる。
                    (if (or (= substr_cnt 0) (string= s _blank_s_))
                      (setf word_s (concatenate 'string word_s s _blank_s_))
                      (setf word_s (concatenate 'string word_s s)))
                    ; 【７】変換対象を次の部分文字列に移す。
                    (setf left (1- right))
                    ; 【８】直近の各種状態を保持する。
                    (setf last_b b last_s s o_mode n_mode)
                    ; 【９】直近の有効な部分文字列を保持する。
                    (setf effective (and (= *grade* 2) (not (formatter-decode-p b))))
                    (if effective
                      (setf effective_last_s s)
                      nil)
                    ; 結果をワードリストに連結する。
                    ; 英語グレード2でリーダーがある場合は、リーダーリストを使用する。
                    (if (and (= *grade* 2) (= substr_cnt 0))
                      (setf wlst_leader (append wlst_leader (list (list substr_cnt n_mode *grade* s b nil))))
                      (setf wlst (append wlst (list (list substr_cnt n_mode *grade* s b nil))))))
                  ; 変換対象が 1バイトになっても変換に失敗するのは、内部エラー。
                  (if (= (length s) 1)
                    (progn
                      (setf chunkErr t)
                      (setf errList (push (list lineNo offset n_mode b) errList)))
                    nil))))))
        ; 複合した新語をデコード用辞書に登録する。
        (if (and *regist_dict*
                 (= *grade* 2)
                 (null *forceGrade1*)
                 (null chunkErr)
                 word_b
                 word_s
                 (> substr_cnt 1)
                 (> (length word_b) 1))
          (let ((ub_sym (intern (string-upcase word_b))))
            (if (and (null (assoc ub_sym *us_grade2_dec_dict*))
                     (null (find-if #'(lambda (x) (string= x (subseq word_b 0 (length x)))) _inhibit_leading_brl_list_))
                     (null (search-chars word_b _inhibit_include_brl_list_)))
              ; 英語グレード２辞書の結合も、LiteraryUS の *us_grade2* に準拠する。
              ; ※トークンなので、_wholeWordMask_ で登録する。
              (progn
                (setf word_s (substitute-string word_s _blank_w_ ""))
                (setf word_s (substitute-string word_s _blank_s_ ""))
                (push
                  (list ub_sym n_mode _wholeWordMask_ word_s 0)
                  *us_grade2_dec_dict*)
                (if (= *debug* 4)
                  (format t "~%デコード用辞書登録[~S]←[~S]" word_s ub_sym)
                  nil))
              nil))
          nil))
      nil)
    (if wlst
      ; リーダーがあった場合は、ワードリストの先頭のメンバにリーダーリストを格納する。
      (if wlst_leader
        (setf (nth _idxLeaderList_ (nth 0 wlst)) wlst_leader)
        nil)
      (setf wlst wlst_leader))
    (values wlst n_mode n_mode last_s last_b offset errList)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 decodeBrailleChunkSubstr
;; 【機能　】 1チャンクの部分文字列の点字文字列→墨字文字列の変換をする。
;; 【入力　】 w      : 点字文字列
;;            i_left     : チャンクの冒頭からの部分文字列の開始オフセット
;;            i_right    : チャンクの冒頭からの部分文字列の終了オフセット
;;            i_n_mode   : 次回の言語モード
;;            i_o_mode   : 前回の言語モード
;;            i_last_b   : 前回の点字
;;            substr_cnt : チャンクの部分文字列カウンタ
;;            inhibitMedialDot : 
;; 【出力　】 (墨字文字列 墨字 点字 英語グレード2情報 言語モード 変更フラグ 内在する大文字符)
;; 【例外　】 なし
;; 【使用例】 (jp) (decodeBrailleChunkSubstr "QG\"V" 0 1 "JP" "JP" "" 0 nil)
;;            → "テ"
;;            (g2us) (decodeBrailleChunkSubstr ")\\T" 0 3 "us" "us" "" 0 nil)
;;            → "without"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun decodeBrailleChunkSubstr (w i_left i_right i_n_mode i_o_mode i_last_b
         substr_cnt inhibitMedialDot)
  (let ((n_mode i_n_mode) (o_mode i_o_mode) (last_b i_last_b)
        (chunkLen (length w))
        (s nil) b (g2info nil) (modified nil) (underlyingCapital nil))
    (setf b (subseq w i_left i_right))
    ; brlLen で絞ってはいけない！！
    (if (or (null *maxDicBrailleLength*) (<= (length b) *maxDicBrailleLength*))
      (progn
        ; 1語デコード前処理（言語モードを決定し、適切な言語を選択する。）
        (setf n_mode (wordPreProcess o_mode b last_b *grade* nil))
        (if (constant-char-stringp b #\Space)
          ; 半角のスペースのみで構成された文字列は、変換しない。
          (setf s (wide-string b *grade*))
          (let ((whole nil) (init nil) (mid nil) (fin nil))
;(format t "~%~S,~S,~S,~S,~S,~S,~S,~S" w i_left i_right i_n_mode i_o_mode i_last_b substr_cnt inhibitMedialDot)
            (if (and (= *grade* 2) *useG2dic* (null *forceGrade1*)
                     (string= (string-downcase (subseq n_mode 0 1)) "u"))
              (let (hyphenated (len (length b)))
                ; ハイフネーションの場合は、続く語を無条件で末尾結合とする。
                ; ハイフンで連結された語の場合は、続く語の冒頭が構成符号である場合を除き末尾結合とする。
                (setf hyphenated
                       (and (> (length last_b) 0)
                            ; 先行ワードの末端が "-" である。
                            (string= (subseq last_b (1- (length last_b))) _hyphen_s_)
                            ; 当該の点字の冒頭は構成符号ではない。
                            (not
                              (or
                                (and (> len 0) (formatter-decode-p (subseq b 0 1)))
                                (and (> len 1) (formatter-decode-p (subseq b 0 2)))))))
                ; 英語グレード２デコード時は、先ず結合モード決定と辞書検索をする。
                (multiple-value-setq
                  (whole init mid fin b s modified g2info)
                  (decideCombination
                    i_left (subseq w 0 i_left) (subseq w i_right chunkLen)
                    b s hyphenated substr_cnt inhibitMedialDot *grade* nil))
                (if g2info
                  (cond
                    ((and (>= (length s) 2) (string= (subseq s 0 2) ",,")) (setf n_mode (wordPreProcess o_mode ",," last_b *grade* nil)))
                    ((and (>= (length s) 1) (string= (subseq s 0 1) ",")) (setf n_mode (wordPreProcess o_mode "," last_b *grade* nil))))
                  nil))
              ; 英語グレード２以外の場合は、効果が薄いので辞書検索をしない。
              nil)
            ; 1語デコード主処理（ここの (length b) は必須！！）
            (if (and (null s) (<= (length b) *maxLangBrailleLength*))
              ; デコード結果の墨字、内在している大文字符の有無
              (let (lang)
                ; 強制グレード１の場合は、グレード１を使用する。
                (if *forceGrade1*
                  (setf lang (getLang n_mode t 1))
                  (setf lang (getLang n_mode)))
                (multiple-value-setq
                  (s underlyingCapital)
                  (brailleDecoder b lang whole init mid fin)))
              nil)
            (if (and (= *debug* 2) (= *grade* 2))
              (let (brlLen)
                (if b (setf brlLen (length b)) nil)
                (debug2-dec 2 2 *forceGrade1* (not (null g2info)) inhibitMedialDot
                  whole init mid fin brlLen (subseq b (1- brlLen) brlLen) i_left
                  (subseq w 0 i_left) b (subseq w i_right chunkLen) substr_cnt s _dbgContentMask_))
              nil))))
      nil)
    (if (= *debug* 1)
      (debug1-desc 1 5 *grade* o_mode n_mode b s nil
        (seeksym *represent* (cdr (assoc (intern b) (getLang n_mode))))
        *bracketStats* _dbgContentMask_)
      nil)
    (values s b g2info n_mode modified underlyingCapital)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 分割・連結の例外リスト
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Rule 1-4. nextWord が 1マス略語の場合
;  自身の冒頭が文章記号である場合は、略語の適用が許可されない。
; eg. (ocAbbrWord-led-by-punc-p '(2 "us" 2 " " " " NIL) '(1 "us" 2 "and" "2" ((0 "us" 2 "\"" "8" NIL))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun ocAbbrWord-led-by-punc-p (curWord nextWord)
  (let (target)
    (and
      curWord
      (string= (nth _idxText_ curWord) _blank_s_)
      nextWord
      (or
        (ocAbbrWord-leader-p (nth _idxText_ nextWord))
        (ocAbbrWord-member-p (nth _idxText_ nextWord)))
      (and (setf target (car (last (nth _idxLeaderList_ nextWord))))
           (listp target)
           (inhibit-ocAbbrWord-neighbour-p
             (nth _idxText_ target) (nth _idxBraille_ target))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Rule 1-4. lastWord が 1マス略語の場合
;  後続のワードの冒頭が文章記号である場合は、略語の適用が許可されない。
; eg. (ocAbbrWord-followed-by-punc-p '(1 "us" 2 "by" "0" NIL) " " '(1 "us" 2 "a" "A" ((0 "us" 2 "\"" "8" NIL))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun ocAbbrWord-followed-by-punc-p (lastWord curText nextWord)
  (let (target)
    (and
      lastWord
      (string= (subseq curText 0 1) _blank_s_)
      nextWord
      (or
        (ocAbbrWord-leader-p (nth _idxText_ lastWord))
        (ocAbbrWord-member-p (nth _idxText_ lastWord)))
      (or
        (and (setf target (first (nth _idxLeaderList_ nextWord)))
             (listp target)
             (inhibit-ocAbbrWord-neighbour-p
               (nth _idxText_ target) (nth _idxBraille_ target)))
        ; 辞書から取得したワードなど、文章記号は本文にも存在するため。
        (inhibit-ocAbbrWord-neighbour-p
          (nth _idxText_ nextWord) (nth _idxBraille_ nextWord))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Rule 4-3. nextWord が低下略語の場合
;  自身の冒頭が文章記号である場合は、略語の適用が許可されない。
; eg. (lwAbbrWord-led-by-punc-p '(2 "us" 2 " " " " NIL) '(1 "us" 2 "be" "2" ((0 "us" 2 "\"" "8" NIL))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun lwAbbrWord-led-by-punc-p (curWord nextWord)
  (let (target)
    (and
      curWord
      (string= (nth _idxText_ curWord) _blank_s_)
      nextWord
      (or
        (lwAbbrWord-leader-p (nth _idxText_ nextWord))
        (lwAbbrWord-member-p (nth _idxText_ nextWord)))
      (and (setf target (car (last (nth _idxLeaderList_ nextWord))))
           (listp target)
           (inhibit-lwAbbrWord-neighbour-p
             (nth _idxText_ target) (nth _idxBraille_ target))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Rule 4-3. lastWord が低下略語の場合
;  後続のワードの冒頭が文章記号である場合は、略語の適用が許可されない。
; eg. (lwAbbrWord-followed-by-punc-p '(1 "us" 2 "by" "0" NIL) " " '(1 "us" 2 "a" "A" ((0 "us" 2 "\"" "8" NIL))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun lwAbbrWord-followed-by-punc-p (lastWord curText nextWord)
  (let (target)
    (and
      lastWord
      (string= (subseq curText 0 1) _blank_s_)
      nextWord
      (or
        (lwAbbrWord-leader-p (nth _idxText_ lastWord))
        (lwAbbrWord-member-p (nth _idxText_ lastWord)))
      (or
        (and (setf target (first (nth _idxLeaderList_ nextWord)))
             (listp target)
             (inhibit-lwAbbrWord-neighbour-p
               (nth _idxText_ target) (nth _idxBraille_ target)))
;;;        (convine-terminate-p nextNextWord)
        ; 辞書から取得したワードなど、文章記号は本文にも存在するため。
        (inhibit-lwAbbrWord-neighbour-p
          (nth _idxText_ nextWord) (nth _idxBraille_ nextWord))))))

; 分割・連結必須の1マス略語ではない。
(defun inhibited-ocAbbrWord-order-p (lastWord curText nextWord nextNextWord)
  (and lastWord
       (string= curText _blank_s_)
       nextWord
       ; and/for/of/with の場合は、分割・連結必須語順("for a", "for the")などが分割・連結の対象。
       (and (ocAbbrWord-leader-p (nth _idxText_ lastWord))
            (not (and (prefer-divide-p (nth _idxText_ lastWord) (nth _idxText_ nextWord))
                      (convine-terminate-p nextNextWord))))))

; 後続がアトミックなワードか空白である。
(defun convine-terminate-p (nextNextWord)
  (or
    (null nextNextWord)
    (and
      nextNextWord
      (or (string= (subseq (nth _idxText_ nextNextWord) 0 1) _blank_s_)
          (= (nth _idxWordNumber_ nextNextWord) 1)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 g2WordList2String
;; 【機能　】 1行分の英語グレード2のワードリストとの分割・連結の処理をし、墨字または点字の
;;          文字列を生成する。
;; 【入力　】 wlst      : ワードリスト
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 墨字または点字の文字列
;; 【例外　】 なし
;; 【使用例】 (g2WordList2String '((1 "us" 2 "to" "6" nil) (2 "us" 2 " " " " nil) (1 "us" 2 "learn" "LE>N" nil)) t)
;;            → "6LE>N"
;;            (g2WordList2String '((0 "us" 2 "," "," nil) (0 "us" 2 "," "," nil)) t)
;;            → ",,"
;;            (g2WordList2String '((1 "us" 2 "to" "6" ((0 "us" 2 "," "," nil) (0 "us" 2 "," "," nil))) (2 "us" 2 " " " " nil) (1 "us" 2 "learn" "LE>N" nil)) t)
;;            → ",,6LE>N"
;;            (g2WordList2String '((1 "us" 2 "to" "6" nil) (2 "us" 2 "learn" "le>n" nil) (3 "us" 2 "　" " " nil)) nil)
;;            → "to learn"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun g2WordList2String (wlst encode)
  (let (ret (lastWord nil) (nthItem 0))
    (if encode
      ; エンコード
      (let ((brailleList nil) leader brl txt newLineCnt
           (after-prep-inf-conj nil) (force-raw-next nil) (lwAbbrWordFollowedByPunc nil))
        ;-----------------------------------------------------------------------
        (if (= *debug* 6)
          (debug6-flags 6 1 nil nil nil nil nil nil nil nil _dbgHeaderMask_)
          nil)
        (if (= *debug* 7)
          (debug6-intern 6 1 nil nil nil nil nil nil nil nil nil nil _dbgHeaderMask_)
          nil)
        ;-----------------------------------------------------------------------
        (dolist (w wlst)
          ; 大文字符などのリーダーがあった場合は、再起呼び出しで事前処理する。
          (if (nth _idxLeaderList_ w)
            (setf leader (g2WordList2String (nth _idxLeaderList_ w) encode))
            (setf leader nil))
          ; 通常のワードリスト処理
          (setf brl (nth _idxBraille_ w) txt (nth _idxText_ w))
          (if (null lastWord)
            ; 冒頭のトークン
            (setf brailleList (append brailleList (list leader brl)))
            (let (nextWord nextNextWord inhibitedOcAbbrWordOrder lwAbbrWordLedByPunc
                  ocAbbrWordLedByPunc ocAbbrWordFollowedByPunc)
              (setf nextWord (nth (1+ nthItem) wlst)
                    nextNextWord (nth (+ nthItem 2) wlst))
              ;-----------------------------------------------------------------
              ; 各種フラグのセット
              ;-----------------------------------------------------------------
              ; 1マス略語の連結禁止語順フラグのセット
              (setf inhibitedOcAbbrWordOrder (inhibited-ocAbbrWord-order-p lastWord txt nextWord nextNextWord))
              ; 1マス略語(and,for,of,with)の連結禁止フラグのセット
              (setf ocAbbrWordLedByPunc (ocAbbrWord-led-by-punc-p w nextWord)
                    ocAbbrWordFollowedByPunc (ocAbbrWord-followed-by-punc-p lastWord txt nextWord))
              ; 低下略語(to,into,by)の連結禁止フラグのセット
              (setf lwAbbrWordLedByPunc (lwAbbrWord-led-by-punc-p w nextWord)
                    lwAbbrWordFollowedByPunc (lwAbbrWord-followed-by-punc-p lastWord txt nextWord))
              ;-----------------------------------------------------------------
              (if (= *debug* 6)
                (debug6-flags 6 1 txt brl after-prep-inf-conj force-raw-next
                  ocAbbrWordLedByPunc ocAbbrWordFollowedByPunc lwAbbrWordLedByPunc lwAbbrWordFollowedByPunc _dbgContentMask_)
                nil)
              ;-----------------------------------------------------------------
              (cond
                ; 連結の結果または文章記号に続いた低下略語により、後続の変換が禁則されている場合は変換しない。
                (force-raw-next
                  (progn
                    ; 出力リストにアペンドする。無変換フラグは使用直後にオフにする。
                    (setf brailleList (append brailleList (list leader (string-upcase txt)))
                          force-raw-next nil)))
                ; 前置詞・不定詞・接続詞に続く空白は原則的に出力しない（連結）。
                ; 【例外】 禁則された語順の場合は連結しない。
                ; 【例外】 1マス略語や低下略語に句読が隣接している場合は連結しない。 
                ((and after-prep-inf-conj
                      (string= brl _blank_s_)
                      (not inhibitedOcAbbrWordOrder)
                      (not lwAbbrWordLedByPunc)
                      (not lwAbbrWordFollowedByPunc)
                      (not ocAbbrWordFollowedByPunc)
                      (not ocAbbrWordLedByPunc))
                  (setf force-raw-next
                    (and lastWord
                         nextWord
                         (inhibited-lwAbbrWord-convert-order-p (nth _idxText_ lastWord) (nth _idxText_ nextWord))
                         (convine-terminate-p nextNextWord))))
                (t (progn
                     ; 連結禁止リーダーに連なった前置詞・不定詞・接続詞の点字を無変換（墨字）にする。
                     (if (and after-prep-inf-conj
                              (string= (subseq brl 0 1) _blank_s_)
                              lwAbbrWordFollowedByPunc)
                       (setf (car (last brailleList)) (string-upcase (nth _idxText_ lastWord)))
                       nil)
                    ; 出力リストにアペンドする。
                     (setf brailleList (append brailleList (list leader brl)))
                    ; 文章記号に続いた低下略語は、当該ワード（空白）を出力した後、変換を禁止する。
                    (if lwAbbrWordLedByPunc
                      (setf force-raw-next t)
                      nil)                     
                    ; *inhibit-convine-list* にあるものが検出された際は、
                    ; after-prep-inf-conj をリセットし、連結に備える。
                     (if (and after-prep-inf-conj
                              (string= brl _blank_s_)
                              inhibitedOcAbbrWordOrder)
                       (setf after-prep-inf-conj nil)
                       nil))))
                    ;-----------------------------------------------------------
                    (if (= *debug* 7)
                      (debug6-intern 7 1 txt brl after-prep-inf-conj force-raw-next
                        ocAbbrWordLedByPunc ocAbbrWordFollowedByPunc lwAbbrWordLedByPunc
                        lwAbbrWordFollowedByPunc lastWord nextWord _dbgContentMask_)
                      nil)
                    ;-----------------------------------------------------------
;                  (format t "~%+++ after-prep-inf-conj:~S, text:~S, braille:~S, ~S, ~S" after-prep-inf-conj txt brl lastWord nextWord)
            ))
          ; 空白以外の場合、連結フラグを評価する。
          ; "by" を除いて原則的には連続した前置詞・不定詞・接続詞があっても連結しない。
          (if (string/= (subseq brl 0 1) _blank_s_)
            (cond
              ; 評価の対象は、冒頭の部分文字列。(cf.encodeBrailleChunk)
              ((= (nth _idxWordNumber_ w) 1)
                (progn
                  ; 原則的には連続した前置詞・不定詞・接続詞を連結対象にはしない。
                  ; 【例外】"by" (NFBTrans では結合させている。)
                  (setf after-prep-inf-conj
                      (and (or (null after-prep-inf-conj) (string= txt "by"))
                           (us_prep-inf-conj-p txt nil t)))))
              (t (setf after-prep-inf-conj nil)))
            nil)
          (setf lastWord w)
          (incf nthItem))
        (setf ret (join-string brailleList ""))
        ; ハイフネーションによる疑似改行を改行に変換する。
        (setf ret (substitute-string ret _NewLine_ (string _sym_CR_)))
        ; ハイフネーションによる改行の数をカウントする。
        (setf newLineCnt (count _sym_CR_ ret))
        ; ヘッダを挿入する。
        (if *emitHeader* (putHeader ret) nil)
        ; 出力行カウンタをインクリメントする。
        (setf *gOutputLineCnt* (+ *gOutputLineCnt* newLineCnt 1)))
      ; デコード
      (let ((textList nil) leader txt brl nextWord)
        (dolist (w wlst)
          ; 大文字符などのリーダーがあった場合は、再起呼び出しで事前処理する。
          (if (nth _idxLeaderList_ w)
            (setf leader (g2WordList2String (nth _idxLeaderList_ w) encode))
            (setf leader nil))
          ; 通常のワードリスト処理
          (setf nextWord (nth (1+ nthItem) wlst))
          ; 括弧の調整
;          (setf (nth _idxText_ w) (maintainBrace (nth _idxBraille_ w) (nth _idxText_ w) lastWord nextWord *grade*))
          (setf txt (nth _idxText_ w)
                brl (nth _idxBraille_ w))
          (if (null lastWord)
            ; 冒頭のトークン
            (setf textList (append textList (list leader txt)))
            ; 事前が前置詞・不定詞・接続詞の場合は、原則的に空白を追加する。（分割）
            ; 【例外】後続が句読や空白や#（ナンバーサイン）の場合は分割しない。
            ;        ここでは punctuation-p でなければならない。
            (if (and (us_prep-inf-conj-p (nth _idxText_ lastWord) nil t)
                     (string/= txt _blank_w_)
                     (string/= txt _numeralSign_s_)
                     (not (punctuation-p txt brl))
                     (or (= (nth _idxWordNumber_ w) 2) (= (nth _idxWordNumber_ w) 1) (= (nth _idxWordNumber_ w) 0))
                     (or
                       ; and/for/of/with 以外の場合は、無条件で分割する。
                       (not (ocAbbrWord-leader-p (nth _idxText_ lastWord)))
                       ; and/for/of/with の場合は、分割・連結必須語順("for a", "for the")などが分割対象。
                       (prefer-divide-p (nth _idxText_ lastWord) txt)))
              (setf textList (append textList (list _blank_s_ leader txt)))
              (setf textList (append textList (list leader txt)))))
          (setf lastWord w)
          (incf nthItem))
        ; 出力行カウンタをインクリメントする。
        (incf *gOutputLineCnt*)
        ; 英語グレード２は、後の便宜を図って半角文字列に変換する。
        (setf ret (narrow-string (join-string textList "") *grade*))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 g1WordList2String
;; 【機能　】 1行分のワードリストから、墨字または点字の文字列を生成する。
;; 【入力　】 wlst      : ワードリスト
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 墨字または点字の文字列
;; 【例外　】 なし
;; 【使用例】 (g1WordList2String '((1 "us" 2 "to" "6" nil) (2 "us" 2 " " " " nil) (1 "us" 2 "learn" "LE>N" nil)) t)
;;            → "6LE>N"
;;            (g1WordList2String '((0 "us" 2 "," "," nil) (0 "us" 2 "," "," nil)) t)
;;            → ",,"
;;            (g1WordList2String '((1 "us" 2 "to" "6" ((0 "us" 2 "," "," nil) (0 "us" 2 "," "," nil))) (2 "us" 2 " " " " nil) (1 "us" 2 "learn" "LE>N" nil)) t)
;;            → ",,6LE>N"
;;            (g1WordList2String '((1 "us" 2 "to" "6" nil) (2 "us" 2 "learn" "le>n" nil) (3 "us" 2 "　" " " nil)) nil)
;;            → "to learn"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun g1WordList2String (wlst encode)
  (let (ret (lastWord nil) (nthItem 0))
    (if encode
      ; エンコード
      (let (newLineCnt)
        (setf ret (join-string (mapcar #'(lambda (x) (nth _idxBraille_ x)) wlst) ""))
        ; ハイフネーションによる疑似改行を改行に変換する。
        (setf ret (substitute-string ret _NewLine_ (string _sym_CR_)))
        ; ハイフネーションによる改行の数をカウントする。
        (setf newLineCnt (count _sym_CR_ ret))
        ; ヘッダを挿入する。
        (if *emitHeader* (putHeader ret) nil)
        ; 出力行カウンタをインクリメントする。
        (setf *gOutputLineCnt* (+ *gOutputLineCnt* newLineCnt 1)))
      ; デコード
      (let ((textList nil) nextWord)
        (dolist (w wlst)
          ; 通常のワードリスト処理
          (setf nextWord (nth (1+ nthItem) wlst))
          ; 括弧の調整
;          (setf txt (maintainBrace (nth _idxBraille_ w) (nth _idxText_ w) lastWord nextWord *grade*))
          ; 出力リストにアペンドする。
          (setf textList (append textList (list (nth _idxText_ w))))
          (setf lastWord w)
          (incf nthItem))
        ; 出力行カウンタをインクリメントする。
        (incf *gOutputLineCnt*)
        (setf ret (join-string textList ""))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 putHeader
;; 【機能　】 ヘッダを挿入する。
;; 【入力　】 s : 文字列
;; 【出力　】 墨字または点字の文字列
;; 【例外　】 なし
;; 【使用例】 (putHeader )
;;            → "6LE>N"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun putHeader (s)
  (let (lst r0 q0 r1 q1)
    (multiple-value-setq (q0 r0) (truncate (1+ *gOutputLineCnt*) *pageLines*))
    (multiple-value-setq (q1 r1) (truncate (+ *gOutputLineCnt* (count _sym_CR_ s) 1) *pageLines*))
    (setf lst (split-string s (string _sym_CR_)))
    (cond
      ((and (<= r0 1) (<= 1 r1))
        (let (n map)
          (setf map (mapa-b #'1+ (1- r0) (1- r1) 1))
          (setf n (position 1 map))
;(format t "~%~S, ~S" n map)
          (if n
            (setf (nth n lst) (format nil "~36A#~3A" (nth n lst) q1))
            nil)))
      ((and (>= r0 r1))
        (let (n map)
          (setf map (mapa-b #'1- (1- r0) (1- r1) 1))
          (setf n (position 1 map))
;(format t "~%~S, ~S" n map)
          (if n
            (setf (nth n lst) (format nil "~36A#~3A" (nth n lst) q1))
            nil)))
      (t nil))
    (join-string lst (string _sym_CR_))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 maintainBrace
;; 【機能　】 エンコード・デコード時のカッコのモードを管理する。
;;           尚、括弧の状態（*bracketStats*）を初期化するタイミングは以下の３箇所。
;;           1.プログラム起動時
;;           2.ファイルの変換開始時
;;           3.空白行を検出したタイミング（英語グレード２のみ）
;; 【入力　】 brl       : 点字文字列
;;            txt       : 墨字文字列
;;            grade     : グレード
;; 【出力　】 括弧の状態遷移の有無を t/nil で返す。
;; 【例外　】 なし
;; 【使用例】 (maintainBrace "7" "（" 2 t) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun maintainBrace (brl txt lastWord nextWord grade)
  (let ((ret txt) (lastBrlEnd nil) (nextBrlStart nil))
    (if lastWord
      (setf lastBrlEnd (subseq (nth _idxBraille_ lastWord) (1- (length (nth _idxBraille_ lastWord)))))
      nil)
    (if nextWord
      (setf nextBrlStart (subseq (nth _idxBraille_ nextWord) 0 1))
      nil)

;(format t "~%~S, ~S, ~S, ~S, ~S" lastBrlEnd brl txt lastBrlEnd nextBrlStart)

    (if (/= grade 2)
      (cond
        ((and (string= brl "-") (or (string= txt "「") (string= txt "」")))
          (progn
            (if (or (null lastBrlEnd) (string= lastBrlEnd _blank_s_)) (setf ret "「") nil)
            (if (or (null nextBrlStart) (string/= lastBrlEnd _blank_s_)) (setf ret "」") nil)))
        ((and (string= brl "7") (or (string= txt "（") (string= txt "）")))
          (progn
            (if (or (null lastBrlEnd) (string= lastBrlEnd _blank_s_)) (setf ret "（") nil)
            (if (or (null nextBrlStart) (string/= lastBrlEnd _blank_s_)) (setf ret "）") nil))))
      ; 英語グレード２のデコードでは、シングルクォートは開始・終了にそれぞれ異なる
      ; 内部コードを使用している。ここでのメンテは不要で後で [`]→['] の置換をしている。
      (cond
        ((and (string= brl "7") (or (string= txt "(") (string= txt ")")))
          (progn
            (if (or (null lastBrlEnd) (string= lastBrlEnd _blank_s_)) (setf ret "(") nil)
            (if (or (null nextBrlStart) (string= nextBrlStart _blank_s_)) (setf ret ")") nil)))
        ((and (string= brl "_7") (or (string= txt "<") (string= txt ">")))
          (progn
            (if (or (null lastBrlEnd) (string= lastBrlEnd _blank_s_)) (setf ret "<") nil)
            (if (or (null nextBrlStart) (string= nextBrlStart _blank_s_)) (setf ret ">") nil)))
        ((and (string= brl ".7") (or (string= txt "{") (string= txt "}")))
          (progn
            (if (or (null lastBrlEnd) (string= lastBrlEnd _blank_s_)) (setf ret "{") nil)
            (if (or (null nextBrlStart) (string= nextBrlStart _blank_s_)) (setf ret "}") nil)))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 hyphenateAndJustificate
;; 【機能　】 ハイフォネートする。
;; 【入力　】 wlst      : ワードリスト
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】  nil : 何もする必要がない。
;;            1 : ハイフネーション
;;            2 : ジャスティフィケーション(改行する)
;; 【例外　】 なし
;; 【使用例】 (hyphenateAndJustificate )
;;            → 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun hyphenateAndJustificate (w2 wlst tlst hyphenOffset)
  ; 英語グレード２ではエンコード時に被エンコード文字列の末端に空白を1つ付与している。
  (let ((ret nil) (token0 nil) (token1 nil)
       (curBytes (wlstBrailleLength wlst)) (newBytes (1- (wlstBrailleLength tlst))))
    (if (> hyphenOffset 0)
      (setf curBytes (- curBytes hyphenOffset))
      nil)
    ;---------------------------------------------------------------------------
    (if (= *debug* 9)
      (format t "~%~S ~S ~S" curBytes *rowBytes* tlst)
      nil)
    ;---------------------------------------------------------------------------
    (if (> (+ curBytes newBytes) *rowBytes*)
      (let (candidates candidate)
        (setf candidates (getHyphenateCandidates *us_hyphen_dict* (narrow-string w2 *grade*)))
        (setf candidate (car candidates))
        ; 最悪に備えて、ジャスティフィケーションを設定する。
        (setf ret 1)
        (cond
          ; ハイフネーション不可(候補は1つで、トークンは 1つ)
          ((and (= (length candidates) 1) (= (length candidates) 1))
             (progn
               (progn
                 ; ジャスティフィケーション
                 (if (= *debug* 9)
                   (format t "~%~S ではハイフネーションは無理ですよ。" candidate)
                   nil)
                 (setf token0 "" token1 (nth 0 candidate)))))
          ; ハイフネーション可能(候補は1つで、トークンは 2つ)
          ((and (= (length candidates) 1) (= (length (car candidates)) 2))
             (progn
               ; ハイフン付与分も含める。
               (if (>= *rowBytes* (+ curBytes (length (car candidate)) 1))
                 (progn
                   ; ハイフネーション
                   (if (= *debug* 9)
                     (format t "~%~S でハイフネーションしましょう。" candidate)
                     nil)
                   (setf token0 (nth 0 candidate) token1 (nth 1 candidate))
                   (setf ret 1))
                 (progn
                   ; ジャスティフィケーション
                   (if (= *debug* 9)
                     (format t "~%~S ではハイフネーションは無理ですよ。" candidate)
                     nil)
                   (setf token0 "" token1 (nth 0 candidate))))))
          ; ハイフネーション可能(候補が複数存在し、トークンは 2つ)
          (t
             (progn
               (dolist (candidate candidates)
                 ; ハイフン付与分も含める。
                 (if (>= *rowBytes* (+ curBytes (length (car candidate)) 1))
                   (if (null token0)
                     (progn
                       ; ハイフネーション
                       (if (= *debug* 9)
                         (format t "~%~S でハイフネーションしましょう。" candidate)
                         nil)
                       (setf token0 (nth 0 candidate) token1 (nth 1 candidate))
                       (setf ret 1))
                     nil)
                   (progn
                     ; ジャスティフィケーション
                     (if (= *debug* 9)
                       (format t "~%~S ではハイフネーションは無理ですよ。" candidate)
                       nil)
                     (setf token0 "" token1 (nth 0 candidate)))))))))
      nil)
    (values ret token0 token1)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wlstBrailleLength
;; 【機能　】 ワードリストの点字文字列の総バイト数を求める。
;; 【入力　】 wlst : ワードリスト
;; 【出力　】 点字の文字列の総バイト数
;; 【例外　】 なし
;; 【使用例】 (wlstBrailleLength '((1 "Us" 2 "this" "?" ((0 "Us" 2 "," "," NIL))) (2 "us" 2 " " " " NIL) (1 "us" 2 "is" "IS" NIL) (2 "us" 2 " " " " NIL)))
;;            → 6
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wlstBrailleLength (wlst)
  (let ((ret 0))
    (dolist (w wlst)
      ; 通常の点字 1ワードの文字列のバイト数を求めて加算する。
      (setf ret (+ ret (length (nth _idxBraille_ w))))
      ; 構成符号などの点字リストの文字列数の総バイト数を再起呼び出しで求めて加算する。
      (if (nth _idxLeaderList_ w)
        (setf ret (+ ret (wlstBrailleLength (nth _idxLeaderList_ w))))
        nil))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 initLang
;; 【機能　】 各言語の連想配列を初期化する。
;; 【入力　】 グレード(1-2)
;; 【出力　】 デフォルトの言語の連想配列
;; 【例外　】 なし
;; 【使用例】 (initLang 1 "us") → *kana*
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun initLang (grade default_english &optional (jp nil))
  ; 各言語を初期化する。
  (initRepresent)
  (initCommon)
  (initNumber)
  (initPronunciation)
  (initLiteraryUS)
  (initLiteraryUK)
  (initKana)
  ; 括弧管理情報を初期化する。
  (setf *bracketStats* 0)
  ; 言語名リストを作成する。
  (setf *langNameList*
    (list "DI" "MA" "SC" "PR" "JP" "US1" "US2" "UK1" "UK2" "CS" "CN" "CK" "NA"))
  ; 言語リストを作成する。
  (setf *langList*
    (list *number* *mathematics* *science* *pronunciation* *kana*
      *us_grade1* *us_grade2* *uk_grade1* *uk_grade2*
      *cbcSign* *cbcNumber* *cbcKana* *nabcc*))
  ; 学習結果を各言語の辞書に反映する。
  (if (and (string= *defaultEnglish* "uk")
          (> (length *grade2_enc_dict*) (length *uk_grade2_enc_dict*)))
    (setf *uk_grade2_enc_dict* *grade2_enc_dict*)
    nil)
  (if (and (string= *defaultEnglish* "uk")
           (> (length *grade2_dec_dict*) (length *uk_grade2_dec_dict*)))
    (setf *uk_grade2_dec_dict* *grade2_dec_dict*)
    nil)
  (if (and (string= *defaultEnglish* "us")
           (> (length *grade2_enc_dict*) (length *us_grade2_enc_dict*)))
    (setf *us_grade2_enc_dict* *grade2_enc_dict*)
    nil)
  (if (and (string= *defaultEnglish* "us")
           (> (length *grade2_dec_dict*) (length *us_grade2_dec_dict*)))
    (setf *us_grade2_dec_dict* *grade2_dec_dict*)
    nil)
  ; グレードを設定する。
  (setf *grade* grade)
  ; デフォルト英語を設定する。
  (setf *defaultEnglish* default_english)
  ; 英語グレード２の辞書を設定する。
  (if (string= (string-downcase default_english) "uk")
    (setf *grade2_enc_dict* *uk_grade2_enc_dict*
          *grade2_dec_dict* *uk_grade2_dec_dict*)
    (setf *grade2_enc_dict* *us_grade2_enc_dict*
          *grade2_dec_dict* *us_grade2_dec_dict*))
  ; モードを設定する。
  (if (and jp (= *grade* 1))
    (setf *defaultMode* "JP")
    (setf *defaultMode* *defaultEnglish*))
  (setf *curMode* *defaultMode*)
  ; 言語を初期化する。
  (setf *curLang* (getLang *curMode* t) *defaultLang* (getLang *defaultMode* t))
  (if *verbose*
    (progn
      (format t "~%++++++++++++++++++++++++++++++++++++++++++++")
      (format t "~%グレード　　　　　：~S" *grade*)
      (format t "~%デフォルト英語　　：~S" *defaultEnglish*)
      (format t "~%デフォルトモード　：~S" *defaultMode*)
      (format t "~%カレントモード　　：~S" *curMode*)
      (format t "~%墨字の最大長　　　：~S" *maxLangSumijiLength*)
      (format t "~%点字の最大長　　　：~S" *maxLangBrailleLength*)
      (if (= *grade* 2)
        (progn
          (format t "~%--------------------------------------------")
          (format t "~%辞書登録　　　　　：~S" *regist_dict*)
          (format t "~%辞書参照　　　　　：~S" *lookup_dict*)
          (format t "~%辞書の墨字の最大長：~S" *maxDicSumijiLength*)
          (format t "~%辞書の点字の最大長：~S" *maxDicBrailleLength*))
        nil)
      (format t "~%++++++++++++++++++++++++++++++++++++++++++++"))
    nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 getLang
;; 【機能　】 点字の言語モードに応じた言語の連想配列を取得する。
;; 【入力　】 mode : 点字の言語モード
;; 【出力　】 言語の連想配列
;; 【例外　】 なし
;; 【使用例】 (getLang "JP") → *kana*
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun getLang (mode &optional (findIt nil) (grade *grade*))
  (let ((ret *curLang*))
    (if findIt
      (progn
        (cond
          ((string= mode "JP") (setf ret *kana*))
          ((string= mode "DI") (setf ret *number*))
          (t
            (if (= grade 2)
              (cond
                ((string= mode "uk") (setf ret *uk_grade2*))
                ((string= mode "Uk") (setf ret *uk_grade2*))
                ((string= mode "UK") (setf ret *uk_grade2*))
                ((string= mode "us") (setf ret *us_grade2*))
                ((string= mode "Us") (setf ret *us_grade2*))
                ((string= mode "US") (setf ret *us_grade2*))
                ((string= mode "fq") (setf ret *us_grade1*))
                ((string= mode "Fq") (setf ret *us_grade1*))
                ((string= mode "FQ") (setf ret *us_grade1*))
                ((string= mode "ac") (setf ret *us_grade1*))
                ((string= mode "Ac") (setf ret *us_grade1*))
                ((string= mode "AC") (setf ret *us_grade1*)))
              (cond
                ((string= mode "uk") (setf ret *uk_grade1*))
                ((string= mode "Uk") (setf ret *uk_grade1*))
                ((string= mode "UK") (setf ret *uk_grade1*))
                ((string= mode "us") (setf ret *us_grade1*))
                ((string= mode "Us") (setf ret *us_grade1*))
                ((string= mode "US") (setf ret *us_grade1*))
                ((string= mode "fq") (setf ret *us_grade1*))
                ((string= mode "Fq") (setf ret *us_grade1*))
                ((string= mode "FQ") (setf ret *us_grade1*))
                ((string= mode "ac") (setf ret *us_grade1*))
                ((string= mode "Ac") (setf ret *us_grade1*))
                ((string= mode "AC") (setf ret *us_grade1*))))))
        (cond
          ((or (string= mode "JP") (string= mode "DI"))
            (progn
              (setf *maxLangSumijiLength* (nth 1 (assoc (intern mode) *maxItemLengthList*)))
              (setf *maxLangBrailleLength* (nth 2 (assoc (intern mode) *maxItemLengthList*)))))
          (t
            (if (= grade 1)
              (if (string= (string-downcase *defaultEnglish*) "uk")
                (progn
                  (setf *maxLangSumijiLength* (nth 1 (assoc (intern "UK1") *maxItemLengthList*)))
                  (setf *maxLangBrailleLength* (nth 2 (assoc (intern "UK1") *maxItemLengthList*))))
                (progn
                  (setf *maxLangSumijiLength* (nth 1 (assoc (intern "US1") *maxItemLengthList*)))
                  (setf *maxLangBrailleLength* (nth 2 (assoc (intern "US1") *maxItemLengthList*)))))
              (if (string= (string-downcase *defaultEnglish*) "uk")
                (progn
                  (setf *maxLangSumijiLength* (nth 1 (assoc (intern "UK2") *maxItemLengthList*)))
                  (setf *maxLangBrailleLength* (nth 2 (assoc (intern "UK2") *maxItemLengthList*))))
                (progn
                  (setf *maxLangSumijiLength* (nth 1 (assoc (intern "US2") *maxItemLengthList*)))
                  (setf *maxLangBrailleLength* (nth 2 (assoc (intern "US2") *maxItemLengthList*))))))))
        (setf *lastMode* *curMode* *lastLang* *curLang*)
        (setf *curMode* mode *curLang* ret))
      nil)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 getMaxItemLength
;; 【機能　】 それぞれの言語の連想配列から墨字・点字の最大長を取得する。
;; 【入力　】 なし
;; 【出力　】 なし
;; 【例外　】 なし
;; 【使用例】 (getMaxItemLength) → nil
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun getMaxItemLength ()
  (let (lst sumiLen brlLen (i 0))
    (setf *maxItemLengthList* nil)
    (dolist (xx *langList*)
      (if xx
        (progn
          (setf lst (mapcar #'(lambda (x) (length (string (car x)))) xx))
          (setf sumiLen (apply 'max lst))
          (setf lst (mapcar #'(lambda (x) (braille-bytes (logand (cdr x) _AllClearBits_))) xx))
          (setf brlLen (apply 'max lst))
          (pushnew (list (intern (nth i *langNameList*)) sumiLen brlLen) *maxItemLengthList*))
        nil)
      (incf i))))

(defun jp (&optional (debug 0))
  (setf *bracketStats* 0)
  (setf *hankaku* nil)
  (setf *regist_dict* nil)
  (setf *lookup_dict* nil)
  (initLang 1 "us" t)
  (dbg debug))

(defun g1 (&optional (debug 0))
  (setf *bracketStats* 0)
  (setf *hankaku* t)
  (setf *regist_dict* nil)
  (setf *lookup_dict* nil)
  (initLang 1 "us")
  (dbg debug))

(defun g2us (&optional (debug 0))
  (setf *bracketStats* 0)
  (setf *hankaku* t)
  (setf *regist_dict* t)
  (setf *lookup_dict* t)
  (initLang 2 "us")
  (dbg debug))

(defun g2uk (&optional (debug 0))
  (setf *bracketStats* 0)
  (setf *hankaku* t)
  (setf *regist_dict* t)
  (setf *lookup_dict* t)
  (initLang 2 "uk")
  (dbg debug))

(defun study (v)
  (if v
    (princ "Start studying.")
    (princ "Stop studying."))
  (setf *regist_dict* v))

(defun dic (v)
  (if v
    (princ "Start looking up dictionary.")
    (princ "Stop looking up dictionary."))
  (setf *lookup_dict* v))

; エラーリストアイテムの4番目の項目が変換できなかった文字列である。
(defun extract-err (errList nthItem)
  (let ((lst nil) (i 0) flatLst)
    (setf flatLst (flatten errList))
    (dolist (x flatLst)
      ; 新規の変換できなかった文字列を作業用リストに追加する。
      (if (and (= (mod i nthItem) (1- nthItem)) (null (find x lst :test #'string=)))
        (push x lst)
        nil)
      (incf i))
    ; 変換できなかった文字列をソートする。
    (values (sort lst #'string-lessp) (nth 2 flatLst))))

(defun print-usage ()
  (format t "~%Usage:\
clisp -C Braille.lisp input_file [output_file] [grade] [row_bytes]\
                      [page_lines] [to_hankaku] [emit_signs] [emit_jp_prep]\
                      [emit_header] [emit_blank_paddings]\
input_file          : 入力ファイル名\
output_file         : 出力ファイル名\
grade               : グレード (1-2)\
row_bytes           : 【エンコード時】一行のバイト数 (16-128)\
page_lines          : 【エンコード時】一頁の行数 (8-128)\
put_catital         : 【エンコード時】大文字符自動付与の要否 (y/n/t/nil)\
to_hankaku          : 【デコード時　】英数文字の半角変換の要否 (y/n/t/nil)\
emit_signs          : 【デコード時　】数符・外字符・ACBC符・大文字符出力の要否 (y/n/t/nil)\
emit_jp_prep        : 【デコード時　】濁音、半濁音を前置きの要否 (y/n/t/nil)\
emit_header         : ヘッダ（含改ページ）出力の要否 (y/n/t/nil)\
emit_blank_paddings : 空行への空白のパディング出力の要否 (y/n/t/nil)\
\
eg.\
 $ cd ~A/projects/brc/lisp\
 $ clisp -C Braille.lisp ../UEBC/brf/bbrl.txt nil 2 32  22  t nil nil nil n y\
 $ clisp -C Braille.lisp ../UEBC/brf/bbrl.brf nil 2 nil nil y   y   n   n y"
 "~"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 初期化開始
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 言語連想配列を初期化する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(plain)
(bold)
(format t "~%~% +++ Initializing language association list...~%")
(plain)
(initLang 1 "us" t)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; それぞれの言語の連想配列から墨字・点字の最大長を取得する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(bold)
(format t " +++ Getting max lengths of text and braille in each vocabulary...~%")
(plain)
(getMaxItemLength)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 英語グレード２の辞書を作成する。（CLISP でないと .ext ファイルが正しく生成されない！！）
; エラーが発生した場合はプログラムを終了する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(bold)
(format t " +++ Generating grade 2 US English dictionary...~%")
(plain)
(let (err)
  (if _genG2Dic_
    (progn
      (setf err (genUsGrade2Dictionary "../dic/us_grade2.txt"))
      (if err
        (progn
          (bold)
          (red)
          (format t "~%Error [~A] has occured while [~S] grade 2 US English braille dictionary.~%~S" err "generating")
          (plain)
          (quit))
        (progn
          (setf err (saveUsGrade2ExtFile "../dic/us_grade2.ext"))
          (if err
            (progn
              (bold)
              (red)
              (format t "~%Error [~A] has occured while [~S] extract file of grade 2 US English braille dictionary.~%~S" err "writing")
              (plain)
              (quit))
            nil))))
    (progn
      (setf err (loadUsGrade2ExtFile "../dic/us_grade2.ext"))
      (if err
        (progn
          (bold)
          (red)
          (format t "~%Error [~A] has occured while [~S] extract file of grade 2 US English braille dictionary.~%~S" err "reading")
          (plain)
          (quit))
        nil))))

(bold)
(format t " +++ Generating grade 2 British English braille dictionary...~%")
(plain)
(let (err)
  (if _genG2Dic_
    (progn
      (setf err (genUkGrade2Dictionary "../dic/us_grade2.txt"))
      (if err
        (progn
          (bold)
          (red)
          (format t "~%Error [~A] has occured while [~S] grade 2 British English braille dictionary.~%~S" err "generating")
          (plain)
          (quit))
        (progn
          (setf err (saveUkGrade2ExtFile "../dic/uk_grade2.ext"))
          (if err
            (progn
              (bold)
              (red)
              (format t "~%Error [~A] has occured while [~S] extract file of grade 2 British English braille dictionary.~%~S" err "writing")
              (plain)
              (quit))
            nil))))
    (progn
      (setf err (loadUkGrade2ExtFile "../dic/uk_grade2.ext"))
      (if err
        (progn
          (bold)
          (red)
          (format t "~%Error [~A] has occured while [~S] extract file of grade 2 British English braille dictionary.~%~S" err "reading")
          (plain)
          (quit))
        nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 英語グレード２辞書データでアポストロフィーが正しく処理されていない場合は不正であるので、
; プログラムを終了する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(bold)
(format t " +++ Verifying grade 2 US English braille dictionary...~%")
(plain)
(if (null (assoc '|C'T| *us_grade2_dec_dict*))
  (progn
    (bold)
    (red)
    (format t "Wrong grade 2 US English braille dictionary data was detected.~%")
    (plain)
    (quit))
  nil)

(bold)
(format t " +++ Verifying grade 2 British English braille dictionary...~%")
(plain)
(if (null (assoc '|C'T| *uk_grade2_dec_dict*))
  (progn
    (bold)
    (red)
    (format t "Wrong grade 2 British English braille dictionary data was detected.~%")
    (plain)
    (quit))
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 英語ハイフネーションの辞書を作成する。
; エラーが発生した場合はプログラムを終了する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(bold)
(format t " +++ Generating US English hyphenation dictionary...~%")
(plain)
(let (dic err)
  (progn
    (multiple-value-setq
      (dic err) (genHyphenationDictionary "../dic/hyph_en_GB.dic"))
    (if err
      (progn
        (bold)
        (red)
        (format t "~%Error [~A] has occured while [~S] US English hyphenation dictionary.~%~S" err "generating")
        (plain)
        (quit))
      (setf *uk_hyphen_dict* dic))))

(bold)
(format t " +++ Generating British English hyphenation dictionary...~%")
(plain)
(let (dic err)
  (progn
    (multiple-value-setq
      (dic err) (genHyphenationDictionary "../dic/hyph_en_US.dic"))
    (if err
      (progn
        (bold)
        (red)
        (format t "~%Error [~A] has occured while [~S] British English hyphenation dictionary.~%~S" err "generating")
        (plain)
        (quit))
      (setf *us_hyphen_dict* dic))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; URL リストを作成する。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(bold)
(format t " +++ Generating US URL list...~%")
(plain)
(if (not (makeUsUrlList "../dic/url.txt"))
  (progn
    (bold)
    (red)
    (format t "~%Error has occured while generating of US URL list.")
    (plain)
    (quit))
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 英語グレード２の辞書モードを初期化する。（動的な辞書登録なし、辞書参照あり）
; 現状では、大文字符、連続大文字符、三重連続大文字符がデコード結果に反映されない問題あり。
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(setf *regist_dict* nil)
(setf *lookup_dict* t)

(setf *nonAlphaList* (non-alphaList))

(bold)
(green)
(format t " +++ Initialization of BrailleConverter was successfully completed.~%")
(plain)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 初期化終了
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(if (string= (compiler-version) "clisp")
  nil
  (setf *args* nil))

(if (> (length *args*) 0)
  (let ((input_file nil) (output_file nil) (grade 1) (nth_line nil) (nth_token nil)
        (row_bytes 32) (page_lines 22) (to_hankaku nil) (emit_signs nil)
        (emit_jp_prep nil) (put_capital nil) (emit_header nil) (emit_blank_paddings t)
        (YorN '(0 1)) errList lst (method "trans") (ext "") s)
    (format t " +++ 変換を開始しました。~%")
    ; 入力ファイル名
    (setf input_file (nth 0 *args*))
    (setf lst (split-string input_file "."))
    (if (> (length lst) 1)
      (setf ext (car (last lst)))
      nil)
    ; 変換か逆変換の判別は拡張子で自動判別する。
    (if (or (string= ext "bse")
            (string= ext "brf")
            (string= ext "brl"))
      (setf method "backtrans")
      nil)
    ; 出力ファイル名
    (if (> (length *args*) 1)
      (progn
        (setf s (string-downcase (nth 1 *args*)))
        (if (or (string= s "n") (string= s "nil"))
          (setf output_file nil)
          (setf output_file (nth 1 *args*))))
      nil)
    ; グレード
    (if (> (length *args*) 2)
      (if (string= (nth 2 *args*) "2") (setf grade 2) (setf grade 1))
      nil)
    ; 一行のバイト数
    (if (> (length *args*) 3)
      (if (numberp (nth 3 *args*))
        (setf row_bytes (nth 3 *args*))
        (setf row_bytes 32))
      nil)
    ; 一頁の行数
    (if (> (length *args*) 4)
      (if (numberp (nth 4 *args*))
        (setf page_lines (nth 4 *args*))
        (setf page_lines 22))
      nil)
    ; 大文字符自動付与の要否
    (if (> (length *args*) 5)
      (progn
        (setf s (string-downcase (nth 5 *args*)))
        (setf put_capital (or (string= s "y") (string= s "t"))))
      nil)
    ; 英数文字の半角変換の要否
    (if (> (length *args*) 6)
      (progn
        (setf s (string-downcase (nth 6 *args*)))
        (setf to_hankaku (or (string= s "y") (string= s "t"))))
      nil)
    ; 数符・外字符・ACBC符・大文字符出力の要否
    (if (> (length *args*) 7)
      (progn
        (setf s (string-downcase (nth 7 *args*)))
        (setf emit_signs (or (string= s "y") (string= s "t"))))
      nil)
    ; 濁音、半濁音を前置きの要否
    (if (> (length *args*) 8)
      (progn
        (setf s (string-downcase (nth 8 *args*)))
        (setf emit_jp_prep (or (string= s "y") (string= s "t"))))
      nil)
    ; ヘッダ・改ページ出力の要否
    (if (> (length *args*) 9)
      (progn
        (setf s (string-downcase (nth 9 *args*)))
        (setf emit_header (or (string= s "y") (string= s "t"))))
      nil)
    ; 空行への空白のパディング出力の要否
    (if (> (length *args*) 10)
      (progn
        (setf s (string-downcase (nth 10 *args*)))
        (setf emit_blank_paddings (or (string= s "y") (string= s "t"))))
      nil)
    ; グレードを設定する。
    (if (= grade 2) (g2us) (jp))
    ; 変換を実行する。
    (format t "~%;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;~%")
    (if (string= method "trans")
      (setf errList (encodeBrailleFile input_file output_file nth_line nth_token row_bytes page_lines  t            put_capital emit_header emit_blank_paddings))
      (setf errList (decodeBrailleFile input_file output_file nth_line nth_token     to_hankaku        t emit_signs emit_jp_prep emit_header emit_blank_paddings)))
    (format t ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;~%")
    (if (string= method "trans")
      (format t "~40A: ~A~%~40A: ~A~%~40A: ~A~%~40A: ~S~%~40A: ~S~%~40A: ~S~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~~%"
        "変換方向" "墨字 >> 点字"
        "入力ファイル名" input_file
        "出力ファイル名" (if output_file (string output_file) (string "なし"))
        "グレード" grade
        "一行のバイト数" row_bytes
        "一頁の行数" page_lines
        "大文字符自動付与の要否" (if put_capital (nth 0 YorN) (nth 1 YorN))
        "ヘッダ・改ページ出力の要否" (if emit_header (nth 0 YorN) (nth 1 YorN))
        "空行への空白のパディング出力の要否" (if emit_blank_paddings (nth 0 YorN) (nth 1 YorN)))
      (format t "~40A: ~A~%~40A: ~A~%~40A: ~A~%~40A: ~S~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~40A: ~[する~;しない~]~%~%"
        "変換方向" "点字 >> 墨字"
        "入力ファイル名" input_file
        "出力ファイル名" (if output_file (string output_file) (string "なし"))
        "グレード" grade
        "英数文字の半角変換の要否" (if to_hankaku (nth 0 YorN) (nth 1 YorN))
        "数符・外字符・ACBC符・大文字符出力の要否" (if emit_signs (nth 0 YorN) (nth 1 YorN))
        "濁音、半濁音を前置きの要否" (if emit_jp_prep (nth 0 YorN) (nth 1 YorN))
        "ヘッダ・改ページ出力の要否" (if emit_header (nth 0 YorN) (nth 1 YorN))
        "空行への空白のパディング出力の要否" (if emit_blank_paddings (nth 0 YorN) (nth 1 YorN))))
    (if errList
      (multiple-value-bind (lst lang) (extract-err errList 4)
        (format t "【メッセージ】変換中にエラーが発生しました。~%　以下の文字は変換できませんでした。[~A]~%　~S" lang lst))
      (format t "【メッセージ】変換が正常に終了しました。" ))
    (format t "~%;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;"))
;  (print-usage)
)
t